package siarchive.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.sql.SQLException;

import org.junit.After;
import org.junit.Test;

import siarchive.components.Position;
import siarchive.components.Status;
import siarchive.i18n.Language;
import siarchive.persistence.Account;
import siarchive.persistence.Alliance;
import siarchive.persistence.DatabaseFactory;
import siarchive.persistence.Note;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.persistence.dao.AccountDao;
import siarchive.persistence.dao.AllianceDao;
import siarchive.persistence.dao.PlanetDao;
import siarchive.persistence.dao.PlayerDao;

public class PlanetTest
{
    @After
    public final void tearDown() {
        DatabaseFactory.close();
    }

    @Test
    public void test() throws SQLException
    {
        DatabaseFactory.open();
        Account account;
        AccountDao accountDao = DatabaseFactory.getDao( AccountDao.class );
        Alliance alliance;
        AllianceDao allianceDao = DatabaseFactory.getDao( AllianceDao.class );
        Player player;
        PlayerDao playerDao = DatabaseFactory.getDao( PlayerDao.class );
        Planet planet;
        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );

        account = new Account();
        account.setName( "testAccount1" );
        account.setLanguage( Language.german );
        account.setIndex( 1 );
        account.setDefault( false );
        account = accountDao.save( account );

        alliance = new Alliance();
        alliance.setAccount( account.getId() );
        // the empty alliance will get ID 0
        alliance.setAllianceId( 0 );
        alliance.setName( "" );
        alliance = allianceDao.save(alliance);

        player = new Player();
        player.setName( "Bluehawk" + 1 );
        player.setAccount( account.getId() );
        player.setAlliance(alliance);
        player.setPoints( 30000000L + 1 * 1000);
        player.setUpdateTime( System.currentTimeMillis() );
        player.setStatus( Status.enemy );
        player.setSecondaryStatus( null );
        playerDao.save( player );

        planet = new Planet();
        planet.setName( "Colony");
        planet.setAccount( account.getId() );
        planet.setOwner( player );
        planet.setPosition( Position.createId( 1, 1, 1 ) );
        planet.setUpdateTime( System.currentTimeMillis() );
        planetDao.save( planet );

        account = new Account();
        account.setName( "testAccount2" );
        account.setLanguage( Language.german );
        account.setIndex( 1 );
        account.setDefault( false );
        account = accountDao.save( account );

        alliance = new Alliance();
        alliance.setAccount( account.getId() );
        // the empty alliance will get ID 0
        alliance.setAllianceId( 0 );
        alliance.setName( "" );
        alliance = allianceDao.save(alliance);

        player = new Player();
        player.setName( "Bluehawk" + 2 );
        player.setAccount( account.getId() );
        player.setAlliance(alliance);
        player.setPoints( 30000000L + 2 * 1000);
        player.setUpdateTime( System.currentTimeMillis() );
        player.setStatus( Status.enemy );
        player.setSecondaryStatus( null );
        playerDao.save( player );

        planet = new Planet();
        planet.setName( "Colony");
        planet.setAccount( account.getId() );
        planet.setOwner( player );
        planet.setPosition( Position.createId( 1, 1, 2 ) );
        planet.setUpdateTime( System.currentTimeMillis() );
        planetDao.save( planet );
        DatabaseFactory.commit();


        assertEquals( 2, planetDao.count() );
        assertEquals( 1, planetDao.countByAccount( account.getId() ) );
        planet = planetDao.find( account.getId(), Position.createId( 1, 1, 1 ) );
        assertEquals( null, planet );
        planet = planetDao.find( account.getId(), Position.createId( 1, 1, 2 ) );
        assertNotNull( planet );

        Note notes = new Note();
        notes.setText( "some Note" );
        planet.setNotes( notes );
        planetDao.save( planet );
        DatabaseFactory.commit();

        planet = new Planet();
        planet.setAccount( account.getId() );
        planet.setName( "Home");
        planet.setOwner( player );
        planet.setPosition( Position.createId( 1, 1, 2 ) );
        planet.setUpdateTime( System.currentTimeMillis() );
        planet.setRetainNotes( true );
        planetDao.save( planet );
        DatabaseFactory.commit();

        planet = planetDao.find( account.getId(), Position.createId( 1, 1, 2 ) );
        assertNotNull( planet );
        assertNotNull( planet.getNotes() );
        assertEquals( "some Note", planet.getNotes().getText() );

        account = accountDao.find( "testAccount1" );
        accountDao.delete( account );
        account = accountDao.find( "testAccount2" );
        accountDao.delete( account );
        DatabaseFactory.commit();
        assertEquals( 0, accountDao.count() );
        assertEquals( 0, planetDao.count() );

        DatabaseFactory.commit();
    }

    @Test
    public void timeTest() throws SQLException
    {
        DatabaseFactory.open();
        Account account;
        AccountDao accountDao = DatabaseFactory.getDao( AccountDao.class );
        Alliance alliance;
        AllianceDao allianceDao = DatabaseFactory.getDao( AllianceDao.class );
        Player player;
        PlayerDao playerDao = DatabaseFactory.getDao( PlayerDao.class );
        Planet planet;
        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );

        account = new Account();
        account.setName( "timeTestAccount" );
        account.setLanguage( Language.german );
        account.setIndex( 1 );
        account.setDefault( false );
        account = accountDao.save( account );

        alliance = new Alliance();
        alliance.setAccount( account.getId() );
        // the empty alliance will get ID 0
        alliance.setAllianceId( 0 );
        alliance.setName( "" );
        alliance = allianceDao.save(alliance);

        player = new Player();
        player.setName( "TimeTest");
        player.setAccount( account.getId() );
        player.setAlliance(alliance);
        player.setPoints( 30_000_000L + 1 * 1000);
        player.setUpdateTime( System.currentTimeMillis() );
        player.setStatus( Status.enemy );
        player.setSecondaryStatus( null );
        playerDao.save( player );

        long time = 10_000_000_000L;

        for(int sys = 1; sys < 11; sys++)
        {
            for(int pla = 1; pla < 17; pla += 5 )
            {
                planet = new Planet();
                planet.setName( "Colony");
                planet.setAccount( account.getId() );
                planet.setOwner( player );
                planet.setPosition( Position.createId( 1, sys, pla ) );
                planet.setUpdateTime( time );
                planetDao.save( planet );
                time += 1000000;
            }
        }

        int start;
        int end;
        start = Position.createId(1, 1, 1);
        end = Position.createId(1, 1, 16);
        Long min1 = planetDao.minTimeByPosition(account.getId(), start, end);
        assertEquals(Long.valueOf(10_000_000_000L), min1);

        start = Position.createId(1, 2, 1);
        end = Position.createId(1, 2, 16);
        Long min2 = planetDao.minTimeByPosition(account.getId(), start, end);
        assertEquals(Long.valueOf(10_004_000_000L), min2);

        start = Position.createId(1, 8, 1);
        end = Position.createId(1, 8, 16);
        Long min8 = planetDao.minTimeByPosition(account.getId(), start, end);
        assertEquals(Long.valueOf(10_028_000_000L), min8);

        start = Position.createId(1, 12, 1);
        end = Position.createId(1, 12, 16);
        Long min12 = planetDao.minTimeByPosition(account.getId(), start, end);
        assertEquals(null, min12);

        account = accountDao.find( "timeTestAccount" );
        accountDao.delete( account );
        DatabaseFactory.commit();
    }

}
