/****************************************
 *  COPYRIGHT (C) 2023
 *  Holger Graf
 ****************************************/
package siarchive.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import com.fasterxml.jackson.jr.ob.JSON;

import siarchive.components.Position;
import siarchive.galaxy.SystemData;
import siarchive.components.Flavor;
import siarchive.components.PlanetData;

/**
 *
 */
public class SystemDataConverter {

    static class PlanetDataBean {

        private Position coords;
        private int temperature;
        private int x;
        private int y;


        public PlanetDataBean() {}

        public Position getCoords() {
            return coords;
        }

        public void setCoords( Object coords ) {
            this.coords = Position.createFromId(Position.createId(coords.toString()));
        }

        public int getTemperature() {
            return temperature;
        }

        public void setTemperature( int temperature ) {
            this.temperature = temperature;
        }

        public int getX() {
            return x;
        }

        public void setX( int x ) {
            this.x = x;
        }

        public int getY() {
            return y;
        }

        public void setY( int y ) {
            this.y = y;
        }
    }

    static class PlanetDataConvertBean {

        private String coords;
        private int temperature;
        private int x;
        private int y;


        public PlanetDataConvertBean() {}

        public String getCoords() {
            return coords;
        }

        public void setCoords( String coords ) {
            this.coords = coords;
        }

        public int getTemperature() {
            return temperature;
        }

        public void setTemperature( int temperature ) {
            this.temperature = temperature;
        }

        public int getX() {
            return y;
        }

        public void setX( int x ) {
            this.x = x;
        }

        public int getY() {
            return x;
        }

        public void setY( int y ) {
            this.y = y;
        }

    }

    private static int getMapId(Flavor flavor) {
        int mapID;
        switch(flavor) {
            case speed:
            case solaris:
                mapID = Flavor.speed.getId();
                break;
            default:
                mapID = flavor.getId();
                break;
        }
        return mapID;
    }

    /**
     * @param args
     */
    public static void main( String[] args ) {
        if(args.length > 0 && "-c".equals(args[0])) {
            convert();
        }
        else {
            check();
        }
    }

    private static void convert() {
        for(Flavor flavor : Flavor.values()) {
            List<PlanetDataConvertBean> planets = null;
            try (InputStream input = new FileInputStream(flavor.name() +".json")) {
                planets = new ArrayList<>(Arrays.asList(JSON.std.beanFrom(PlanetDataConvertBean[].class, input)));
                Iterator<PlanetDataConvertBean> it = planets.iterator();
                while(it.hasNext()) {
                    PlanetDataConvertBean bean = it.next();
                    if(bean.getX() == 0 && bean.getY() == 0 && bean.getTemperature() == 0) {
                        it.remove();
                    }
                }
                Collections.sort(planets, new Comparator<PlanetDataConvertBean>() {
                    @Override
                    public int compare( PlanetDataConvertBean o1, PlanetDataConvertBean o2 ) {
                        Position p1 = Position.createFromId(Position.createId(o1.getCoords()));
                        Position p2 = Position.createFromId(Position.createId(o2.getCoords()));
                        return p1.compareTo(p2);
                    }
                });
            }
            catch(Exception e) {
                e.printStackTrace();
            }
            File oFile = new File(flavor.getId() + ".json");
            try {
                JSON.std
                .with(JSON.Feature.PRETTY_PRINT_OUTPUT)
                .with(JSON.Feature.PRESERVE_FIELD_ORDERING)
                .write(planets, oFile);
            }
            catch(IOException e) {
                e.printStackTrace();
            }
        }
    }

    private static void check() {
        for(Flavor version : Flavor.values()) {
            try (InputStream input = new FileInputStream(getMapId(version) +".json")) {
                PlanetDataBean[] planets = JSON.std.beanFrom(PlanetDataBean[].class, input);
                System.out.println(version.name());
                for(PlanetDataBean bean : planets) {
                    PlanetData data = SystemData.getPlanetData(version, bean.getCoords());
                    if(data.getX() != bean.getX()
                    || data.getY() != bean.getY()
                    || data.getTemperature() != bean.getTemperature()) {
                        System.out.print(bean.getCoords());
                        System.out.print("\t{ ");
                        System.out.print(bean.getX());
                        System.out.print(", ");
                        System.out.print(bean.getY());
                        System.out.print(", ");
                        System.out.print(bean.getTemperature());
                        System.out.print(" }");
                        System.out.print("\t <-> ");
                        System.out.print("\t{ ");
                        System.out.print(data.getX());
                        System.out.print(", ");
                        System.out.print(data.getY());
                        System.out.print(", ");
                        System.out.print(data.getTemperature());
                        System.out.print(" }");
                        System.out.println();
                    }
                }
            }
            catch(Exception e) {
                e.printStackTrace();
            }
        }
    }

}
