/****************************************
 *  COPYRIGHT (C) 2009-2024
 *  Holger Graf
 ****************************************/
package siarchive.table;

import java.awt.Font;
import java.awt.event.MouseEvent;

import javax.swing.ListSelectionModel;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import siarchive.DataManager;
import siarchive.components.Cost;
import siarchive.components.Fleet;
import siarchive.persistence.Alliance;
import siarchive.persistence.Note;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;

/**
 * @author graf
 *
 */
public class TooltipTable extends MinimumRowHeightTable
{
    private static final long serialVersionUID = 5374293775888082770L;

    private Font font;

    public TooltipTable()
    {
        super();
    }

    /**
     * @param dm
     * @param cm
     * @param sm
     */
    public TooltipTable( TableModel dm, TableColumnModel cm, ListSelectionModel sm )
    {
        super( dm, cm, sm );
    }

    /**
     * @param dm
     * @param cm
     */
    public TooltipTable( TableModel dm, TableColumnModel cm )
    {
        super( dm, cm );
    }

    /**
     * @param dm
     */
    public TooltipTable( TableModel dm )
    {
        super( dm );
    }

    protected JTableHeader createDefaultTableHeader()
    {
        return new TooltipTableHeader(columnModel);
    }

    @Override
    public String getToolTipText(MouseEvent event)
    {
        String tooltip = null;
        int viewRow = rowAtPoint( event.getPoint() );
        int viewColumn = columnAtPoint( event.getPoint() );
        int modelRow = convertRowIndexToModel( viewRow );
        int modelColumn = convertColumnIndexToModel( viewColumn );
        if( modelRow >= 0 && modelColumn >= 0 )
        {
            Object value = getModel().getValueAt( modelRow, modelColumn );
            if( value instanceof String )
            {
                tooltip = value.toString();
                if( tooltip.trim().length() == 0 )
                {
                    tooltip = null;
                }
            }
            else if( value instanceof Number )
            {
                long number = ((Number)value).longValue();
                tooltip = DataManager.format(number);
            }
            else if( value instanceof Planet )
            {
                tooltip = value.toString();
                if( tooltip.trim().length() == 0 )
                {
                    tooltip = null;
                }
            }
            else if( value instanceof Alliance )
            {
                Alliance alliance = ( Alliance )value;
                StringBuilder sb = new StringBuilder();
                sb.append( alliance.toString() );
                if( alliance.getAllianceId() > 0 )
                {
                    sb.append( " [id:" ).append( alliance.getAllianceId() ).append( ']' );
                }
                tooltip = sb.toString();
                if( tooltip.trim().length() == 0 )
                {
                    tooltip = null;
                }
            }
            else if( value instanceof Player )
            {
                Player player = ( Player )value;
                StringBuilder sb = new StringBuilder();
                sb.append( player.getName() );
                if( player.getPlayerId() >= 0 )
                {
                    sb.append( " [uid:" ).append( player.getPlayerId() ).append( ']' );
                }
                tooltip = sb.toString();
                if( tooltip.trim().length() == 0 )
                {
                    tooltip = null;
                }
            }
            else if( value instanceof Note )
            {
                tooltip = value.toString();
                if( tooltip.trim().length() == 0 )
                {
                    tooltip = null;
                }
            }
            else if( value instanceof Cost && getModel() instanceof CostRenderer )
            {
                tooltip = ((CostRenderer)getModel()).renderCost( ( Cost )value );
            }
            else if( value instanceof Fleet && getModel() instanceof FleetRenderer )
            {
                tooltip = ((FleetRenderer)getModel()).renderFleet( (Fleet)value );
            }
        }
        return tooltip;
    }

    @Override
    public Font getFont()
    {
        Font font = this.font;
        if( font == null )
        {
            UIDefaults uiDefaults = UIManager.getDefaults();
            String key = "Table.font";
            font = ( Font )uiDefaults.get( key );
        }
        return font;
    }

    @Override
    public void setFont( Font font )
    {
        super.setFont( font );
        this.font = font;
    }
}
