/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/
package siarchive.table;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import javax.swing.Icon;
import javax.swing.table.AbstractTableModel;

import siarchive.DataManager;
import siarchive.Resources;
import siarchive.components.Cost;
import siarchive.components.Resource;
import siarchive.components.Status;
import siarchive.persistence.Alliance;
import siarchive.persistence.Note;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;

/**
 * @author graf
 *
 */
public class PlanetTableModel extends AbstractTableModel implements CostRenderer
{
    public static final int OWNER = 0;
    public static final int OWNERNOTES = OWNER + 1;
    public static final int ALLIANCE = OWNERNOTES + 1;
    public static final int ALLIANCETAG = ALLIANCE + 1;
    public static final int ALLIANCENOTES = ALLIANCETAG + 1;
    public static final int STATUS = ALLIANCENOTES + 1;
    public static final int POINTS = STATUS + 1;
    public static final int PLANET = POINTS + 1;
    public static final int PLANETNOTES = PLANET + 1;
    public static final int DEBRIS = PLANETNOTES + 1;
    public static final int EVENT = DEBRIS + 1;
    public static final int ASTRO = EVENT + 1;

    private static final long serialVersionUID = -1770345678362128348L;
    private DataManager dataManager;
    private List<Planet> data = new ArrayList<Planet>();
    /**
     *
     */
    public PlanetTableModel( DataManager dataManager )
    {
        this.dataManager = dataManager;
    }

    /**
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount()
    {
        return ASTRO + 1;
    }

    /**
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount()
    {
        return data.size();
    }

    public String getColumnName( int columnIndex )
    {
        String rv ="";
        switch( columnIndex )
        {
            case OWNER:
                rv = dataManager.getI18nText( "GalaxyView.owner" );
                break;
            case ALLIANCE:
                rv = dataManager.getI18nText( "GalaxyView.alliance" );
                break;
            case ALLIANCETAG:
                rv = dataManager.getI18nText( "" );
                break;
            case STATUS:
                rv = dataManager.getI18nText( "GalaxyView.status" );
                break;
            case POINTS:
                rv = dataManager.getI18nText( "GalaxyView.points" );
                break;
            case PLANET:
                rv = dataManager.getI18nText( "PlanetView.coords" );
                break;
            case OWNERNOTES:
            case ALLIANCENOTES:
            case PLANETNOTES:
                rv = "/";
                break;
            case DEBRIS:
            case EVENT:
            case ASTRO:
                break;
        }
        return rv;
    }


    @Override
    public Class<?> getColumnClass( int columnIndex )
    {
        Class<?> rv = String.class;
        switch( columnIndex )
        {
            case OWNER:
                rv = Player.class;
                break;
            case ALLIANCE:
                rv = Alliance.class;
                break;
            case ALLIANCETAG:
            case STATUS:
                rv = String.class;
                break;
            case PLANET:
                rv = Planet.class;
                break;
            case POINTS:
                rv = Number.class;
                break;
            case OWNERNOTES:
            case ALLIANCENOTES:
            case PLANETNOTES:
                rv = Note.class;
                break;
            case DEBRIS:
                rv = Cost.class;
                break;
            case EVENT:
            case ASTRO:
                rv = Icon.class;
                break;
        }
        return rv;
    }

    /**
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt( int rowIndex, int columnIndex )
    {
        Object rv = "";
        Planet planet = data.get( rowIndex );
        Player owner = planet.getOwner();
        switch( columnIndex )
        {
            case OWNER:
                if( owner != null )
                {
                    rv = planet.getOwner();
                }
                break;
            case OWNERNOTES:
                if( owner != null )
                {
                    rv = planet.getOwner().getNotes();
                }
                break;
            case ALLIANCE:
                if( owner != null && owner.getAlliance() != null )
                {
                    rv = owner.getAlliance();
                }
                break;
            case ALLIANCETAG:
                if( owner != null && owner.getAlliance() != null )
                {
                    rv = owner.getAlliance().getTag();
                }
                break;
            case ALLIANCENOTES:
                if( owner != null && owner.getAlliance() != null )
                {
                    rv = owner.getAlliance().getNotes();
                }
                break;
            case STATUS:
                if( owner != null )
                {
                    if( owner.getStatus() != null )
                    {
                        rv = dataManager.getI18nText( owner.getStatus().name() );
                        if( owner.getStatus() == Status.locked )
                        {
                            if( owner.getSecondaryStatus() != null && owner.getSecondaryStatus() != Status.none )
                            {
                                StringBuilder sb = new StringBuilder();
                                sb.append( rv );
                                sb.append( " (" );
                                sb.append( dataManager.getI18nText( owner.getSecondaryStatus().name() ) );
                                sb.append( ") " );
                                rv = sb.toString();
                            }
                        }
                    }
                }
                break;
            case POINTS:
                if( owner != null )
                {
                    rv = Long.valueOf( owner.getPoints() );
                }
                break;
            case PLANET:
                rv = planet;
                break;
            case PLANETNOTES:
                rv = planet.getNotes();
                break;
            case DEBRIS:
                rv = planet.getDebris();
                break;
            case EVENT:
                if( planet.isEvent() )
                {
                    rv = Resources.getIcon( "/planet_event.png" );
                }
                break;
            case ASTRO:
                if( planet.hasAsteroid() )
                {
                    rv = Resources.getIcon( "/moon.png" );
                }
                break;
        }
        return rv;
    }

    public boolean isCellEditable(int rowIndex, int columnIndex)
    {
        return false;
    }

    public Planet getRow( int row )
    {
        return data.get( row );
    }

    public void removeRows( int[] modelRows ) throws SQLException
    {
        // delete back to front to preserve indices
        Arrays.sort( modelRows );
        List<Planet> oldPlanets = new ArrayList<Planet>();
        for( int i = modelRows.length - 1; i >= 0; i-- )
        {
            oldPlanets.add(  data.remove( modelRows[i] ) );
        }
        dataManager.removePlanetsCascade( oldPlanets );
        dataManager.commit();
        fireTableDataChanged();
    }

    public List<Planet> getExportRows( int[] modelRows )
    {
        List<Planet> result = new ArrayList<Planet>();
        for( int i = 0; i < modelRows.length; i++ )
        {
            result.add( data.get( modelRows[i] ) );
        }
        return result;
    }

    public List<Planet> getExportRows()
    {
        List<Planet> result = new ArrayList<Planet>();
        result.addAll( data );
        return result;
    }

    public void removeAll() throws SQLException
    {
        dataManager.removePlanetsCascade( data );
        dataManager.commit();
        data.clear();
        fireTableDataChanged();
    }

    public Collection<Planet> getData()
    {
        List<Planet> data = new ArrayList<Planet>();
        data.addAll( this.data );
        return data;
    }

    public void setData( Collection<Planet> newData )
    {
        data.clear();
        data.addAll( newData );
        fireTableDataChanged();
    }

    public void update()
    {
        data.clear();
        // repaint header
        fireTableStructureChanged();
    }

    public String renderCost( Cost debris )
    {
        String rv = null;
        if( debris != null && !debris.isEmpty() )
        {
            StringBuilder sb = new StringBuilder();
            sb.append( "<html>" );
            sb.append( "<table>" );
            if( debris.getIron() > 0 )
            {
                sb.append( "<tr><td>" );
                sb.append( dataManager.getI18nText( Resource.iron.name() ) );
                sb.append( "</td><td>" );
                sb.append( DataManager.format( debris.getIron() ) );
                sb.append( "</td></tr>" );
            }
            if( debris.getMetal() > 0 )
            {
                sb.append( "<tr><td>" );
                sb.append( dataManager.getI18nText( Resource.metal.name() ) );
                sb.append( "</td><td>" );
                sb.append( DataManager.format( debris.getMetal() ) );
                sb.append( "</td></tr>" );
            }
            if( debris.getKryptonite() > 0 )
            {
                sb.append( "<tr><td>" );
                sb.append( dataManager.getI18nText( Resource.kryptonite.name() ) );
                sb.append( "</td><td>" );
                sb.append( DataManager.format( debris.getKryptonite() ) );
                sb.append( "</td></tr>" );
            }
            if( debris.getSpice() > 0 )
            {
                sb.append( "<tr><td>" );
                sb.append( dataManager.getI18nText( Resource.spice.name() ) );
                sb.append( "</td><td>" );
                sb.append( DataManager.format( debris.getSpice() ) );
                sb.append( "</td></tr>" );
            }
            sb.append( "</table>" );
            sb.append( "</html>" );
            rv = sb.toString();
        }
        return rv;
    }
}
