/****************************************
 *  COPYRIGHT (C) 2008
 *  Holger Graf
 ****************************************/
package siarchive.table;

import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.table.AbstractTableModel;

import siarchive.MainFrame;

/**
 * @author graf
 *
 */
public class FontTableModel extends AbstractTableModel
{
    private static final long serialVersionUID = -1405529512618782948L;

    protected MainFrame mainFrame;
    List<FontLine> data;

    public FontTableModel( MainFrame mainFrame )
    {
        this.mainFrame = mainFrame;

        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        List<String> fonts = Arrays.asList( ge.getAvailableFontFamilyNames() );
        UIDefaults uiDefaults = UIManager.getDefaults();
        Font font = ( Font )uiDefaults.get( "Table.font" );
        int size = font.getSize();
        String def = font.getName();
        
        data = new ArrayList<FontLine>();
        
        for( String name : fonts )
        {
            FontLine line = new FontLine();
            line.name = name;
            line.font = new Font(name, Font.PLAIN, size );
            line.selected = def.equals( line.name );
            // only add fonts which can display at least their own name
            if( line.font.canDisplayUpTo( name ) == -1 )
            {
                data.add( line );
            }
        }
    }

    /**
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount()
    {
        return 3;
    }

    @Override
    public Class<?> getColumnClass( int columnIndex )
    {
        Class<?> rv = String.class;
        switch( columnIndex )
        {
            case 0:
                rv = String.class;
                break;
            case 1:
                rv = Font.class;
                break;
            case 2:
                rv = Boolean.class;
                break;
        }
        return rv;
    }

    /**
     * @see javax.swing.table.AbstractTableModel#getColumnName(int)
     */
    @Override
    public String getColumnName( int columnIndex )
    {
        String rv = null;
        return rv;
    }


    /**
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount()
    {
        return data.size();
    }

    /**
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt( int rowIndex, int columnIndex )
    {
        FontLine target = data.get( rowIndex );
        Object rv = null;
        switch( columnIndex )
        {
            case 0:
                rv = target.name;
                break;
            case 1:
                rv = target.font;
                break;
            case 2:
                rv = Boolean.valueOf( target.selected );
                break;
        }
        return rv;
    }
    
    @Override
	public boolean isCellEditable(int rowIndex, int columnIndex)
	{
    	boolean rv = (columnIndex == 2 );
		return rv;
	}

    @Override
	public void setValueAt(Object value, int rowIndex, int columnIndex)
	{
        FontLine target = data.get( rowIndex );
        switch( columnIndex )
        {
            case 2:
                boolean newDefault = Boolean.parseBoolean( value.toString() );
                if( newDefault )
                {
                    // reset others
                    for( FontLine line : data )
                    {
                        line.selected = false;
                    }
                    target.selected = newDefault;
                }
                fireTableDataChanged();
                break;
        }
	}

    public Font getFont()
    {
        Font rv = null;
        for( FontLine line : data )
        {
            if( line.selected )
            {
                rv = line.font;
                break;
            }
        }
        return rv;
    }
    
    public void setFontSize( float size )
    {
        for( FontLine line : data )
        {
            line.font = line.font.deriveFont( size );
        }
        fireTableDataChanged();
    }

    private class FontLine
	{
	    protected Font font;
	    protected String name;
	    protected boolean selected;
	}

}
