/****************************************
 *  COPYRIGHT (C) 2021
 *  Holger Graf
 ****************************************/
package siarchive.table;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.swing.table.AbstractTableModel;

import siarchive.DataManager;
import siarchive.components.BlackHoleLevel;
import siarchive.components.Fleet;
import siarchive.components.Resource;
import siarchive.components.Ship;
import siarchive.persistence.BlackHole;

/**
 * @author graf
 *
 */
public class BlackHoleTableModel extends AbstractTableModel implements FleetRenderer
{
    public static final int POSITION = 0;
    public static final int LEVEL = POSITION + 1;
    public static final int ATTACKED = LEVEL + 1 ;
    public static final int POWER = ATTACKED + 1 ;
    public static final int FLEET = POWER + 1 ;
    public static final int SPICE = FLEET + 1 ;
    public static final int EMBLEMS = SPICE + 1 ;
    public static final int STARS = EMBLEMS + 1 ;

    private static final long serialVersionUID = -1770345678362128348L;
    private DataManager dataManager;
    private List<BlackHole> data = new ArrayList<BlackHole>();
    /**
     *
     */
    public BlackHoleTableModel( DataManager dataManager )
    {
        this.dataManager = dataManager;
    }

    /**
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount()
    {
        return STARS + 1;
    }

    /**
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount()
    {
        return data.size();
    }

    public String getColumnName( int columnIndex )
    {
        String rv ="";
        switch( columnIndex )
        {
            case POSITION:
                rv = dataManager.getI18nText( "BlackHole.coords" );
                break;
            case LEVEL:
                rv = dataManager.getI18nText( "BlackHole.level" );
                break;
            case ATTACKED:
                rv = dataManager.getI18nText( "BlackHole.attacked" );
                break;
            case POWER:
                rv = dataManager.getI18nText( "BlackHole.power" );
                break;
            case FLEET:
                rv = dataManager.getI18nText( "BlackHole.ships" );
                break;
            case SPICE:
                rv = dataManager.getI18nText( Resource.spice.name() );
                break;
            case EMBLEMS:
                rv = dataManager.getI18nText( "BlackHole.rewards.emblems" );
                break;
            case STARS:
                rv = dataManager.getI18nText( "BlackHole.rewards.stars" );
                break;
        }
        return rv;
    }


    @Override
    public Class<?> getColumnClass( int columnIndex )
    {
        Class<?> rv = String.class;
        switch( columnIndex )
        {
            case ATTACKED:
                rv = Boolean.class;
                break;
            case EMBLEMS:
            case POSITION:
            case POWER:
            case SPICE:
            case STARS:
                rv = Number.class;
                break;
            case FLEET:
                rv = Fleet.class;
                break;
            case LEVEL:
                rv = BlackHoleLevel.class;
                break;
        }
        return rv;
    }

    /**
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt( int rowIndex, int columnIndex )
    {
        Object rv = "";
        BlackHole blackHole = data.get( rowIndex );
        switch( columnIndex )
        {
            case POSITION:
                rv = blackHole.getPosition();
                break;
            case LEVEL:
                rv = BlackHoleLevel.fromLevel(blackHole.getLevel());
                break;
            case ATTACKED:
                rv = blackHole.isAttacked();
                break;
            case POWER:
                rv = (blackHole.isSetup() && blackHole.getPower() != 0) ? blackHole.getPower() : null;
                break;
            case FLEET:
                rv = blackHole.getShips();
                break;
            case SPICE:
                rv = (blackHole.isSetup() && blackHole.getSpice() != 0) ? blackHole.getSpice() : null;
                break;
            case EMBLEMS:
                rv = (blackHole.isSetup() && blackHole.getEmblems() != 0) ? blackHole.getEmblems() : null;
                break;
            case STARS:
                rv = (blackHole.isSetup() && blackHole.getStars() != 0) ? blackHole.getStars() : null;
                break;
        }
        return rv;
    }

    public boolean isCellEditable(int rowIndex, int columnIndex)
    {
        return false;
    }


    public BlackHole getRow( int row )
    {
        return data.get( row );
    }

    public void removeRows( int[] modelRows ) throws SQLException
    {
        // delete back to front to preserve indices
        Arrays.sort( modelRows );
        List<BlackHole> list = new ArrayList<>();
        for( int i = modelRows.length - 1; i >= 0; i-- )
        {
            list.add(  data.remove( modelRows[i] ) );
        }
        dataManager.removeRecords( list );
        dataManager.commit();
        fireTableDataChanged();
    }

    public List<BlackHole> getExportRows( int[] modelRows )
    {
        List<BlackHole> result = new ArrayList<>();
        for( int i = 0; i < modelRows.length; i++ )
        {
            result.add( data.get( modelRows[i] ) );
        }
        return result;
    }

    public List<BlackHole> getExportRows()
    {
        List<BlackHole> result = new ArrayList<>();
        result.addAll( data );
        return result;
    }

    public void removeAll() throws SQLException
    {
        dataManager.removeRecords( data );
        dataManager.commit();
        data.clear();
        fireTableDataChanged();
    }

    public void setData( Collection<BlackHole> newData )
    {
        data.clear();
        data.addAll( newData );
        fireTableDataChanged();
    }

    public void update()
    {
        data.clear();
        // repaint header
        fireTableStructureChanged();
    }

    @Override
    public String renderFleet( Fleet ships ) {
        String rv = null;
        if( ships != null && !ships.isEmpty() )
        {
            StringBuilder sb = new StringBuilder();
            sb.append( "<html>" );
            sb.append( "<table>" );
            for(Map.Entry<Ship, Long> entry : ships.entrySet()) {
                sb.append( "<tr><td>" );
                sb.append( dataManager.getI18nText( entry.getKey().name() ) );
                sb.append( "</td><td>" );
                sb.append( DataManager.format( entry.getValue() ) );
                sb.append( "</td></tr>" );
            }
            sb.append( "</table>" );
            sb.append( "</html>" );
            rv = sb.toString();
        }
        return rv;
    }

    @Override
    public String renderFleetOneLine( Fleet ships ) {
        String rv = null;
        if( ships != null && !ships.isEmpty() )
        {
            StringBuilder sb = new StringBuilder();
            for(Map.Entry<Ship, Long> entry : ships.entrySet()) {
                sb.append( dataManager.getI18nText( entry.getKey().name() ) );
                sb.append( ": " );
                sb.append( DataManager.format( entry.getValue() ) );
                sb.append( ", " );
            }
            sb.delete(sb.lastIndexOf(","), sb.lastIndexOf(",") + 2);
            rv = sb.toString();
        }
        return rv;
    }

}
