/****************************************
 *  COPYRIGHT (C) 2010, 2023
 *  Holger Graf
 ****************************************/
package siarchive.table;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.swing.Icon;
import javax.swing.table.AbstractTableModel;

import siarchive.DataManager;
import siarchive.Resources;
import siarchive.components.Position;
import siarchive.components.Priority;
import siarchive.components.Status;
import siarchive.persistence.BattleReport;
import siarchive.persistence.Note;

/**
 * @author graf
 *
 */
public class BattleReportTableModel extends AbstractTableModel
{
    public static final int TIME = 0;
    public static final int PLANET = TIME + 1;
    public static final int ASTRO = PLANET + 1;
    public static final int ATTACKER = ASTRO + 1;
    public static final int DEFENDER = ATTACKER + 1;
    public static final int NOTES = DEFENDER + 1 ;
    public static final int PRIORITY = NOTES + 1 ;

    private static final long serialVersionUID = -1770345678362128348L;
    private DataManager dataManager;
    private List<BattleReport> data = new ArrayList<BattleReport>();
    /**
     *
     */
    public BattleReportTableModel( DataManager dataManager )
    {
        this.dataManager = dataManager;
    }

    /**
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount()
    {
        return PRIORITY + 1;
    }

    /**
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount()
    {
        return data.size();
    }

    public String getColumnName( int columnIndex )
    {
        String rv ="";
        switch( columnIndex )
        {
            case PLANET:
                rv = dataManager.getI18nText( "PlanetView.coords" );
                break;
            case TIME:
                rv = dataManager.getI18nText( "Report.time" );
                break;
            case ATTACKER:
                rv = dataManager.getI18nText( "Report.attacker" );
                break;
            case DEFENDER:
                rv = dataManager.getI18nText( "Report.defender" );
                break;
            case PRIORITY:
                rv = dataManager.getI18nText( "Report.priority" );
                break;
            case NOTES:
                rv = "/";
                break;
            case ASTRO:
                break;
        }
        return rv;
    }


    @Override
    public Class<?> getColumnClass( int columnIndex )
    {
        Class<?> rv = String.class;
        switch( columnIndex )
        {
            case PLANET:
            case TIME:
            case ATTACKER:
            case DEFENDER:
                rv = String.class;
                break;
            case PRIORITY:
                rv = Status.class;
                break;
            case NOTES:
                rv = Note.class;
                break;
            case ASTRO:
                rv = Icon.class;
                break;
        }
        return rv;
    }

    /**
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt( int rowIndex, int columnIndex )
    {
        Object rv = "";
        BattleReport report = data.get( rowIndex );
        StringBuilder sb;
        switch( columnIndex )
        {
            case PLANET:
                sb = new StringBuilder();
                sb.append( Position.createFromId( report.getPosition() ).toString() );
                if(report.getPlanetName() != null) {
                    sb.append(" ").append(report.getPlanetName());
                }
                rv = sb.toString();
                break;
            case ASTRO:
                if( report.isAsteroid() ) {
                    rv = Resources.getIcon( "/moon.png" );
                }
                break;
            case ATTACKER:
                rv = report.getAttacker();
                break;
            case DEFENDER:
                rv = report.getDefender();
                break;
            case PRIORITY:
                rv = report.getFlag();
                break;
            case NOTES:
                rv = report.getNotes();
                break;
            case TIME:
                rv = DataManager.getDateFormat().format( new Date( report.getCreationTime() ) );
        }
        return rv;
    }

    public boolean isCellEditable(int rowIndex, int columnIndex)
    {
        return ( columnIndex == PRIORITY );
    }

    public void setValueAt(Object value, int rowIndex, int columnIndex)
    {
        switch( columnIndex )
        {
            case PRIORITY:
                Priority priority = null;
                if( value instanceof Priority)
                {
                    priority = ( Priority )value;
                }
                BattleReport report = data.get( rowIndex );
                report.setFlag( priority );
                try
                {
                    dataManager.save( report );
                    dataManager.commit();
                }
                catch( SQLException e )
                {
                    e.printStackTrace();
                }
                break;
        }
    }

    public BattleReport getRow( int row )
    {
        return data.get( row );
    }

    public void removeRows( int[] modelRows ) throws SQLException
    {
        // delete back to front to preserve indices
        Arrays.sort( modelRows );
        List<BattleReport> oldReports = new ArrayList<BattleReport>();
        for( int i = modelRows.length - 1; i >= 0; i-- )
        {
            oldReports.add(  data.remove( modelRows[i] ) );
        }
        dataManager.removeRecords( oldReports );
        dataManager.commit();
        fireTableDataChanged();
    }

    public List<BattleReport> getExportRows( int[] modelRows )
    {
        List<BattleReport> result = new ArrayList<BattleReport>();
        for( int i = 0; i < modelRows.length; i++ )
        {
            result.add( data.get( modelRows[i] ) );
        }
        return result;
    }

    public List<BattleReport> getExportRows()
    {
        List<BattleReport> result = new ArrayList<BattleReport>();
        result.addAll( data );
        return result;
    }

    public void removeAll() throws SQLException
    {
        dataManager.removeRecords( data );
        dataManager.commit();
        data.clear();
        fireTableDataChanged();
    }

    public void setData( Collection<BattleReport> newData )
    {
        data.clear();
        data.addAll( newData );
        fireTableDataChanged();
    }

    public void update()
    {
        data.clear();
        // repaint header
        fireTableStructureChanged();
    }
}
