/****************************************
 *  COPYRIGHT (C) 2008
 *  Holger Graf
 ****************************************/
package siarchive.table;

import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.Icon;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.components.EnumWrapper;
import siarchive.components.Flavor;
import siarchive.i18n.Language;
import siarchive.persistence.Account;



/**
 * @author graf
 *
 */
public class AccountTableModel extends AbstractTableModel implements MouseListener
{
    private static final long serialVersionUID = -1405529512618782948L;

    List<Account> data;
    private Icon upArrow;
    private Icon downArrow;

    protected MainFrame mainFrame;

    public final static int NAME = 0;
    public final static int DEFAULT = 1;
    public final static int LANGUAGE = 2;
    public final static int VERSION = 3;
    public final static int OUTPOST = 4;
    public final static int DOWN = 5;
    public final static int UP = 6;
    public final static int LAST = UP + 1;

    public AccountTableModel( MainFrame mainFrame )
    {
        this.mainFrame = mainFrame;
        upArrow = DataManager.getIcon( "/button_up.gif" );
        downArrow = DataManager.getIcon( "/button_down.gif" );
        try
        {
            data = new ArrayList<Account>( mainFrame.getDataManager().getAccounts() );
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
    }

    /**
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount()
    {
        return LAST;
    }


    /**
     * @see javax.swing.table.AbstractTableModel#getColumnName(int)
     */
    @Override
    public String getColumnName( int columnIndex )
    {
        String rv ="";
        DataManager cd = mainFrame.getDataManager();

        switch( columnIndex )
        {
            case NAME:
                rv = cd.getI18nText( "Account.name" );
                break;
            case DEFAULT:
                rv = cd.getI18nText( "Account.isdefault" );
                break;
            case LANGUAGE:
                rv = cd.getI18nText( "Account.language" );
                break;
            case VERSION:
                rv = cd.getI18nText( "Account.version" );
                break;
            case OUTPOST:
                rv = cd.getI18nText( "Account.outpost" );
                break;
            case DOWN:
            case UP:
                rv = "";
                break;
        }
        return rv;
    }


    /**
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount()
    {
        return data.size();
    }

    /**
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt( int rowIndex, int columnIndex )
    {
        Account target = data.get( rowIndex );
        DataManager cd = mainFrame.getDataManager();
        Object rv = null;
        switch( columnIndex )
        {
            case NAME:
                rv = target.getName();
                break;
            case DEFAULT:
                rv = target.isDefault();
                break;
            case LANGUAGE:
                rv = cd.getI18nText( target.getLanguage().name() );
                break;
            case VERSION:
                rv = cd.getI18nText( target.getFlavor().name() );
                break;
            case OUTPOST:
                rv = target.isOutpost();
                break;
            case DOWN:
                if( rowIndex < data.size() - 1 )
                {
                    rv = downArrow;
                }
                break;
            case UP:
                if( rowIndex > 0 )
                {
                    rv = upArrow;
                }
                break;
        }
        return rv;
    }

    @Override
	public boolean isCellEditable(int rowIndex, int columnIndex)
	{
    	boolean rv = true;

    	if( columnIndex == UP || columnIndex == DOWN )
    	{
    		rv = false;
    	}

		return rv;
	}


	@SuppressWarnings("unchecked")
    @Override
	public void setValueAt(Object value, int rowIndex, int columnIndex)
	{
        Account target = data.get( rowIndex );
        switch( columnIndex )
        {
            case NAME:
                target.setName( value.toString() );
                break;
            case DEFAULT:
                boolean newDefault = Boolean.parseBoolean( value.toString() );
                if( newDefault )
                {
                    // reset others
                    for( Account account : data )
                    {
                        account.setDefault( false );
                    }
                }
                target.setDefault( newDefault );
                fireTableDataChanged();
                break;
            case LANGUAGE:
                target.setLanguage( ( (EnumWrapper<Language>)value).getBase() );
                break;
            case VERSION:
                target.setFlavor( ( (EnumWrapper<Flavor>)value).getBase() );
                break;
            case OUTPOST:
                boolean newOutpost = Boolean.parseBoolean( value.toString() );
                target.setOutpost(newOutpost);
                break;
        }

        try
        {
            mainFrame.getDataManager().save( target );
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
	}

    public void addRow( Account target )
    {
        DataManager cd = mainFrame.getDataManager();
        int lastRow = data.size();
        data.add( target );
        fireTableRowsInserted( lastRow, lastRow );
        try
        {
            cd.save( target );
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
    }

    public void removeRows( int[] rows )
    {
        // since deleting rows will change the indices we'll get the objects first
        List<Account> rows2delete = new ArrayList<Account>();
        DataManager cd = mainFrame.getDataManager();
        for( int row :rows )
        {
            rows2delete.add( data.get( row ) );
        }
        try
        {
            for( Account account : rows2delete )
            {
                cd.delete( account );
            }
            data.removeAll( rows2delete );
        }
        catch( SQLException ex )
        {
            try
            {
                cd.rollback();
            }
            catch( SQLException e )
            {}
            mainFrame.showErrorInfo( ex );
        }
        fireTableDataChanged();
    }

    public boolean hasDefault()
    {
        boolean rv = false;
        for( Account account : data )
        {
            if( account.isDefault() )
            {
                rv = true;
                break;
            }
        }
        return rv;
    }

    public void mouseClicked( MouseEvent e )
    {
        Object o = e.getSource();
        if( o instanceof JTable )
        {
            JTable table = (JTable)o;
            int row = table.rowAtPoint(e.getPoint());
            int col = table.columnAtPoint(e.getPoint());
            // convert to model
            col = table.convertColumnIndexToModel( col );
            if ( SwingUtilities.isLeftMouseButton(e) && e.getClickCount() == 1 )
            {
                if( col == UP || col == DOWN )
                {
                    int start = -1;
                    int stop = -1;
                    switch( col )
                    {
                        case UP:
                            start = row - 1;
                            stop = row + 1;
                            break;
                        case DOWN:
                            start = row;
                            stop = row + 2;
                            break;
                    }

                    if( start >= 0 && stop <= data.size() )
                    {
                        DataManager cd = mainFrame.getDataManager();
                        // switch them over
                        Collections.rotate( data.subList( start, stop ), 1 );
                        int index = data.get( start ).getIndex();
                        data.get( start ).setIndex( data.get( stop - 1 ).getIndex() );
                        data.get( stop - 1 ).setIndex( index );
                        try
                        {
                            cd.save( data.get( start ) );
                            cd.save( data.get( stop - 1 ) );
                            fireTableRowsUpdated( start, stop - 1 );
                            if( col == UP)
                            {
                                table.getSelectionModel().setSelectionInterval( start, start );
                            }
                            else
                            {
                                table.getSelectionModel().setSelectionInterval( stop - 1, stop - 1 );
                            }
                        }
                        catch( SQLException ex)
                        {
                            mainFrame.showErrorInfo( ex );
                        }
                    }
                }
            }
        }
    }

    public void mouseEntered( MouseEvent e )
    {}

    public void mouseExited( MouseEvent e )
    {}

    public void mousePressed( MouseEvent e )
    {}

    public void mouseReleased( MouseEvent e )
    {}
}
