/****************************************
 *  COPYRIGHT (C) 2012-2024
 *  Holger Graf
 ****************************************/
package siarchive.search;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.persistence.Account;
import siarchive.table.ButtonTableCellRenderer;
import siarchive.table.MinimumRowHeightTable;
import siarchive.table.RowHeaderTableModel;

/**
 * @author graf
 *
 */
public abstract class SearchPane extends JPanel
{
    private static final long serialVersionUID = -1982149770393241652L;
    private MainFrame mainFrame;
    private JPanel center;
    private DataManager dataManager;
    private Account currentAccount = null;
    private JButton searchButton = new JButton();
    private JTable rowHeader;
    protected final int ROW_HEADER_WIDTH = 48;


    public SearchPane( MainFrame parent )
    {
        super();
        this.mainFrame = parent;
        this.dataManager = parent.getDataManager();
        setLayout( new BorderLayout() );
        setBorder( null );
        center = createCenterPanel();
        add( center, BorderLayout.CENTER );
        searchButton.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionSearch();
            }
        } );
    }

    public MainFrame getMainFrame() {
        return mainFrame;
    }

    public File getCurrentDir()
    {
        return dataManager.getCurrentDir();
    }

    public void setCurrentDir( File currentDir )
    {
        dataManager.setCurrentDir( currentDir );
    }

    public String getExtension()
    {
        return dataManager.getExtension();
    }

    public void update()
    {
        getMainFrame().getRootPane().setDefaultButton( searchButton );
        searchButton.setText( dataManager.getI18nText( "Search.searchCommand" ) );
        updateLabel();
        updateTable();
        currentAccount = getDataManager().getActiveAccount();
    }

    protected DataManager getDataManager() {
        return dataManager;
    }

    protected boolean accountChanged()
    {
        Account newAccount = getDataManager().getActiveAccount();
        return( currentAccount != null && !currentAccount.equals( newAccount )
         || newAccount != null && !newAccount.equals( currentAccount )
        );
    }

    protected void adjustColumnWidth( int modelIndex, int width )
    {
        JTable table = getTable();
        TableColumnModel columnModel = table.getColumnModel();
        TableColumn column;
        int index = table.convertColumnIndexToView( modelIndex );
        if( index != -1 )
        {
            // column is visible
            column = columnModel.getColumn( index );
            column.setMinWidth( width );
            column.setMaxWidth( width );
            column.setPreferredWidth( width );
        }
    }

    protected JButton getSearchButton() {
        return searchButton;
    }

    protected JTable createRowHeaderTable() {
        rowHeader = new MinimumRowHeightTable( new RowHeaderTableModel() )
        {
            private static final long serialVersionUID = -8707317721680383319L;

            public Dimension getPreferredScrollableViewportSize()
            {
                Dimension size = getPreferredSize();
                size.width = ROW_HEADER_WIDTH;
                return size;
            }
        };
        rowHeader.setAutoResizeMode( JTable.AUTO_RESIZE_OFF );
        rowHeader.setRowSelectionAllowed(false);
        DefaultTableCellRenderer rowRenderer = new ButtonTableCellRenderer();
        rowRenderer.setHorizontalAlignment( SwingConstants.RIGHT );
        rowHeader.setDefaultRenderer( Integer.class, rowRenderer );
        TableColumnModel columnModel = rowHeader.getColumnModel();
        TableColumn column;
        column = columnModel.getColumn( 0 );
        column.setMinWidth( ROW_HEADER_WIDTH );
        column.setMaxWidth( ROW_HEADER_WIDTH );
        column.setPreferredWidth( ROW_HEADER_WIDTH );

        return rowHeader;
    }

    protected void linkToRowHeader( final JTable table ) {
        final RowHeaderTableModel rowModel = (RowHeaderTableModel)rowHeader.getModel();
        table.getModel().addTableModelListener( new TableModelListener() {
            public void tableChanged( TableModelEvent e )
            {
                final TableModel tableModel = ( TableModel )e.getSource();
                rowModel.setRows( tableModel.getRowCount() );
            }
        });

//        // extend selection to row header
//        final ListSelectionListener l = new ListSelectionListener() {
//            @Override
//            public void valueChanged( ListSelectionEvent e ) {
//                if (e.getValueIsAdjusting())
//                    return;
//                ListSelectionModel selectionModel = rowHeader.getSelectionModel();
//                int[] selection = table.getSelectionModel().getSelectedIndices();
//                selectionModel.setValueIsAdjusting(true);
//                selectionModel.clearSelection();
//                for(int row : selection) {
//                    selectionModel.addSelectionInterval(row, row);
//                }
//                selectionModel.setValueIsAdjusting(false);
//            }
//        };
//        table.getSelectionModel().addListSelectionListener(l);
    }

    protected void updateColumnNames() {
        JTable table = getTable();
        TableColumnModel columnModel = table.getColumnModel();
        TableColumn column;
        // force reevaluation of table header texts
        for( int i = 0; i < columnModel.getColumnCount(); i++ )
        {
            column = columnModel.getColumn( i );
            column.setHeaderValue( table.getColumnName( i ) );
        }
    }

    protected abstract JPanel createCenterPanel();
    protected abstract void actionSearch();
    protected abstract JTable getTable();
    protected abstract void updateTable();
    protected abstract void updateLabel();

}
