/****************************************
 *  COPYRIGHT (C) 2010-2024
 *  Holger Graf
 ****************************************/
package siarchive.search;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ScrollPaneConstants;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import org.w3c.dom.Document;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.Parser;
import siarchive.components.Cost;
import siarchive.components.EnumWrapper;
import siarchive.components.ImportExportPane;
import siarchive.components.ModulePane;
import siarchive.components.NoteBox;
import siarchive.components.NumberComparator;
import siarchive.components.Position;
import siarchive.components.PositionField;
import siarchive.components.SearchOption;
import siarchive.components.Status;
import siarchive.persistence.Alliance;
import siarchive.persistence.Note;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.table.AllianceCellRenderer;
import siarchive.table.AlternateLineTableCellRenderer;
import siarchive.table.DebrisTableCellRenderer;
import siarchive.table.IconTableCellRenderer;
import siarchive.table.NotesTableCellRenderer;
import siarchive.table.PlanetTableModel;
import siarchive.table.PlayerCellRenderer;
import siarchive.table.PointsRenderer;
import siarchive.table.TooltipTable;

/**
 * @author graf
 *
 */
@SuppressWarnings("rawtypes")
public class PlanetPane extends SearchPane implements ModulePane, ImportExportPane
{
    private static final long serialVersionUID = -2063793108262189275L;

    private PlanetTableModel model;
    private List<Planet> exportList;

    private JLabel searchLabel;
    private JComboBox box;
    private JButton deleteSelected;
    private JButton deleteAll;
    private JButton exportSelected;
    private JButton exportAll;

    private JTextField alliance;
    private JTextField player;
    private JComboBox playerState;
    private JTextField planet;
    private JComboBox planetState;
    private PositionField startPosition;
    private PositionField stopPosition;
    private JComboBox positionState;
    private JSpinner startPoints;
    private JSpinner stopPoints;
    private JComboBox pointsState;
    private CardLayout cardLayout;
    private JPanel cardPanel;
    private TableRowSorter<TableModel> sorter;

    private JMenuItem popupGoto;
    private JMenuItem popupAlliance;
    private JMenuItem popupPlayer;
    private JMenuItem popupSystem;
    private JMenuItem popupSpyReports;
    private JMenuItem popupBattleReports;
    private JTable table;
    private Planet menuPlanet;
    private NumberComparator numberComparator;
    private PlanetComparator planetComparator;

    /**
     * @param parent
     */
    public PlanetPane( MainFrame parent )
    {
        super(parent);
    }

    protected JPanel createCenterPanel()
    {
        exportList = new ArrayList<Planet>();
        searchLabel = new JLabel();
        box = new JComboBox();
        deleteSelected = new JButton();
        deleteAll = new JButton();
        exportSelected = new JButton();
        exportAll = new JButton();
        alliance = new JTextField();
        player = new JTextField();
        playerState = new JComboBox();
        planet = new JTextField();
        planetState = new JComboBox();
        startPosition = new PositionField( false );
        stopPosition = new PositionField( false );
        positionState = new JComboBox();
        startPoints = new JSpinner( new SpinnerNumberModel( 0, 0, Integer.MAX_VALUE, 1 ) );
        stopPoints = new JSpinner( new SpinnerNumberModel( 0, 0, Integer.MAX_VALUE, 1 ) );
        pointsState = new JComboBox();
        popupGoto = new JMenuItem();
        popupAlliance = new JMenuItem();
        popupPlayer = new JMenuItem();
        popupSystem = new JMenuItem();
        popupSpyReports = new JMenuItem();
        popupBattleReports = new JMenuItem();
        menuPlanet = null;
        numberComparator = new NumberComparator();
        planetComparator = new PlanetComparator();

        final MainFrame mainFrame = getMainFrame();
        JPanel centerPanel = new JPanel( new BorderLayout() );
        JPanel controlPanel = new JPanel( new BorderLayout() );
        controlPanel.setBorder( BorderFactory.createEmptyBorder( 2, 3, 15, 3) );
        JPanel boxPanel = new JPanel( new GridLayout( 2, 1, 0, 0 ) );
        updateSearchOptions();
        boxPanel.add( searchLabel );
        boxPanel.add( box );
        controlPanel.add( boxPanel, BorderLayout.NORTH );

        cardLayout = new CardLayout();
        JPanel optionPanel = new JPanel( new BorderLayout() );
        cardPanel = new JPanel( cardLayout );
        JPanel panel;
        panel = new JPanel( new BorderLayout() );
        panel.add( alliance, BorderLayout.NORTH );
        cardPanel.add( panel, SearchOption.alliances.name() );
        panel = new JPanel( new GridLayout( 3, 1 ) );
        panel.add( player );
        panel.add( playerState );
        panel.add( new JLabel() );
        cardPanel.add( panel, SearchOption.players.name() );
        panel = new JPanel( new GridLayout( 3, 1) );
        panel.add( planet );
        panel.add( planetState );
        panel.add( new JLabel() );
        cardPanel.add( panel, SearchOption.planets.name() );
        panel = new JPanel(new GridLayout( 3, 1 ));
        panel.add( startPosition );
        panel.add( stopPosition );
        panel.add( positionState );
        cardPanel.add( panel, SearchOption.systems.name() );
        panel = new JPanel(new GridLayout( 3, 1 ));
        panel.add( startPoints );
        panel.add( stopPoints );
        panel.add( pointsState );
        cardPanel.add( panel, SearchOption.points.name() );
        optionPanel.add( cardPanel, BorderLayout.NORTH );
        JPanel buttonPanel = new JPanel( new BorderLayout() );
        buttonPanel.add( getSearchButton(), BorderLayout.NORTH );
        optionPanel.add( buttonPanel, BorderLayout.CENTER );
        controlPanel.add( optionPanel, BorderLayout.CENTER );

        box.addItemListener( new ItemListener()
        {
            @SuppressWarnings("unchecked")
            public void itemStateChanged( ItemEvent e )
            {
                if( e.getStateChange() == ItemEvent.SELECTED )
                {
                    SearchOption searchOption = ((EnumWrapper<SearchOption>)e.getItem()).getBase();
                    cardLayout.show( cardPanel, searchOption.name() );
                    switch ( searchOption )
                    {
                        case alliances:
                            sorter.toggleSortOrder( PlanetTableModel.ALLIANCE );
                            break;
                        case players:
                            sorter.toggleSortOrder( PlanetTableModel.OWNER );
                            break;
                        case planets:
                        case systems:
                            sorter.toggleSortOrder( PlanetTableModel.PLANET );
                            break;
                        case points:
                            sorter.toggleSortOrder( PlanetTableModel.POINTS );
                            break;
                        default:
                            break;
                    }
                }
            }

        });

        buttonPanel = new JPanel( new GridLayout( 4, 1, 0, 0 ) );
        buttonPanel.add( exportSelected );
        buttonPanel.add( exportAll );
        buttonPanel.add( deleteSelected );
        buttonPanel.add( deleteAll );

        controlPanel.add( buttonPanel, BorderLayout.SOUTH );
        centerPanel.add( controlPanel, BorderLayout.WEST );

        final JTable rowHeader = createRowHeaderTable();
        model = new PlanetTableModel( getDataManager() );
        table = new TooltipTable( model );
        linkToRowHeader(table);

        JScrollPane scrollPane = new JScrollPane( table,
                        JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                        JScrollPane.HORIZONTAL_SCROLLBAR_NEVER );
        table.setDefaultRenderer( String.class, new AlternateLineTableCellRenderer() );
        table.setDefaultRenderer( Alliance.class, new AllianceCellRenderer() );
        table.setDefaultRenderer( Player.class, new PlayerCellRenderer() );
        table.setDefaultRenderer( Planet.class, new AlternateLineTableCellRenderer() );
        table.setDefaultRenderer( Number.class, new PointsRenderer() );
        table.setDefaultRenderer( Icon.class, new IconTableCellRenderer() );
        table.setDefaultRenderer( Note.class, new NotesTableCellRenderer() );
        table.setDefaultRenderer( Cost.class, new DebrisTableCellRenderer() );
        sorter = new TableRowSorter<TableModel>();
        sorter.setModel( model );
        sorter.setComparator( PlanetTableModel.POINTS, numberComparator );
        sorter.setComparator( PlanetTableModel.PLANET, planetComparator );
        sorter.setMaxSortKeys( 3 );
        table.setRowSorter( sorter );
        scrollPane.setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );
        scrollPane.setRowHeaderView( rowHeader );
        scrollPane.setCorner( ScrollPaneConstants.UPPER_LEFT_CORNER, rowHeader.getTableHeader() );
        centerPanel.add( scrollPane, BorderLayout.CENTER );

        table.getSelectionModel().addListSelectionListener( new ListSelectionListener()
        {
            public void valueChanged( ListSelectionEvent e )
            {
                if( !e.getValueIsAdjusting() )
                {
                    boolean enable = ( e.getFirstIndex() >= 0 );
                    deleteSelected.setEnabled( enable );
                    exportSelected.setEnabled( enable );
                }
            }
        });
        popupGoto.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionGoto();
            }
        } );
        popupAlliance.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowAlliance();
            }
        } );
        popupPlayer.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowPlayer();
            }
        } );
        popupSystem.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowSystem();
            }
        } );
        popupBattleReports.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowBattleReports();
            }
        } );
        popupSpyReports.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowSpyReports();
            }
        } );
        table.addMouseListener( new MouseAdapter()
        {
            @Override
            public void mousePressed( MouseEvent event )
            {
                if( event.isPopupTrigger() )
                {
                    showPopup( event );
                }
            }

            @Override
            public void mouseReleased( MouseEvent event )
            {
                if( event.isPopupTrigger() )
                {
                    showPopup( event );
                }
            }

            @Override
            public void mouseClicked( MouseEvent event )
            {
                if( event.getClickCount() >= 2 )
                {
                    Point point = new Point( event.getX(), event.getY() );
                    int modelColumn = table.convertColumnIndexToModel( table.columnAtPoint( point ) );
                    int modelRow = table.convertRowIndexToModel( table.rowAtPoint( point ) );
                    switch( modelColumn )
                    {
                        case PlanetTableModel.PLANETNOTES:
                        {
                            Planet planet = model.getRow( modelRow );
                            NoteBox<Planet> box = new NoteBox<Planet>( mainFrame, planet );
                            mainFrame.showModalBox( box );
                            model.fireTableCellUpdated( modelRow, modelColumn );
                            mainFrame.updatePane();
                        }
                        break;
                        case PlanetTableModel.OWNERNOTES:
                        {
                            Player player = model.getRow( modelRow ).getOwner();
                            if( player != null )
                            {
                                NoteBox<Player> box = new NoteBox<Player>( mainFrame, player );
                                mainFrame.showModalBox( box );
                                model.fireTableDataChanged();
                            }
                        }
                        break;
                        case PlanetTableModel.ALLIANCENOTES:
                        {
                            Player player = model.getRow( modelRow ).getOwner();
                            if( player != null )
                            {
                                Alliance alliance = player.getAlliance();
                                if( alliance != null )
                                {
                                    NoteBox<Alliance> box = new NoteBox<Alliance>( mainFrame, alliance );
                                    mainFrame.showModalBox( box );
                                    model.fireTableDataChanged();
                                }
                            }
                        }
                        break;
                        default:
                        {
                            menuPlanet = model.getRow( modelRow );
                            actionGoto();
                        }
                        break;
                    }
                }
            }

            private void showPopup( MouseEvent event )
            {
                Point menuPosition = new Point( event.getX(), event.getY() );
                int viewRow = table.rowAtPoint( menuPosition );
                int modelRow = table.convertRowIndexToModel( viewRow );
                if( modelRow >= 0  )
                {
                    menuPlanet = model.getRow( modelRow );

                    JPopupMenu popup = new JPopupMenu();
                    popup.add( popupGoto );
                    if( menuPlanet.getOwner() != null )
                    {
                        popup.add( popupPlayer );
                        if( menuPlanet.getOwner().getAlliance() != null )
                        {
                            popup.add( popupAlliance );
                        }
                    }
                    popup.add( popupSystem );
                    popup.add( popupSpyReports );
                    popup.add( popupBattleReports );
                    popup.pack();

                    popup.show(table, event.getX(), event.getY() );
                }
            }


        } );
        deleteAll.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                mainFrame.dbOn();
                try
                {
                    model.removeAll();
                }
                catch( SQLException ex )
                {
                    mainFrame.showErrorInfo( ex );
                }
                mainFrame.dbOff();
                mainFrame.updatePane();
            }

        });
        deleteSelected.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                int[] rows = table.getSelectedRows();
                int[] modelRows = new int[rows.length];
                for( int i = 0; i < rows.length; i++ )
                {
                    modelRows[i] = table.convertRowIndexToModel( rows[i] );
                }
                mainFrame.dbOn();
                try
                {
                    model.removeRows( modelRows );
                }
                catch( SQLException ex )
                {
                    mainFrame.showErrorInfo( ex );
                }
                mainFrame.dbOff();
                mainFrame.updatePane();
            }

        });
        exportSelected.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                int[] rows = table.getSelectedRows();
                int[] modelRows = new int[rows.length];
                for( int i = 0; i < rows.length; i++ )
                {
                    modelRows[i] = table.convertRowIndexToModel( rows[i] );
                }
                exportList.clear();
                exportList.addAll( model.getExportRows( modelRows ) );
                getDataManager().saveData( PlanetPane.this );
            }

        });
        exportAll.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                exportList.clear();
                exportList.addAll( model.getExportRows() );
                getDataManager().saveData( PlanetPane.this );
            }

        });

        return centerPanel;
    }

    protected void actionShowSpyReports()
    {
        JTabbedPane pane = getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( getDataManager().getI18nText( "MainFrame.spyReports" ) );
        if( tabIndex >= 0 )
        {
            SpyReportSummaryPane summaryPane = ( SpyReportSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowSystem()
    {
        JTabbedPane pane = getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( getDataManager().getI18nText( "MainFrame.planets" ) );
        if( tabIndex >= 0 )
        {
            PlanetSummaryPane summaryPane = ( PlanetSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( Position.createFromId( menuPlanet.getPosition() ) );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowPlayer()
    {
        JTabbedPane pane = getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( getDataManager().getI18nText( "MainFrame.planets" ) );
        if( tabIndex >= 0 )
        {
            PlanetSummaryPane summaryPane = ( PlanetSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet.getOwner() );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowAlliance()
    {
        JTabbedPane pane = getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( getDataManager().getI18nText( "MainFrame.planets" ) );
        if( tabIndex >= 0 )
        {
            PlanetSummaryPane summaryPane = ( PlanetSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet.getOwner().getAlliance() );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowBattleReports()
    {
        JTabbedPane pane = getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( getDataManager().getI18nText( "MainFrame.battleReports" ) );
        if( tabIndex >= 0 )
        {
            BattleReportSummaryPane summaryPane = ( BattleReportSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionGoto()
    {
        Position position = Position.createFromId( menuPlanet.getPosition() );
        getMainFrame().showGalaxyView( position.getGalaxy(), position.getSystem(), true );
    }

    @SuppressWarnings("unchecked")
    protected void actionSearch()
    {
        MainFrame mainFrame = getMainFrame();
        DataManager dataManager = getDataManager();
        SearchOption searchOption = ((EnumWrapper<SearchOption>)box.getSelectedItem()).getBase();
        Status state;
        Collection<Planet> result = null;
        mainFrame.dbOn();
        try
        {
            switch( searchOption )
            {
                case alliances:
                    result = dataManager.findPlanetsByAlliance( alliance.getText() );
                    break;
                case planets:
                    state = ((EnumWrapper<Status>)planetState.getSelectedItem()).getBase();
                    result = dataManager.findPlanetsByNameAndStatus( planet.getText(), state );
                    break;
                case players:
                    state = ((EnumWrapper<Status>)playerState.getSelectedItem()).getBase();
                    result = dataManager.findPlanetsByPlayerNameAndStatus( player.getText(), state );
                    break;
                case points:
                    state = ((EnumWrapper<Status>)pointsState.getSelectedItem()).getBase();
                    result = dataManager.findPlanetsByPlayerPointsAndStatus( ((Number)startPoints.getValue()).longValue(), ((Number)stopPoints.getValue()).longValue(), state );
                    break;
                case systems:
                    state = ((EnumWrapper<Status>)positionState.getSelectedItem()).getBase();
                    result = dataManager.findPlanetsBySystemsAndStatus( startPosition.getPosition( 1 ), stopPosition.getPosition( 16 ), state );
                    break;
            }
        }
        catch( Exception ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        mainFrame.dbOff();
        if( result != null )
        {
            model.setData( result );
        }
    }


    public void openPosition( Position position )
    {
        update();
        this.startPosition.setPosition( position );
        this.stopPosition.setPosition( position );
        this.positionState.setSelectedIndex( 0 );
        this.box.setSelectedItem( new EnumWrapper<SearchOption>( getDataManager(), SearchOption.systems ) );
        cardLayout.show( cardPanel, SearchOption.systems.name() );
        actionSearch();
    }

    public void openPlayer( Player player )
    {
        update();
        this.player.setText( player.getName() );
        this.playerState.setSelectedIndex( 0 );
        this.box.setSelectedItem( new EnumWrapper<SearchOption>( getDataManager(), SearchOption.players ) );
        cardLayout.show( cardPanel, SearchOption.players.name() );
        actionSearch();
    }

    public void openAlliance( Alliance alliance )
    {
        update();
        this.alliance.setText( alliance.getName() );
        this.box.setSelectedItem( new EnumWrapper<SearchOption>( getDataManager(), SearchOption.alliances ) );
        cardLayout.show( cardPanel, SearchOption.alliances.name() );
        actionSearch();
    }

    protected void updateTable()
    {
        // clear table on account change only
        if( accountChanged() )
        {
            model.update();
            sorter.setComparator( PlanetTableModel.POINTS, numberComparator );
            sorter.setComparator( PlanetTableModel.PLANET, planetComparator );
            sorter.setMaxSortKeys( 3 );
        }
        int width;
        width = 20;
        adjustColumnWidth( PlanetTableModel.OWNERNOTES, width );
        adjustColumnWidth( PlanetTableModel.ALLIANCENOTES, width );
        adjustColumnWidth( PlanetTableModel.PLANETNOTES, width );
        width = 120;
        adjustColumnWidth( PlanetTableModel.STATUS, width );
        width = 75;
        adjustColumnWidth( PlanetTableModel.POINTS, width );
        width = 16;
        adjustColumnWidth( PlanetTableModel.DEBRIS, width );
        adjustColumnWidth( PlanetTableModel.EVENT, width );
        adjustColumnWidth( PlanetTableModel.ASTRO, width );
        width = 39;
        adjustColumnWidth( PlanetTableModel.ALLIANCETAG, width );

        updateColumnNames();
    }

    @Override
    protected JTable getTable()
    {
        return table;
    }

    protected void updateLabel()
    {
        if( accountChanged() )
        {
            updateSearchOptions();
        }

        DataManager dataManager = getDataManager();
        searchLabel.setText( dataManager.getI18nText( "Search.search" ) );
        deleteSelected.setText( dataManager.getI18nText( "delete" ) );
        deleteAll.setText( dataManager.getI18nText( "deleteAll" ) );
        exportSelected.setText( dataManager.getI18nText( "export" ) );
        exportAll.setText( dataManager.getI18nText( "exportAll" ) );
        popupGoto.setText( dataManager.getI18nText( "Popup.goto" ) );
        popupAlliance.setText( dataManager.getI18nText( "Popup.showAlliance" ) );
        popupPlayer.setText( dataManager.getI18nText( "Popup.showPlayer" ) );
        popupSystem.setText( dataManager.getI18nText( "Popup.showSystem" ) );
        popupSpyReports.setText( dataManager.getI18nText( "Popup.spyReports" ) );
        popupBattleReports.setText( dataManager.getI18nText( "Popup.battleReports" ) );

        deleteSelected.setEnabled( false );
        deleteAll.setEnabled( true );
        exportSelected.setEnabled( false );
        exportAll.setEnabled( true );
    }

    @SuppressWarnings("unchecked")
    private void updateSearchOptions()
    {
        Object selectedItem = box.getSelectedItem();
        SearchOption option = SearchOption.players;
        DataManager dataManager = getDataManager();
        if( selectedItem != null )
        {
            option = ((EnumWrapper<SearchOption>)selectedItem).getBase();
        }
        box.removeAllItems();
        SearchOption[] options = SearchOption.values();
        for( SearchOption searchOption : SearchOption.forPlanets() )
        {
            box.addItem( new EnumWrapper<SearchOption>( dataManager, searchOption ) );
        }
        box.setSelectedItem( new EnumWrapper<SearchOption>( dataManager, option ) );
        box.setMaximumRowCount( options.length );

        playerState.removeAllItems();
        planetState.removeAllItems();
        pointsState.removeAllItems();
        positionState.removeAllItems();
        Status[] states = Status.values();
        for( Status status : states )
        {
            playerState.addItem( new EnumWrapper<Status>( dataManager, status ) );
            planetState.addItem( new EnumWrapper<Status>( dataManager, status ) );
            pointsState.addItem( new EnumWrapper<Status>( dataManager, status ) );
            positionState.addItem( new EnumWrapper<Status>( dataManager, status ) );
        }
        alliance.setText( "*" );
        player.setText( "*" );
        planet.setText( "*" );
        startPosition.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        startPosition.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        startPosition.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
        stopPosition.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        stopPosition.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        stopPosition.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
    }

    public Document getData( Parser xmlParser )
    {
        Document doc = null;
        if( xmlParser != null && getDataManager().getActiveAccount() != null )
        {
            doc = xmlParser.getDocument();
            getDataManager().createAppRoot( doc );
            getMainFrame().exportPlanets( doc, exportList );
        }
        return doc;
    }

    public String getFileName()
    {
        StringBuilder sb = new StringBuilder();
        sb.append( DataManager.getFileFormat().format( new Date() ) ).append( ' ' );
        sb.append( getDataManager().getI18nText( "Planet" ) );
        return sb.toString();
    }

    protected class PlanetComparator implements Comparator<Planet>
    {
        public int compare( Planet o1, Planet o2 )
        {
            return o1.getPosition() - o2.getPosition();
        }
    }
}
