/****************************************
 *  COPYRIGHT (C) 2021-2024
 *  Holger Graf
 ****************************************/
package siarchive.search;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.RowSorter.SortKey;
import javax.swing.ScrollPaneConstants;
import javax.swing.SortOrder;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import org.w3c.dom.Document;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.Parser;
import siarchive.Resources;
import siarchive.components.AttackStatus;
import siarchive.components.BlackHoleLevel;
import siarchive.components.Cost;
import siarchive.components.EnumWrapper;
import siarchive.components.ImportExportPane;
import siarchive.components.ModulePane;
import siarchive.components.NumberComparator;
import siarchive.components.Position;
import siarchive.components.PositionField;
import siarchive.components.SearchOption;
import siarchive.components.Ship;
import siarchive.persistence.BlackHole;
import siarchive.table.AlternateLineTableCellRenderer;
import siarchive.table.AttackStatusListCellRenderer;
import siarchive.table.BlackHoleLevelListCellRenderer;
import siarchive.table.BlackHoleTableCellRenderer;
import siarchive.table.BlackHoleTableModel;
import siarchive.table.CheckBoxTableCellRenderer;
import siarchive.table.FleetTableCellRenderer;
import siarchive.table.PointsRenderer;
import siarchive.table.PositionRenderer;
import siarchive.table.TooltipTable;

/**
 * @author graf
 *
 */
public class BlackHolePane extends SearchPane implements ImportExportPane, ModulePane {

    private static final long serialVersionUID = 7709964306577830291L;

    private boolean saveSelected;
    private JTable table;
    private BlackHoleTableModel model;
    private TableRowSorter<TableModel> sorter;

    private JLabel searchLabel;
    private JComboBox<EnumWrapper<SearchOption>> searchBox;
    private JComboBox<EnumWrapper<BlackHoleLevel>> levelBox;
    private JComboBox<EnumWrapper<AttackStatus>> attackedBox;
    private JButton deleteSelected;
    private JButton deleteAll;
    private JButton exportSelected;
    private JButton exportAll;
    private CardLayout cardLayout;
    private JPanel cardPanel;

    private PositionField startPosition;
    private PositionField stopPosition;
    private FleetComparator fleetComparator;
    private NumberComparator numberComparator;

    private List<SortKey> sortOrderLevel;
    private List<SortKey> sortOrderAttacked;

    public BlackHolePane( MainFrame parent ) {
        super(parent);
    }

    /**
     * @see siarchive.components.ImportExportPane#getFileName()
     */
    @Override
    public String getFileName() {
        StringBuilder sb = new StringBuilder();
        sb.append( DataManager.getFileFormat().format( new Date() ) ).append( ' ' );
        sb.append( getDataManager().getI18nText( "Blackhole" ) );
        return sb.toString();
    }

    /**
     * @see siarchive.components.ImportExportPane#getData(siarchive.Parser)
     */
    @Override
    public Document getData( Parser xmlParser ) {
        Document doc = null;
        if( xmlParser != null && getDataManager().getActiveAccount() != null )
        {
            List<BlackHole> exportList;
            if(saveSelected) {
                int[] rows = table.getSelectedRows();
                int[] modelRows = new int[rows.length];
                for( int i = 0; i < rows.length; i++ ) {
                    modelRows[i] = table.convertRowIndexToModel( rows[i] );
                }
                exportList = model.getExportRows( modelRows );
            }
            else {
                exportList = model.getExportRows();
            }

            doc = xmlParser.getDocument();
            getDataManager().createAppRoot( doc );
            getMainFrame().exportBlackHoles( doc, exportList );
        }
        return doc;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void actionSearch() {
        MainFrame mainFrame = getMainFrame();
        DataManager dataManager = getDataManager();
        SearchOption searchOption = ((EnumWrapper<SearchOption>)searchBox.getSelectedItem()).getBase();
        Collection<BlackHole> result = null;
        mainFrame.dbOn();
        try
        {
            switch( searchOption )
            {
                case level:
                    result = dataManager.findBlackHolesByLevel(((EnumWrapper<BlackHoleLevel>)levelBox.getSelectedItem()).getBase());
                    break;
                case attackstatus:
                    result = dataManager.findBlackHolesByAttackStatus(((EnumWrapper<AttackStatus>)attackedBox.getSelectedItem()).getBase());
                    break;
                case systems:
                    result = dataManager.findBlackHolesBySystems( startPosition.getPosition( 1 ), stopPosition.getPosition( 16 ));
                    break;
                default:
                    break;
            }
        }
        catch( Exception ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        mainFrame.dbOff();
        if( result != null )
        {
            model.setData( result );
        }
    }

    /**
     * @see siarchive.search.SearchPane#getTable()
     */
    @Override
    protected JTable getTable() {
        return table;
    }

    @Override
    protected JPanel createCenterPanel() {

        searchLabel = new JLabel();
        searchBox = new JComboBox<>();
        deleteSelected = new JButton();
        deleteAll = new JButton();
        exportSelected = new JButton();
        exportAll = new JButton();
        levelBox = new JComboBox<>();
        attackedBox = new JComboBox<>();
        startPosition = new PositionField( false );
        stopPosition = new PositionField( false );
        fleetComparator = new FleetComparator();
        numberComparator = new NumberComparator();
        sortOrderLevel = Arrays.asList(new SortKey(BlackHoleTableModel.LEVEL, SortOrder.ASCENDING),
                                       new SortKey(BlackHoleTableModel.POSITION, SortOrder.ASCENDING));
        sortOrderAttacked = Arrays.asList(new SortKey(BlackHoleTableModel.ATTACKED, SortOrder.ASCENDING),
                                          new SortKey(BlackHoleTableModel.POSITION, SortOrder.ASCENDING));

        attackedBox.setRenderer( new AttackStatusListCellRenderer() );
        levelBox.setRenderer( new BlackHoleLevelListCellRenderer() );
        updateOptions();
        final MainFrame mainFrame = getMainFrame();
        JPanel centerPanel = new JPanel( new BorderLayout() );
        JPanel controlPanel = new JPanel( new BorderLayout() );
        controlPanel.setBorder( BorderFactory.createEmptyBorder( 2, 3, 15, 3) );
        JPanel boxPanel = new JPanel( new GridLayout( 2, 1, 0, 0 ) );
        boxPanel.add( searchLabel );
        boxPanel.add( searchBox );
        controlPanel.add( boxPanel, BorderLayout.NORTH );

        cardLayout = new CardLayout();
        JPanel optionPanel = new JPanel( new BorderLayout() );
        cardPanel = new JPanel( cardLayout );
        JPanel panel;
        panel = new JPanel( new BorderLayout() );
        panel.add( levelBox, BorderLayout.NORTH );
        cardPanel.add( panel, SearchOption.level.name() );
        panel = new JPanel( new BorderLayout() );
        panel.add( attackedBox, BorderLayout.NORTH );
        cardPanel.add( panel, SearchOption.attackstatus.name() );
        panel = new JPanel(new GridLayout( 2, 1 ));
        panel.add( startPosition );
        panel.add( stopPosition );
        cardPanel.add( panel, SearchOption.systems.name() );
        optionPanel.add( cardPanel, BorderLayout.NORTH );
        JPanel buttonPanel = new JPanel( new BorderLayout() );
        buttonPanel.add( getSearchButton(), BorderLayout.NORTH );
        optionPanel.add( buttonPanel, BorderLayout.CENTER );
        controlPanel.add( optionPanel, BorderLayout.CENTER );

        buttonPanel = new JPanel( new GridLayout( 4, 1, 0, 0 ) );
        buttonPanel.add( exportSelected );
        buttonPanel.add( exportAll );
        buttonPanel.add( deleteSelected );
        buttonPanel.add( deleteAll );

        controlPanel.add( buttonPanel, BorderLayout.SOUTH );
        centerPanel.add( controlPanel, BorderLayout.WEST );


        final JTable rowHeader = createRowHeaderTable();
        model = new BlackHoleTableModel( getDataManager() );

        final TableCellRenderer attackRenderer = new CheckBoxTableCellRenderer(Resources.getIcon("/swords.png"), Resources.getIcon("/peace.png"));
        final TableCellRenderer fleetRenderer = new FleetTableCellRenderer();
        final TableCellRenderer levelRenderer = new BlackHoleTableCellRenderer();
        final TableCellRenderer numberRenderer = new PointsRenderer();
        final TableCellRenderer positionRenderer = new PositionRenderer();
        table = new TooltipTable(model) {
            private static final long serialVersionUID = -2950557937827140931L;

            @Override
            public TableCellRenderer getCellRenderer( int row, int column ) {
                TableCellRenderer renderer;
                switch(convertColumnIndexToModel(column)) {
                    case BlackHoleTableModel.ATTACKED:
                        renderer = attackRenderer;
                        break;
                    case BlackHoleTableModel.FLEET:
                        renderer = fleetRenderer;
                        break;
                    case BlackHoleTableModel.LEVEL:
                        renderer = levelRenderer;
                        break;
                    case BlackHoleTableModel.POSITION:
                        renderer = positionRenderer;
                        break;
                    case BlackHoleTableModel.EMBLEMS:
                    case BlackHoleTableModel.POWER:
                    case BlackHoleTableModel.SPICE:
                    case BlackHoleTableModel.STARS:
                        renderer = numberRenderer;
                        break;
                    default:
                        renderer = new AlternateLineTableCellRenderer();
                }
                return renderer;
            }
        };
        linkToRowHeader(table);

        JScrollPane scrollPane = new JScrollPane( table,
                        JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                        JScrollPane.HORIZONTAL_SCROLLBAR_NEVER );
        sorter = new TableRowSorter<TableModel>();
        sorter.setModel( model );
        table.setRowSorter( sorter );
        updateSorter();

        scrollPane.setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );
        scrollPane.setRowHeaderView( rowHeader );
        scrollPane.setCorner( ScrollPaneConstants.UPPER_LEFT_CORNER, rowHeader.getTableHeader() );
        centerPanel.add( scrollPane, BorderLayout.CENTER );

        table.getSelectionModel().addListSelectionListener( new ListSelectionListener() {
            public void valueChanged( ListSelectionEvent e )
            {
                if( !e.getValueIsAdjusting() )
                {
                    boolean enable = ( e.getFirstIndex() >= 0 );
                    deleteSelected.setEnabled( enable );
                    exportSelected.setEnabled( enable );
                }
            }
        });
        table.addMouseListener( new MouseAdapter()
        {
            @Override
            public void mouseClicked( MouseEvent event )
            {
                if( event.getClickCount() >= 2 )
                {
                    Point point = new Point( event.getX(), event.getY() );
                    int modelRow = table.convertRowIndexToModel( table.rowAtPoint( point ) );
                    Position position = Position.createFromId( model.getRow( modelRow ).getPosition() );
                    getMainFrame().showGalaxyView( position.getGalaxy(), position.getSystem(), true );
                }
            }

        } );

        searchBox.addItemListener( new ItemListener() {
            @SuppressWarnings("unchecked")
            public void itemStateChanged( ItemEvent e )
            {
                if( e.getStateChange() == ItemEvent.SELECTED )
                {
                    SearchOption searchOption = ((EnumWrapper<SearchOption>)e.getItem()).getBase();
                    cardLayout.show( cardPanel, searchOption.name() );
                    switch ( searchOption )
                    {
                        case level:
                            sorter.setSortKeys(sortOrderLevel);
                            break;
                        case attackstatus:
                            sorter.setSortKeys(sortOrderAttacked);
                            break;
                        case systems:
                            sorter.toggleSortOrder( BlackHoleTableModel.POSITION );
                            break;
                        default:
                            break;
                    }
                }
            }

        });

        deleteAll.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                mainFrame.dbOn();
                try {
                    model.removeAll();
                }
                catch( SQLException ex )
                {
                    mainFrame.showErrorInfo( ex );
                }
                mainFrame.dbOff();
                mainFrame.updatePane();
            }

        });
        deleteSelected.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                int[] rows = table.getSelectedRows();
                int[] modelRows = new int[rows.length];
                for( int i = 0; i < rows.length; i++ ) {
                    modelRows[i] = table.convertRowIndexToModel( rows[i] );
                }
                mainFrame.dbOn();
                try {
                    model.removeRows( modelRows );
                }
                catch( SQLException ex ) {
                    mainFrame.showErrorInfo( ex );
                }
                mainFrame.dbOff();
                mainFrame.updatePane();
            }

        });
        exportSelected.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                saveSelected = true;
                getDataManager().saveData( BlackHolePane.this );
            }

        });
        exportAll.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                saveSelected = false;
                getDataManager().saveData( BlackHolePane.this );
            }

        });

        return centerPanel;
    }

    private void updateSorter() {
        sorter.setComparator( BlackHoleTableModel.FLEET, fleetComparator );
        sorter.setComparator(BlackHoleTableModel.POSITION, numberComparator);
        sorter.setComparator(BlackHoleTableModel.POWER, numberComparator);
        sorter.setComparator(BlackHoleTableModel.SPICE, numberComparator);
        sorter.setComparator(BlackHoleTableModel.EMBLEMS, numberComparator);
        sorter.setComparator(BlackHoleTableModel.STARS, numberComparator);
        sorter.setMaxSortKeys( 3 );
        sorter.setSortKeys(sortOrderLevel);
    }

    @Override
    protected void updateTable() {
        // clear table on account change only
        if( accountChanged() )
        {
            model.update();
            updateSorter();
        }
        int width;
        width = 45;
        adjustColumnWidth( BlackHoleTableModel.LEVEL, width );
        adjustColumnWidth( BlackHoleTableModel.ATTACKED, width );
        width = 95;
        adjustColumnWidth( BlackHoleTableModel.POWER, width );
        adjustColumnWidth( BlackHoleTableModel.SPICE, width );
        width = 55;
        adjustColumnWidth( BlackHoleTableModel.EMBLEMS, width );
        adjustColumnWidth( BlackHoleTableModel.STARS, width );
        width = 65;
        adjustColumnWidth( BlackHoleTableModel.POSITION, width );

        updateColumnNames();
    }

    @Override
    protected void updateLabel() {
        if( accountChanged() ) {
            updateOptions();
        }

        DataManager dataManager = getDataManager();
        searchLabel.setText( dataManager.getI18nText( "Search.search" ) );
        deleteSelected.setText( dataManager.getI18nText( "delete" ) );
        deleteAll.setText( dataManager.getI18nText( "deleteAll" ) );
        exportSelected.setText( dataManager.getI18nText( "export" ) );
        exportAll.setText( dataManager.getI18nText( "exportAll" ) );

        deleteSelected.setEnabled( false );
        deleteAll.setEnabled( true );
        exportSelected.setEnabled( false );
        exportAll.setEnabled( true );
    }

    @SuppressWarnings("unchecked")
    private void updateOptions() {
        Object selectedItem = searchBox.getSelectedItem();
        SearchOption option = SearchOption.level;
        DataManager dataManager = getDataManager();
        if( selectedItem != null )
        {
            option = ((EnumWrapper<SearchOption>)selectedItem).getBase();
        }
        searchBox.removeAllItems();
        SearchOption[] options = SearchOption.forBlackHoles();
        for( SearchOption searchOption : options )
        {
            searchBox.addItem( new EnumWrapper<SearchOption>( dataManager, searchOption ) );
        }
        searchBox.setSelectedItem( new EnumWrapper<SearchOption>( dataManager, option ) );
        searchBox.setMaximumRowCount( options.length );

        levelBox.removeAllItems();
        for( BlackHoleLevel status : BlackHoleLevel.values()) {
            levelBox.addItem(new EnumWrapper<BlackHoleLevel>( dataManager, status ));
        }
        levelBox.setSelectedIndex(0);
        levelBox.setMaximumRowCount( levelBox.getItemCount());

        attackedBox.removeAllItems();
        for( AttackStatus status : AttackStatus.values()) {
            attackedBox.addItem(new EnumWrapper<AttackStatus>( dataManager, status ));
        }
        attackedBox.setSelectedIndex(0);
        attackedBox.setMaximumRowCount( attackedBox.getItemCount());

        startPosition.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        startPosition.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        startPosition.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
        stopPosition.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        stopPosition.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        stopPosition.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
    }

    protected class FleetComparator implements Comparator<Map<Ship, Long>>
    {
        public int compare( Map<Ship, Long> fleet1, Map<Ship, Long> fleet2 )
        {
            if(fleet1 == fleet2) {
                return 0;
            }
            if(fleet1 == null) {
                return -1;
            }
            else if(fleet2 == null) {
                return 1;
            }
            return spiceValue(fleet1).compareTo(spiceValue(fleet2));
        }

        private Cost spiceValue(Map<Ship, Long> ships) {
            Cost cost = new Cost(0, 0, 0, 0);
            for(Map.Entry<Ship, Long> entry : ships.entrySet()) {
                Cost shipCost = entry.getKey().getCost().multiply(entry.getValue().longValue());
                cost.add(shipCost);
            }
            return cost;
        }
    }

}
