/****************************************
 *  COPYRIGHT (C) 2010-2024
 *  Holger Graf
 ****************************************/
package siarchive.search;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import org.w3c.dom.Document;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.Parser;
import siarchive.components.DateTimeField;
import siarchive.components.EnumWrapper;
import siarchive.components.ImportExportPane;
import siarchive.components.ModulePane;
import siarchive.components.NoteBox;
import siarchive.components.Position;
import siarchive.components.PositionField;
import siarchive.components.Priority;
import siarchive.components.PriorityComparator;
import siarchive.components.SearchOption;
import siarchive.persistence.BattleReport;
import siarchive.persistence.Planet;
import siarchive.reports.BattleReportView;
import siarchive.table.AlternateLineTableCellRenderer;
import siarchive.table.BattleReportTableModel;
import siarchive.table.IconTableCellRenderer;
import siarchive.table.NotesTableCellRenderer;
import siarchive.table.PointsRenderer;
import siarchive.table.PriorityListCellRenderer;
import siarchive.table.PriorityTableCellRenderer;
import siarchive.table.TooltipTable;

/**
 * @author graf
 *
 */
@SuppressWarnings("rawtypes")
public class BattleReportPane extends SearchPane implements ModulePane, ImportExportPane
{
    private static final long serialVersionUID = -2063793108262189275L;

    private JTable table;
    private BattleReportTableModel model;
    private List<BattleReport> exportList;

    private JLabel searchLabel;
    private JComboBox box;
    private JButton deleteSelected;
    private JButton deleteAll;
    private JButton exportSelected;
    private JButton exportAll;

    private PositionField startPosition;
    private PositionField stopPosition;
    private JComboBox positionPriority;
    private DateTimeField startTime;
    private DateTimeField stopTime;
    private JComboBox timePriority;
    private JTextField attacker;
    private JComboBox attackerPriority;
    private JTextField defender;
    private JComboBox defenderPriority;
    private CardLayout cardLayout;
    private JPanel cardPanel;
    private TableRowSorter<TableModel> sorter;
    private PriorityComparator priorityComparator;

    private JMenuItem popupGoto;
    private JMenuItem popupShow;
    private Point menuPosition;

    /**
     * @param parent
     */
    public BattleReportPane( MainFrame parent )
    {
        super(parent);
    }

    protected JPanel createCenterPanel()
    {
        exportList = new ArrayList<BattleReport>();
        searchLabel = new JLabel();
        box = new JComboBox();
        deleteSelected = new JButton();
        deleteAll = new JButton();
        exportSelected = new JButton();
        exportAll = new JButton();
        startPosition = new PositionField( false );
        stopPosition = new PositionField( false );
        positionPriority = new JComboBox();
        startTime = new DateTimeField();
        stopTime = new DateTimeField();
        timePriority = new JComboBox();
        attacker = new JTextField();
        attackerPriority = new JComboBox();
        defender = new JTextField();
        defenderPriority = new JComboBox();
        priorityComparator = new PriorityComparator();
        popupGoto = new JMenuItem();
        popupShow = new JMenuItem();
        menuPosition = null;
        model = new BattleReportTableModel( getDataManager() );
        sorter = new TableRowSorter<TableModel>();
        sorter.setModel( model );

        JPanel centerPanel = new JPanel( new BorderLayout() );
        JPanel controlPanel = new JPanel( new BorderLayout() );
        controlPanel.setBorder( BorderFactory.createEmptyBorder( 2, 3, 15, 3) );
        JPanel boxPanel = new JPanel( new GridLayout( 2, 1, 0, 0 ) );
        updateSearchOptions();
        boxPanel.add( searchLabel );
        boxPanel.add( box );
        controlPanel.add( boxPanel, BorderLayout.NORTH );

        cardLayout = new CardLayout();
        JPanel optionPanel = new JPanel( new BorderLayout() );
        cardPanel = new JPanel( cardLayout );
        JPanel panel;
        panel = new JPanel(new GridLayout( 3, 1 ));
        panel.add( startPosition );
        panel.add( stopPosition );
        panel.add( positionPriority );
        cardPanel.add( panel, SearchOption.systems.name() );
        panel = new JPanel(new GridLayout( 3, 1 ));
        panel.add( startTime );
        panel.add( stopTime );
        panel.add( timePriority );
        cardPanel.add( panel, SearchOption.time.name() );
        panel = new JPanel( new GridLayout( 3, 1) );
        panel.add( attacker );
        panel.add( attackerPriority );
        panel.add( new JLabel() );
        cardPanel.add( panel, SearchOption.attacker.name() );
        panel = new JPanel( new GridLayout( 3, 1) );
        panel.add( defender );
        panel.add( defenderPriority );
        panel.add( new JLabel() );
        cardPanel.add( panel, SearchOption.defender.name() );
        optionPanel.add( cardPanel, BorderLayout.NORTH );
        JPanel buttonPanel = new JPanel( new BorderLayout() );
        buttonPanel.add( getSearchButton(), BorderLayout.NORTH );
        optionPanel.add( buttonPanel, BorderLayout.CENTER );
        controlPanel.add( optionPanel, BorderLayout.CENTER );

        box.addItemListener( new ItemListener()
        {
            @SuppressWarnings("unchecked")
            public void itemStateChanged( ItemEvent e )
            {
                if( e.getStateChange() == ItemEvent.SELECTED )
                {
                    SearchOption searchOption = ((EnumWrapper<SearchOption>)e.getItem()).getBase();
                    cardLayout.show( cardPanel, searchOption.name() );
                    switch ( searchOption )
                    {
                        case attacker:
                            sorter.toggleSortOrder( BattleReportTableModel.ATTACKER );
                            break;
                        case defender:
                            sorter.toggleSortOrder( BattleReportTableModel.DEFENDER );
                            break;
                        case systems:
                            sorter.toggleSortOrder( BattleReportTableModel.PLANET );
                            break;
                        case time:
                            sorter.toggleSortOrder( BattleReportTableModel.TIME );
                            break;
                        default:
                            break;
                    }
                }
            }

        });

        buttonPanel = new JPanel( new GridLayout( 4, 1, 0, 0 ) );
        buttonPanel.add( exportSelected );
        buttonPanel.add( exportAll );
        buttonPanel.add( deleteSelected );
        buttonPanel.add( deleteAll );

        controlPanel.add( buttonPanel, BorderLayout.SOUTH );
        centerPanel.add( controlPanel, BorderLayout.WEST );

        final JTable rowHeader = createRowHeaderTable();

        model = new BattleReportTableModel( getDataManager() );

        final TableCellRenderer priorityRenderer = new PriorityTableCellRenderer();
        final TableCellRenderer notesRenderer = new NotesTableCellRenderer();
        JComboBox<Object> editor = new JComboBox<>();
        editor.addItem( "---" );
        for( Priority priority : Priority.values() )
        {
            editor.addItem( priority );
        }
        editor.setRenderer( new PriorityListCellRenderer() );
        final TableCellEditor priorityEditor = new DefaultCellEditor( editor );
        table = new TooltipTable( model )
        {
            private static final long serialVersionUID = -8499738896262825180L;

            @Override
            public TableCellEditor getCellEditor( int row, int column )
            {
                TableCellEditor rv = super.getCellEditor();
                // convert to model
                int modelColumn = convertColumnIndexToModel( column );

                switch( modelColumn )
                {
                    case BattleReportTableModel.PRIORITY:
                        rv = priorityEditor;
                        break;
                }
                return rv;
            }

            @Override
            public TableCellRenderer getCellRenderer( int row, int column )
            {
                TableCellRenderer rv = super.getCellRenderer( row, column );
                // convert to model
                int modelColumn = convertColumnIndexToModel( column );

                switch( modelColumn )
                {
                    case BattleReportTableModel.PRIORITY:
                        rv = priorityRenderer;
                        break;
                    case BattleReportTableModel.NOTES:
                        rv = notesRenderer;
                        break;
                }
                return rv;
            }

        };
        linkToRowHeader(table);

        JScrollPane scrollPane = new JScrollPane( table,
                        JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                        JScrollPane.HORIZONTAL_SCROLLBAR_NEVER );
        table.setDefaultRenderer( String.class, new AlternateLineTableCellRenderer() );
        table.setDefaultRenderer( Number.class, new PointsRenderer() );
        table.setDefaultRenderer( Icon.class, new IconTableCellRenderer() );
        sorter = new TableRowSorter<TableModel>();
        sorter.setModel( model );
        sorter.setComparator( BattleReportTableModel.PRIORITY, priorityComparator );
        sorter.setMaxSortKeys( 3 );
        table.setRowSorter( sorter );
        scrollPane.setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );
        scrollPane.setRowHeaderView( rowHeader );
        scrollPane.setCorner( ScrollPaneConstants.UPPER_LEFT_CORNER, rowHeader.getTableHeader() );
        centerPanel.add( scrollPane, BorderLayout.CENTER );

        table.getSelectionModel().addListSelectionListener( new ListSelectionListener()
        {
            public void valueChanged( ListSelectionEvent e )
            {
                if( !e.getValueIsAdjusting() )
                {
                    boolean enable = ( e.getFirstIndex() >= 0 );
                    deleteSelected.setEnabled( enable );
                    exportSelected.setEnabled( enable );
                }
            }
        });
        deleteAll.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                MainFrame mainFrame = getMainFrame();
                mainFrame.dbOn();
                try
                {
                    model.removeAll();
                }
                catch( SQLException ex )
                {
                    mainFrame.showErrorInfo( ex );
                }
                mainFrame.dbOff();
                mainFrame.updatePane();
            }

        });
        deleteSelected.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                MainFrame mainFrame = getMainFrame();
                int[] rows = table.getSelectedRows();
                int[] modelRows = new int[rows.length];
                for( int i = 0; i < rows.length; i++ )
                {
                    modelRows[i] = table.convertRowIndexToModel( rows[i] );
                }
                mainFrame.dbOn();
                try
                {
                    model.removeRows( modelRows );
                }
                catch( SQLException ex )
                {
                    mainFrame.showErrorInfo( ex );
                }
                mainFrame.dbOff();
                mainFrame.updatePane();
            }

        });
        exportSelected.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                int[] rows = table.getSelectedRows();
                int[] modelRows = new int[rows.length];
                for( int i = 0; i < rows.length; i++ )
                {
                    modelRows[i] = table.convertRowIndexToModel( rows[i] );
                }
                exportList.clear();
                exportList.addAll( model.getExportRows( modelRows ) );
                getDataManager().saveData( BattleReportPane.this );
            }

        });
        exportAll.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                exportList.clear();
                exportList.addAll( model.getExportRows() );
                getDataManager().saveData( BattleReportPane.this );
            }

        });
        popupGoto.addActionListener( new ActionListener()
        {

            public void actionPerformed( ActionEvent e )
            {
                actionGoto();
            }
        } );
        popupShow.addActionListener( new ActionListener()
        {

            public void actionPerformed( ActionEvent e )
            {
                actionShow();
            }
        } );
        table.addMouseListener( new MouseAdapter()
        {
            @Override
            public void mousePressed( MouseEvent event )
            {
                if( event.isPopupTrigger() )
                {
                    JPopupMenu popup = new JPopupMenu();
                    popup.add( popupShow );
                    popup.add( popupGoto );
                    popup.pack();

                    menuPosition = new Point( event.getX(), event.getY() );
                    popup.show(table, event.getX(), event.getY() );
                }
            }

            @Override
            public void mouseReleased( MouseEvent event )
            {
                if( event.isPopupTrigger() )
                {
                    JPopupMenu popup = new JPopupMenu();
                    popup.add( popupShow );
                    popup.add( popupGoto );
                    popup.pack();

                    menuPosition = new Point( event.getX(), event.getY() );
                    popup.show(table, event.getX(), event.getY() );
                }
            }

            @Override
            public void mouseClicked( MouseEvent event )
            {
                if( event.getClickCount() >= 2 )
                {
                    Point point = new Point( event.getX(), event.getY() );
                    int modelColumn = table.convertColumnIndexToModel( table.columnAtPoint( point ) );
                    int modelRow = table.convertRowIndexToModel( table.rowAtPoint( point ) );
                    MainFrame mainFrame = getMainFrame();
                    switch( modelColumn )
                    {
                        case BattleReportTableModel.NOTES:
                            BattleReport report = model.getRow( modelRow );
                            NoteBox<BattleReport> box = new NoteBox<BattleReport>( mainFrame, report );
                            mainFrame.showModalBox( box );
                            model.fireTableCellUpdated( modelRow, modelColumn );
                            break;
                        case BattleReportTableModel.PRIORITY:
                            break;
                        default:
                            menuPosition = new Point( event.getX(), event.getY() );
                            actionShow();
                            break;
                    }
                }
            }
        } );

        attackerPriority.setRenderer( new PriorityListCellRenderer() );
        defenderPriority.setRenderer( new PriorityListCellRenderer() );
        timePriority.setRenderer( new PriorityListCellRenderer() );
        positionPriority.setRenderer( new PriorityListCellRenderer() );

        return centerPanel;
    }

    protected void actionGoto()
    {
        int viewRow = table.rowAtPoint( menuPosition );
        int modelRow = table.convertRowIndexToModel( viewRow );
        if( modelRow >= 0  )
        {
            BattleReport report = model.getRow( modelRow );
            Position position = Position.createFromId( report.getPosition() );
            getMainFrame().showGalaxyView( position.getGalaxy(), position.getSystem(), true );
        }

    }

    protected void actionShow()
    {
        int viewRow = table.rowAtPoint( menuPosition );
        int modelRow = table.convertRowIndexToModel( viewRow );
        if( modelRow >= 0  )
        {
            BattleReport report = model.getRow( modelRow );
            BattleReportView battleReportViewer = new BattleReportView( report, getMainFrame() );
            battleReportViewer.setVisible( true );
        }

    }

    @SuppressWarnings("unchecked")
    protected void actionSearch()
    {
        MainFrame mainFrame = getMainFrame();
        DataManager dataManager = getDataManager();
        SearchOption searchOption = ((EnumWrapper<SearchOption>)box.getSelectedItem()).getBase();
        Priority priority = null;
        Collection<BattleReport> result = null;
        mainFrame.dbOn();
        try
        {
            switch( searchOption )
            {
                case attacker:
                    if( attackerPriority.getSelectedItem() instanceof Priority )
                    {
                        priority = ( Priority )attackerPriority.getSelectedItem();
                    }
                    result = dataManager.findBattleReportsByAttackerAndPriority( attacker.getText(), priority );
                    break;
                case defender:
                    if( defenderPriority.getSelectedItem() instanceof Priority )
                    {
                        priority = ( Priority )defenderPriority.getSelectedItem();
                    }
                    result = dataManager.findBattleReportsByDefenderAndPriority( defender.getText(), priority );
                    break;
                case time:
                    if( timePriority.getSelectedItem() instanceof Priority )
                    {
                        priority = ( Priority )timePriority.getSelectedItem();
                    }
                    result = dataManager.findBattleReportsByTimeAndPriority( startTime.getDate().getTime(), stopTime.getDate().getTime(), priority );
                    break;
                case systems:
                    if( positionPriority.getSelectedItem() instanceof Priority )
                    {
                        priority = ( Priority )positionPriority.getSelectedItem();
                    }
                    result = dataManager.findBattleReportsBySystemsAndPriority( startPosition.getPosition( 1 ), stopPosition.getPosition( 1 ), priority );
                    break;
            }
        }
        catch( Exception ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        mainFrame.dbOff();
        if( result != null )
        {
            model.setData( result );
        }
    }

    protected void updateTable()
    {
        // clear table on account change only
        if( accountChanged() )
        {
            model.update();
            sorter.setComparator( BattleReportTableModel.PRIORITY, priorityComparator );
            sorter.setMaxSortKeys( 3 );
        }
        table.setRowHeight( 16 );

        int width = 20;
        adjustColumnWidth( BattleReportTableModel.PRIORITY, width );
        adjustColumnWidth( BattleReportTableModel.NOTES, width );
        adjustColumnWidth( BattleReportTableModel.ASTRO, width );

        updateColumnNames();
    }

    @Override
    protected JTable getTable()
    {
        return table;
    }

    protected void updateLabel()
    {
        DataManager dataManager = getDataManager();
        if( accountChanged() )
        {
            updateSearchOptions();
        }
        searchLabel.setText( dataManager.getI18nText( "Search.search" ) );
        deleteSelected.setText( dataManager.getI18nText( "delete" ) );
        deleteAll.setText( dataManager.getI18nText( "deleteAll" ) );
        exportSelected.setText( dataManager.getI18nText( "export" ) );
        exportAll.setText( dataManager.getI18nText( "exportAll" ) );
        popupGoto.setText( dataManager.getI18nText( "Popup.goto" ) );
        popupShow.setText( dataManager.getI18nText( "Popup.show" ) );

        deleteSelected.setEnabled( false );
        deleteAll.setEnabled( true );
        exportSelected.setEnabled( false );
        exportAll.setEnabled( true );
    }

    @SuppressWarnings("unchecked")
    private void updateSearchOptions()
    {
        DataManager dataManager = getDataManager();
        Object selectedItem = box.getSelectedItem();
        SearchOption option = SearchOption.attacker;
        if( selectedItem != null )
        {
            option = ((EnumWrapper<SearchOption>)selectedItem).getBase();
        }
        box.removeAllItems();
        SearchOption[] options = SearchOption.forBattleReports();
        for( SearchOption searchOption : options )
        {
            box.addItem( new EnumWrapper<SearchOption>( dataManager, searchOption ) );
        }
        box.setSelectedItem( new EnumWrapper<SearchOption>( dataManager, option ) );
        box.setMaximumRowCount( options.length );
        sorter.toggleSortOrder( BattleReportTableModel.PLANET );

        attackerPriority.removeAllItems();
        defenderPriority.removeAllItems();
        positionPriority.removeAllItems();
        timePriority.removeAllItems();
        Priority[] priorities = Priority.values();
        attackerPriority.addItem( "---" );
        defenderPriority.addItem( "---" );
        positionPriority.addItem( "---" );
        timePriority.addItem( "---" );
        for( Priority priority : priorities )
        {
            attackerPriority.addItem( priority );
            defenderPriority.addItem( priority );
            positionPriority.addItem( priority );
            timePriority.addItem( priority );
        }
        attacker.setText( "*" );
        defender.setText( "*" );
        startPosition.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        startPosition.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        startPosition.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
        stopPosition.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        stopPosition.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        stopPosition.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
    }

    public Document getData( Parser xmlParser )
    {
        Document doc = null;
        if( xmlParser != null && getDataManager().getActiveAccount() != null )
        {
            doc = xmlParser.getDocument();
            getDataManager().createAppRoot( doc );
            getMainFrame().exportBattleReports( doc, exportList );
        }
        return doc;
    }

    public String getExtension()
    {
        return getDataManager().getExtension();
    }

    public String getFileName()
    {
        StringBuilder sb = new StringBuilder();
        sb.append( DataManager.getFileFormat().format( new Date() ) ).append( ' ' );
        sb.append( getDataManager().getI18nText( "BattleReport.battleReport" ) );
        return sb.toString();
    }

    public void setPlanet( Planet planet )
    {
        update();
        Position position = Position.createFromId( planet.getPosition() );
        this.startPosition.setPosition( position );
        this.stopPosition.setPosition( position );
        this.positionPriority.setSelectedIndex( 0 );
        this.box.setSelectedItem( new EnumWrapper<SearchOption>( getDataManager(), SearchOption.attacker ) );
        this.cardLayout.show( cardPanel, SearchOption.attacker.name() );
        actionSearch();
    }
}
