/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/

package siarchive.reports;

import java.awt.Color;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.components.Building;
import siarchive.components.Defense;
import siarchive.components.NoteBox;
import siarchive.components.Position;
import siarchive.components.Research;
import siarchive.components.Resource;
import siarchive.components.Ship;
import siarchive.imports.DateParser;
import siarchive.persistence.SpyReport;

/**
 * @author graf
 *
 */
public class SpyReportView extends ReportView<SpyReport>
{
    private static final long serialVersionUID = 9100127657367061025L;
    private MainFrame mainFrame;

    public SpyReportView( final SpyReport report, final MainFrame mainFrame ) throws HeadlessException
    {
        super( report, mainFrame.getDataManager() );
        this.mainFrame = mainFrame;
        notes.addMouseListener( new MouseAdapter()
        {
            @Override
            public void mouseClicked( MouseEvent event )
            {
                if( event.getClickCount() >= 2 )
                {
                    NoteBox<SpyReport> box = new NoteBox<SpyReport>( SpyReportView.this, mainFrame, report );
                    mainFrame.showModalBox( box );
                    updateIcon();
                    mainFrame.updatePane();
                }
            }
        } );
    }

    @Override
    protected JPanel createCenterPanel( SpyReport spyReport, DataManager dataManager )
    {
        List<JComponent> rows = new ArrayList<JComponent>();
        Map<?, Long> results = SpyReportParser.scanSpyReport( spyReport, dataManager );
        // header
        StringBuilder sb = new StringBuilder();
        sb.append( dataManager.getI18nText( "SpyReport.header" ) );
        sb.append( ' ' );
        sb.append( spyReport.getPlanetName() );
        sb.append( ' ' );
        sb.append( Position.createFromId( report.getPosition() ).toString() );
        sb.append( ' ' );
        sb.append( dataManager.getI18nText( "SpyReport.from" ) );
        sb.append( ' ' );
        sb.append( DateParser.format( report.getCreationTime() ) );

        setTitle( sb.toString() );

        JComponent row = new ReportText( sb.toString() );
        row.setBackground( sectionBg );
        row.setForeground( headerFg );
        ((ReportText)row).setBold();
        rows.add( row );

        // resources
        row = new JPanel( new GridLayout( 1, Resource.values().length, 0, 0) );
        JComponent row2 = new JPanel( new GridLayout( 1, Resource.values().length, 0, 0) );
        for( Resource resource : Resource.values() )
        {
            ReportText label = new ReportText( dataManager.getI18nText( resource.name() ) );
            label.setForeground( fg );
            label.setBackground( defaultBg );
            label.setBold();
            row.add( label );
            ReportText value = new ReportText( DataManager.format( results.get( resource ) ) );
            value.setForeground( fg );
            value.setBackground( defaultBg );
            value.setHorizontalAlignment(  SwingConstants.RIGHT );
            row2.add( value );
        }
        row.setBackground( defaultBg );
        row2.setBackground( defaultBg );
        rows.add( row );
        rows.add( row2 );
        // buildings
        row = new ReportText();
        row.setForeground( sectionFg );
        row.setOpaque( true );
        if( results.containsKey( "SpyReport.buildings" ) )
        {
            row.setBackground( sectionBg );
            ((ReportText)row).setBold();
            ((ReportText)row).setText( dataManager.getI18nText( "SpyReport.buildings" ) );
            rows.add( row );
            // collect buildings
            List<SpyReportField> buildings = new ArrayList<SpyReportField>();
            for( Building building : Building.values() )
            {
                Long level = results.get( building );
                if( level != null )
                {
                    SpyReportField field = new SpyReportField();
                    field.setLabel( dataManager.getI18nText( building.name() ) );
                    field.setValue( DataManager.format( level.longValue() ) );
                    field.setBackground( defaultBg );
                    field.setForeground( fg );
                    buildings.add( field );
                }
            }
            if( buildings.size() > 0 )
            {
                rows.addAll( addFields( buildings ) );
            }
        }
        else
        {
            row.setBackground( missingBg );
            rows.add( row );
        }

        // defense
        row = new ReportText();
        row.setForeground( sectionFg );
        row.setOpaque( true );
        if( results.containsKey( "SpyReport.defense" ) )
        {
            row.setBackground( sectionBg );
            ((ReportText)row).setBold();
            ((ReportText)row).setText( dataManager.getI18nText( "SpyReport.defense" ) );
            rows.add( row );
            // collect defense
            List<SpyReportField> defenses = new ArrayList<SpyReportField>();
            for( Defense defense : Defense.values() )
            {
                Long level = results.get( defense );
                if( level != null )
                {
                    SpyReportField field = new SpyReportField();
                    field.setLabel( dataManager.getI18nText( defense.name() ) );
                    field.setValue( DataManager.format( level.longValue() ) );
                    field.setBackground( defaultBg );
                    field.setForeground( fg );
                    defenses.add( field );
                }
            }
            if( defenses.size() > 0 )
            {
                rows.addAll( addFields( defenses ) );
            }
        }
        else
        {
            row.setBackground( missingBg );
            rows.add( row );
        }

        // ships
        row = new ReportText();
        row.setForeground( sectionFg );
        row.setOpaque( true );
        if( results.containsKey( "SpyReport.ships" ) )
        {
            row.setBackground( sectionBg );
            ((ReportText)row).setBold();
            ((ReportText)row).setText( dataManager.getI18nText( "SpyReport.ships" ) );
            rows.add( row );
            // collect ships
            List<SpyReportField> ships = new ArrayList<SpyReportField>();
            for( Ship ship : Ship.ships() )
            {
                Long level = results.get( ship );
                if( level != null )
                {
                    SpyReportField field = new SpyReportField();
                    field.setLabel( dataManager.getI18nText( ship.name() ) );
                    field.setValue( DataManager.format( level.longValue() ) );
                    field.setBackground( defaultBg );
                    field.setForeground( fg );
                    ships.add( field );
                }
            }
            if( ships.size() > 0 )
            {
                rows.addAll( addFields( ships ) );
            }
        }
        else
        {
            row.setBackground( missingBg );
            rows.add( row );
        }

        // research
        row = new ReportText();
        row.setForeground( sectionFg );
        row.setOpaque( true );
        if( results.containsKey( "SpyReport.research" ) )
        {
            row.setBackground( sectionBg );
            ((ReportText)row).setBold();
            ((ReportText)row).setText( dataManager.getI18nText( "SpyReport.research" ) );
            rows.add( row );
            // collect research
            List<SpyReportField> technologies = new ArrayList<SpyReportField>();
            for( Research research : Research.values() )
            {
                Long level = results.get( research );
                if( level != null )
                {
                    SpyReportField field = new SpyReportField();
                    field.setLabel( dataManager.getI18nText( research.name() ) );
                    field.setValue( DataManager.format( level.longValue() ) );
                    field.setBackground( defaultBg );
                    field.setForeground( fg );
                    technologies.add( field );
                }
            }
            if( technologies.size() > 0 )
            {
                rows.addAll( addFields( technologies ) );
            }
        }
        else
        {
            row.setBackground( missingBg );
            rows.add( row );
        }

        // header
        sb.setLength( 0 );
        sb.append( dataManager.getI18nText( "SpyReport.counterespionage" ) );
        sb.append( ' ' );
        sb.append( results.get( "SpyReport.counterespionage" ) );
        sb.append( '%' );

        // header
        row = new ReportText( sb.toString() );
        row.setBackground( sectionBg );
        row.setForeground( headerFg );
        ((ReportText)row).setBold();
        rows.add( row );

        JPanel centerPanel = new StarPanel( new GridLayout( rows.size(), 1, 0, 0 ) );
        centerPanel.setBackground( sectionBg );
        for( JComponent c : rows )
        {
            centerPanel.add( c );
        }
        centerPanel.setBackground( Color.black );
        centerPanel.setBorder( BorderFactory.createEmptyBorder( 10, 10, 10, 10 ) );
        return centerPanel;
    }

    private List<JComponent> addFields( List<SpyReportField> fields )
    {
        List<JComponent> rows = new ArrayList<JComponent>();
        List<SpyReportField> addFields = new ArrayList<SpyReportField>();
        addFields.addAll( fields );

        int size = fields.size();
        int rowCount = size / 5 + 1;
        // add empty fields as needed
        for( int i = 0; i < 5 * rowCount - size; i++ )
        {
            SpyReportField field = new SpyReportField();
            field.setBackground( defaultBg );
            field.setForeground( fg );
            addFields.add( field );
        }
        Iterator<SpyReportField> iter = addFields.iterator();
        for( int i = 0; i < rowCount; i++ )
        {
            JPanel row = new JPanel( new GridLayout( 1, 5, 0, 0 ) );
            row.setBackground( defaultBg );
            for( int j = 0; j < 5; j++ )
            {
                if( iter.hasNext() )
                {
                    row.add( iter.next() );
                }
            }
            rows.add( row );
        }
        return rows;
    }

    @Override
    protected void updateDb()
    {
        try
        {
            dataManager.save( report );
            dataManager.commit();
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        mainFrame.updatePane();
    }

}
