/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/
package siarchive.reports;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.jr.ob.JSON;

import siarchive.DataManager;
import siarchive.components.Building;
import siarchive.components.Defense;
import siarchive.components.Research;
import siarchive.components.Resource;
import siarchive.components.Ship;
import siarchive.json.JSONTags;
import siarchive.json.spyreport.LevelBean;
import siarchive.json.spyreport.PlanetBean;
import siarchive.json.spyreport.SpyReportBean;
import siarchive.persistence.SpyReport;

/**
 * @author graf
 *
 */
public class SpyReportParser
{
    public static Map<?, Long> scanSpyReport( SpyReport spyReport, DataManager dataManager )
    {
        final String input = spyReport.getData();
        if(input.indexOf(JSONTags.reportType_SpyReport) >= 0) {
            return scanSpyReportJSON(input, dataManager);
        }
        else {
            return scanSpyReport(input, dataManager);
        }
    }

    private static Map<?, Long> scanSpyReportJSON( String input, DataManager dataManager ) {
        Map<Object, Long> result = new HashMap<Object, Long>();

        try {
            final SpyReportBean reportBean = JSON.std.beanFrom(SpyReportBean.class, input);
            final PlanetBean planet = reportBean.getPlanet();
            final List<Long> resources = planet.getResources();
            result.put(Resource.iron, resources.get(0));
            result.put(Resource.metal, resources.get(1));
            result.put(Resource.kryptonite, resources.get(2));
            result.put(Resource.spice, resources.get(3));
            result.put(Resource.energy, planet.getEnergy());

            if(planet.getBuildings() != null) {
                result.put( "SpyReport.buildings", Long.valueOf( 1 ) );
                for(Map.Entry<String, LevelBean> entry : planet.getBuildings().entrySet()) {
                    result.put(Building.fromId(entry.getKey()), Long.valueOf(entry.getValue().getLevel()));
                }
            }
            if(planet.getDefense() != null) {
                result.put( "SpyReport.defense", Long.valueOf( 1 ) );
                for(Map.Entry<String, LevelBean> entry : planet.getDefense().entrySet()) {
                    result.put(Defense.fromId(entry.getKey()), Long.valueOf(entry.getValue().getLevel()));
                }
            }
            if(planet.getShips() != null) {
                result.put( "SpyReport.ships", Long.valueOf( 1 ) );
                for(Map.Entry<String, LevelBean> entry : planet.getShips().entrySet()) {
                    result.put(Ship.fromId(entry.getKey()), Long.valueOf(entry.getValue().getLevel()));
                }
            }
            if(planet.getResearches() != null) {
                result.put( "SpyReport.research", Long.valueOf( 1 ) );
                for(Map.Entry<String, LevelBean> entry : planet.getResearches().entrySet()) {
                    result.put(Research.fromId(entry.getKey()), Long.valueOf(entry.getValue().getLevel()));
                }
            }
            result.put("SpyReport.counterespionage", Long.valueOf(reportBean.getChance()));
        }
        catch(Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    private static Map<?, Long> scanSpyReport( String input, DataManager dataManager )
    {
        Map<Object, Long> result = new HashMap<Object, Long>();
        Matcher m;

        // get resources
        result.putAll( parseResources( input, dataManager ) );
        // check buildings
        Pattern buildingPattern = Pattern.compile( dataManager.getI18nText( "SpyReport.buildings" ) + "\\s*\\n" );
        m = buildingPattern.matcher( input );
        if( m.find() )
        {
            result.put( "SpyReport.buildings", Long.valueOf( 1 ) );
            for( Building building : Building.values() )
            {
                buildingPattern = Pattern.compile( dataManager.getI18nText( building.name() ) + "\\s+(\\d+)" );
                m = buildingPattern.matcher( input );
                if( m.find() )
                {
                    result.put( building, Long.valueOf( m.group( 1 ) ) );
                }
            }
        }
        Pattern defensePattern = Pattern.compile( dataManager.getI18nText( "SpyReport.defense" ) + "\\s*\\n" );
        m = defensePattern.matcher( input );
        if( m.find() )
        {
            result.put( "SpyReport.defense", Long.valueOf( 1 ) );
            for( Defense defense : Defense.values() )
            {
                defensePattern = Pattern.compile( dataManager.getI18nText( defense.name() ) + "\\s+(\\d+)" );
                m = defensePattern.matcher( input );
                if( m.find() )
                {
                    result.put( defense, Long.valueOf( m.group( 1 ) ) );
                }
            }
        }
        // and ships
        Pattern shipPattern = Pattern.compile( dataManager.getI18nText( "SpyReport.ships" ) + "\\s*\\n" );
        m = shipPattern.matcher( input );
        if( m.find() )
        {
            result.put( "SpyReport.ships", Long.valueOf( 1 ) );
            for( Ship ship : Ship.ships() )
            {
                shipPattern = Pattern.compile( dataManager.getI18nText( ship.name() ) + "\\s+(\\d+)" );
                m = shipPattern.matcher( input );
                if( m.find() )
                {
                    result.put( ship, Long.valueOf( m.group( 1 ) ) );
                }
            }
        }
        // finally technology
        Pattern techPattern = Pattern.compile( dataManager.getI18nText( "SpyReport.research" ) + "\\s*\\n" );
        m = techPattern.matcher( input );
        if( m.find() )
        {
            result.put( "SpyReport.research", Long.valueOf( 1 ) );
            for( Research research : Research.values() )
            {
                techPattern = Pattern.compile( dataManager.getI18nText( research.name() ) + "\\s+(\\d+)" );
                m = techPattern.matcher( input );
                if( m.find() )
                {
                    result.put( research, Long.valueOf( m.group( 1 ) ) );
                }
            }
        }
        // counter espionage
        Pattern counterEspionagePattern = Pattern.compile( dataManager.getI18nText( "SpyReport.counterespionage" ) + "\\s+(\\d+)" );
        m = counterEspionagePattern.matcher( input );
        if( m.find() )
        {
            result.put( "SpyReport.counterespionage", Long.valueOf( m.group( 1 ) ) );

        }
        return result;
    }

    private static Map<Resource, Long> parseResources( String data, DataManager dataManager )
    {
        Map<Resource, Long> result = new HashMap<Resource, Long>();

        BufferedReader br = null;
        try
        {
            br = new BufferedReader( new StringReader( data ) );
            StringBuilder sb = new StringBuilder();
            sb.append( "\\s*" );
            for( Resource resource : Resource.values() )
            {
                sb.append( dataManager.getI18nText( resource.name() ).toLowerCase() );
                if( resource != Resource.energy )
                {
                    sb.append( "\\s+" );
                }
            }
            Pattern resourcePattern = Pattern.compile( sb.toString() );
            String line;
            while( ( line = br.readLine() ) != null )
            {
                Matcher m = resourcePattern.matcher( line.toLowerCase() );
                if( m.find() )
                {
                    line = br.readLine();
                    Pattern numPattern = Pattern.compile( "\\s*(\\S+)\\s+(\\S+)\\s+(\\S+)\\s+(\\S+)\\s+(\\S+)" );
                    m = numPattern.matcher( line );
                    if( m.find() )
                    {
                        result.put( Resource.iron, valueOf( m.group( 1 ) ) );
                        result.put( Resource.metal, valueOf( m.group( 2 ) ) );
                        result.put( Resource.kryptonite, valueOf( m.group( 3 ) ) );
                        result.put( Resource.spice, valueOf( m.group( 4 ) ) );
                        result.put( Resource.energy, valueOf( m.group( 5 ) ) );
                        break;
                    }
                }
            }
            br.close();
        }
        catch( IOException ex )
        {}

        return result;
    }

    private static Long valueOf( String value )
    {
        StringBuilder sb = new StringBuilder();
        sb.append( value.toString() );
        for( int i = sb.length() - 1; i >= 0; i-- )
        {
            char ch = sb.charAt( i );
            if( !Character.isDigit( ch ) )
            {
                sb.deleteCharAt( i );
            }
        }
        // if nothing is left, use '0'
        if( sb.length() == 0 )
        {
            sb.append( '0' );
        }

        return Long.valueOf( sb.toString() );
    }
}
