/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/
package siarchive.reports;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.HeadlessException;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;

import siarchive.DataManager;
import siarchive.Resources;
import siarchive.components.Priority;
import siarchive.persistence.Report;
import siarchive.table.PriorityListCellRenderer;

/**
 * @author graf
 *
 */
public abstract class ReportView<T extends Report<T>> extends JFrame
{
    private static final long serialVersionUID = 5418099863685403686L;

    protected static Color defaultBg = new Color( 35, 60, 80);
    protected static Color sectionBg = new Color( 7, 27, 47 );
    protected static Color missingBg = Color.red.darker();
    protected static Color headerFg = new Color( 255, 86, 0 );
    protected static Color sectionFg = new Color( 88, 133, 168 );
    protected static Color columnFg = new Color( 173, 173, 88 );
    protected static Color fg = Color.white;

    protected T report;
    protected JComboBox priority = new JComboBox();
    protected DataManager dataManager;
    protected JLabel notes = new JLabel();

    protected Font bold;
    protected Font font;

    /**
     * @throws HeadlessException
     */
    public ReportView( T report, DataManager dataManager ) throws HeadlessException
    {
        this.report = report;
        this.dataManager = dataManager;
        this.setIconImage( Resources.getIcon( "/symbol.png" ).getImage() );
        enableEvents( AWTEvent.WINDOW_EVENT_MASK );

        JPanel contentPane = ( JPanel )this.getContentPane();
        contentPane.setLayout( new BorderLayout() );
        font = contentPane.getFont();
        bold = contentPane.getFont().deriveFont( Font.BOLD );

        JComponent centerPanel = createCenterPanel( report, dataManager );
        contentPane.add( centerPanel, BorderLayout.CENTER );
        JComponent southPanel = createSouthPanel();
        contentPane.add( southPanel, BorderLayout.SOUTH );
        contentPane.setBorder( null );
        contentPane.requestFocus();
        pack();
        setResizable( false );

        KeyListener l = new KeyAdapter()
        {
            public void keyPressed( KeyEvent event )
            {
                eventKeyPressed( event );
            }
        };
        priority.addKeyListener( l );
        this.addKeyListener( l );
    }

    protected void eventKeyPressed( KeyEvent event )
    {
        if( event.getKeyCode() == KeyEvent.VK_C
         && event.getModifiersEx() == InputEvent.CTRL_DOWN_MASK )
        {
            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            Transferable transferable = new StringSelection( report.getData() );
            clipboard.setContents(transferable, null);
        }
    }

    private JComponent createSouthPanel()
    {
        JPanel southPanel = new StarPanel( 40 );
        southPanel.setLayout( new BorderLayout() );
        southPanel.setBackground( Color.black );
        JPanel priorityPanel = new JPanel();
        priorityPanel.setOpaque( false );
        priorityPanel.setLayout( new FlowLayout( FlowLayout.CENTER, 5, 1 ) );
        priority.setBackground( Color.black );
        priorityPanel.add( priority );
        southPanel.add( priorityPanel, BorderLayout.CENTER );
        priority.addItem( "---" );
        for( Priority priority : Priority.values() )
        {
            this.priority.addItem( priority );
        }
        priority.setSelectedItem( report.getFlag() );
        PriorityListCellRenderer renderer = new PriorityListCellRenderer();
        renderer.setListBackground( Color.black );
        priority.getEditor().getEditorComponent().setBackground( Color.black );
        priority.setFocusable( false );
        priority.setRenderer( renderer );
        priority.addItemListener( new ItemListener() {

            public void itemStateChanged( ItemEvent e )
            {
                if( e.getItem() instanceof Priority )
                {
                    report.setFlag( ( Priority )e.getItem() );
                }
                else
                {
                    report.setFlag( null );
                }
                updateDb();
            }

        });


        notes.setOpaque( false );
        updateIcon();
        southPanel.add( notes, BorderLayout.EAST );
        return southPanel;
    }

    protected void updateIcon()
    {
        ImageIcon icon;
        if( report.getNotes() != null && report.getNotes().getText().length() > 0 )
        {
            icon = Resources.getIcon( "/notes.png" ) ;        }
        else
        {
            icon = Resources.getIcon( "/notes-off.png" );
        }
        notes.setPreferredSize( new Dimension( icon.getIconWidth(), icon.getIconHeight() ) );
        notes.setIcon( icon );
    }

    protected abstract JComponent createCenterPanel( T report, DataManager dataManager );
    protected abstract void updateDb();

}