package siarchive.reports;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.SwingConstants;

import com.fasterxml.jackson.jr.ob.JSON;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.components.Combatant;
import siarchive.components.Defense;
import siarchive.components.Ship;
import siarchive.components.NoteBox;
import siarchive.components.Position;
import siarchive.imports.DateParser;
import siarchive.json.PlanetType;
import siarchive.json.battlereport.AstroBean;
import siarchive.json.battlereport.BattleReportBean;
import siarchive.json.battlereport.CombatantBean;
import siarchive.json.battlereport.DefenderDataBean;
import siarchive.json.battlereport.DestroyedBean;
import siarchive.json.battlereport.FighterBean;
import siarchive.json.battlereport.FirstStrikeBean;
import siarchive.json.battlereport.PlanetBean;
import siarchive.json.battlereport.ResearchesBean;
import siarchive.json.battlereport.RoundBean;
import siarchive.json.battlereport.RoundDataBean;
import siarchive.json.battlereport.StolenBean;
import siarchive.persistence.BattleReport;

public class BattleReportView extends ReportView<BattleReport>
{
    private static final long serialVersionUID = -2463243813223802363L;
    private static final char multiColumnLineMarker = '\t';

    private MainFrame mainFrame;

    /**
     * @param report
     * @param dataManager
     * @throws HeadlessException
     */
    public BattleReportView( final BattleReport report, final MainFrame mainFrame )
                    throws HeadlessException
    {
        super( report, mainFrame.getDataManager() );
        this.mainFrame = mainFrame;
        notes.addMouseListener( new MouseAdapter()
        {
            @Override
            public void mouseClicked( MouseEvent event )
            {
                if( event.getClickCount() >= 2 )
                {
                    NoteBox<BattleReport> box = new NoteBox<BattleReport>( BattleReportView.this, mainFrame, report );
                    mainFrame.showModalBox( box );
                    updateIcon();
                    mainFrame.updatePane();
                }
            }
        } );
    }

    @Override
    protected JComponent createCenterPanel( BattleReport report, DataManager dataManager )
    {
        // header
        StringBuilder sb = new StringBuilder();
        sb.append( dataManager.getI18nText( "BattleReport.battleReport" ) );
        sb.append( ' ' );
        sb.append( Position.createFromId( report.getPosition() ).toString() );
        sb.append( ' ' );
        sb.append( DateParser.format( report.getCreationTime() ) );
        setTitle( sb.toString() );

        JPanel centerPanel;
        if(report.isJSONData()) {
            centerPanel = getBattlePanelJSON();
        }
        else {
            centerPanel = getBattlePanel();
        }

        JScrollPane sp = new JScrollPane( centerPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED )
        {
            private static final long serialVersionUID = 3629371795198306499L;

            @Override
            public Dimension getPreferredSize()
            {
                Dimension size = super.getPreferredSize();
                if( size.height > 600 )
                {
                    size.height = 600;
                }
                size.width += 25;
                if( size.width < 800 )
                {
                    size.width = 800;
                }
                return size;
            }
        };
        sp.setBackground( Color.black );
        return sp;
    }

    private JComponent getSingleLineComponent( String line)
    {
        JLabel row = new ReportLabel();
        row.setHorizontalAlignment( SwingConstants.LEFT );
        row.setText( line );
        Color foreGround;
        if( line.contains( dataManager.getI18nText( "BattleReport.round" ) ) )
        {
            foreGround = headerFg;
            row.setFont( bold );
        }
        else if( line.contains( dataManager.getI18nText( "BattleReport.destroyed" ) ) )
        {
            foreGround = Color.red;
            row.setFont( bold );
        }
        else
        {
            foreGround = fg;
        }
        row.setForeground( foreGround );
        row.setAlignmentX( Component.CENTER_ALIGNMENT );
        return row;
    }


    private JComponent getMultiColumnComponent( String... tableRows )
    {
        JPanel table = new JPanel( new FlowLayout( FlowLayout.CENTER ) );
        table.setOpaque( false );
        String[] parts = tableRows[0].split( Character.toString( multiColumnLineMarker ) );
        JPanel rows = new JPanel( new GridLayout( tableRows.length, parts.length, 0, 0 ) );
        rows.setOpaque( false );

        for( int j = 0; j < tableRows.length; j++)
        {
            parts = tableRows[j].split( Character.toString( multiColumnLineMarker ) );
            for( int i = 0; i < parts.length; i++ )
            {
                JLabel label = new ReportText();
                label.setText( parts[i] );
                if( j == 0 )
                {
                    label.setForeground( sectionFg );
                    label.setBackground( sectionBg );
                    label.setFont( bold );
                }
                else if( i == 0 )
                {
                    label.setForeground( columnFg );
                    label.setBackground( sectionBg );
                    label.setFont( bold );
                }
                else
                {
                    label.setForeground( fg );
                    label.setBackground( defaultBg );
                }
                rows.add( label );
            }
        }
        table.add( rows );
        table.setAlignmentX( Component.CENTER_ALIGNMENT );
        return table;
    }

    private JPanel getBattlePanel()
    {
        JPanel centerPanel = new StarPanel();
        font = centerPanel.getFont();
        bold = centerPanel.getFont().deriveFont( Font.BOLD );
        BoxLayout box = new BoxLayout( centerPanel, BoxLayout.Y_AXIS );
        centerPanel.setLayout( box );
        centerPanel.setBackground( Color.black );

        String data = report.getData();
        int pos = data.indexOf( dataManager.getI18nText( "BattleReport.beginMarker" ) );
        if( pos >= 0 )
        {
            data = data.substring( pos );
        }
        BufferedReader br = new BufferedReader( new StringReader( data ) );
        String line;
        try
        {
            while( ( line = br.readLine() ) != null )
            {
                if( line.length() > 0 )
                {
                    if( line.indexOf( multiColumnLineMarker ) >= 0 )
                    {
                        List<String> tableRows = new ArrayList<String>();
                        while( line != null && line.indexOf( multiColumnLineMarker ) >= 0 )
                        {
                            tableRows.add( line.trim() );
                            line = br.readLine();
                        }
                        JComponent row = getMultiColumnComponent( tableRows.toArray(new String[tableRows.size()]) );
                        centerPanel.add( row );
                    }
                    if( line != null )
                    {
                        JComponent row = getSingleLineComponent(line);
                        centerPanel.add( row );
                    }
                }
            }
        }
        catch( IOException ex )
        {}
        finally
        {
            try
            {
                br.close();
            }
            catch( IOException e )
            {}
        }
        return centerPanel;
    }

    private JPanel getBattlePanelJSON() {
        JPanel centerPanel = new StarPanel();
        BoxLayout box = new BoxLayout( centerPanel, BoxLayout.Y_AXIS );
        centerPanel.setLayout( box );
        centerPanel.setBackground( Color.black );

        String data = report.getData();
        try {
            BattleReportBean reportBean = JSON.std.beanFrom(BattleReportBean.class, data);
            StringBuilder line;

            // header
            line = new StringBuilder();
            line.append(dataManager.getI18nText( "BattleReport.battleReport" ))
                .append(": ")
                .append(reportBean.getInsertDate());
            centerPanel.add(getSingleLineComponent(line.toString()));
            centerPanel.add(getSingleLineComponent("\n"));

            final PlanetBean startPlanet = reportBean.getStartPlanet();
            line.setLength(0);
            line.append(dataManager.getI18nText( "BattleReport.attacker" ))
                .append(" ")
                .append(report.getAttacker())
                .append(" [")
                .append(startPlanet.getCoords())
                .append("] ")
                .append((startPlanet.getType() == PlanetType.MOON) ? "Asteroid" : startPlanet.getName());
            centerPanel.add(getSingleLineComponent(line.toString()));

            line.setLength(0);
            final FighterBean attacker = reportBean.getAttacker();
            ResearchesBean[] researches = new ResearchesBean[2];
            researches[0] = attacker.getResearches();
            line.append(dataManager.getI18nText( "BattleReport.damage"))
                .append(": ")
                .append(researches[0].getDamage())
                .append("0% ")
                .append(" ")
                .append(dataManager.getI18nText( "BattleReport.shield"))
                .append(": ")
                .append(researches[0].getShield())
                .append("0% ")
                .append(" ")
                .append(dataManager.getI18nText( "BattleReport.structure"))
                .append(": ")
                .append(researches[0].getStructure())
                .append("0% ");
            centerPanel.add(getSingleLineComponent(line.toString()));
            centerPanel.add(getSingleLineComponent("\n"));

            final FighterBean defender = reportBean.getDefender();
            final PlanetBean destinationPlanet = reportBean.getDestinationPlanet();
            line.setLength(0);
            line.append(dataManager.getI18nText( "BattleReport.defender" ))
            .append(" ")
            .append(report.getDefender())
            .append(" [")
            .append(destinationPlanet.getCoords())
            .append("] ")
            .append((destinationPlanet.getType() == PlanetType.MOON) ? "Asteroid" : destinationPlanet.getName());
            centerPanel.add(getSingleLineComponent(line.toString()));
            line.setLength(0);
            researches[1] = defender.getResearches();
            line.append(dataManager.getI18nText( "BattleReport.damage"))
                .append(": ")
                .append(researches[1].getDamage())
                .append("0% ")
                .append(" ")
                .append(dataManager.getI18nText( "BattleReport.shield"))
                .append(": ")
                .append(researches[1].getShield())
                .append("0% ")
                .append(" ")
                .append(dataManager.getI18nText( "BattleReport.structure"))
                .append(": ")
                .append(researches[1].getStructure())
                .append("0% ");
            centerPanel.add(getSingleLineComponent(line.toString()));
            centerPanel.add(getSingleLineComponent("\n"));

            // first strike
            final FirstStrikeBean firstStrike = reportBean.getFirstStrike();
            if(firstStrike != null) {
                firstStrike(centerPanel, report, firstStrike);
            }
            // rounds
            for(Map.Entry<String, RoundBean> round : reportBean.getRounds().entrySet()) {
                round(centerPanel, report, round.getValue(), Integer.parseInt(round.getKey()), researches);
            }
            // footer
            centerPanel.add(new JSeparator());
            centerPanel.add(getSingleLineComponent("\n"));
            switch(reportBean.getWinner()) {
                case "1":
                    centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.winMessage",
                                    dataManager.getI18nText( "BattleReport._attacker"))));
                    final StolenBean stolen = reportBean.getStolen();
                    final List<Long> resources = stolen.getResources();
                    centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.plunder",
                                    resources.get(0), resources.get(1), resources.get(2), resources.get(3))));
                    break;
                case "2":
                    centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.winMessage",
                                    dataManager.getI18nText( "BattleReport._defender"))));
                    break;
                case "3":
                    centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.drawMessage")));
                    break;
            }
            centerPanel.add(getSingleLineComponent("\n"));
            final DestroyedBean destroyed = reportBean.getDestroyed();
            final long a_units = destroyed.getAttacker().getUnits();
            final long d_units = destroyed.getDefender().getUnits();
            centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.loss",
                            dataManager.getI18nText( "BattleReport._attacker"), a_units)));
            centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.loss",
                            dataManager.getI18nText( "BattleReport._defender"), d_units)));
            centerPanel.add(getSingleLineComponent("\n"));
            final List<Long> rubble = reportBean.getRubblefield();
            centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.rubble",
                            rubble.get(1), rubble.get(2))));
            final AstroBean astro = reportBean.getAstro();
            if(astro.isCreated()) {
                centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.asteroid")));
            }
            else {
                centerPanel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.astrochance", astro.getChance())));
            }
            centerPanel.add(getSingleLineComponent("\n"));
        }
        catch(Exception e) {
            e.printStackTrace();
        }
        return centerPanel;
    }

    private void firstStrike(JPanel panel, BattleReport report, FirstStrikeBean firstStrike) {
        StringBuilder line = new StringBuilder();

        line.append(dataManager.getI18nText( "BattleReport.round"))
        .append(" - ")
        .append(dataManager.getI18nText( "BattleReport.firstStrike"))
        .append(" -");
        panel.add(getSingleLineComponent(line.toString()));

        final StringBuilder type = new StringBuilder();
        final StringBuilder numberBefore = new StringBuilder();
        final StringBuilder numberAfter = new StringBuilder();
        type.append(dataManager.getI18nText( "BattleReport.type"))
            .append(multiColumnLineMarker)
            .append(dataManager.getI18nText( "BattleReport.bomber"));
        numberBefore.append(dataManager.getI18nText( "BattleReport.number"))
                    .append(multiColumnLineMarker)
                    .append(firstStrike.getBomber());
        panel.add(getMultiColumnComponent(type.toString(), numberBefore.toString()));
        panel.add(getSingleLineComponent("\n"));

        type.setLength(0);
        numberBefore.setLength(0);
        type.append(dataManager.getI18nText( "BattleReport.type"));
        numberBefore.append(dataManager.getI18nText( "BattleReport.numberBefore"));
        numberAfter.append(dataManager.getI18nText( "BattleReport.numberAfter"));
        final Map<String, DefenderDataBean> data = firstStrike.getDefenderdata();
        for(DefenderDataBean entry: data.values()) {
            type.append(multiColumnLineMarker);
            numberBefore.append(multiColumnLineMarker);
            numberAfter.append(multiColumnLineMarker);
            type.append(entry.getName());
            numberBefore.append(entry.getAmountBefore());
            numberAfter.append(entry.getAmountAfter());
        }
        panel.add(getMultiColumnComponent(type.toString(), numberBefore.toString(), numberAfter.toString()));

        panel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.bomberResult", firstStrike.getDestroyTotal())));
        panel.add(getSingleLineComponent("\n"));
    }

    private void round(JPanel panel, BattleReport report, RoundBean round, int index, ResearchesBean[] researches) {
        StringBuilder line = new StringBuilder();

        line.append(dataManager.getI18nText( "BattleReport.round"))
            .append(" - ")
            .append(Integer.toString(index))
            .append(" -");
        panel.add(getSingleLineComponent(line.toString()));
        panel.add(getSingleLineComponent("\n"));

        line.setLength(0);
        line.append(dataManager.getI18nText( "BattleReport.attacker"))
        .append(" ")
        .append(report.getAttacker());
        halfRound(panel, line.toString(), round.getAgressor(), researches[0]);
        line.setLength(0);
        line.append(dataManager.getI18nText( "BattleReport.defender"))
        .append(" ")
        .append(report.getDefender());
        halfRound(panel, line.toString(), round.getDefender(), researches[1]);

        roundFooter(panel, round.getData());

        panel.add(getSingleLineComponent("\n"));
    }

    private void halfRound(JPanel panel, String name, Map<String, CombatantBean> data, ResearchesBean researches) {
        panel.add(getSingleLineComponent(name));
        panel.add(getSingleLineComponent("\n"));
        if(data != null) {
            StringBuilder type = new StringBuilder();
            StringBuilder number = new StringBuilder();
            StringBuilder firepower = new StringBuilder();
            StringBuilder shields = new StringBuilder();
            StringBuilder structure = new StringBuilder();
            type.append(dataManager.getI18nText( "BattleReport.type"));
            number.append(dataManager.getI18nText( "BattleReport.number"));
            firepower.append(dataManager.getI18nText( "BattleReport.firepower"));
            shields.append(dataManager.getI18nText( "BattleReport.shields"));
            structure.append(dataManager.getI18nText( "BattleReport.structure"));
            for(Map.Entry<String, CombatantBean> entry: data.entrySet()) {
                Combatant combatant = Defense.fromId(entry.getKey());
                if(combatant == null) {
                    combatant = Ship.fromId(entry.getKey());
                }
                type.append(multiColumnLineMarker);
                number.append(multiColumnLineMarker);
                firepower.append(multiColumnLineMarker);
                shields.append(multiColumnLineMarker);
                structure.append(multiColumnLineMarker);
                type.append(entry.getValue().getName());
                number.append(entry.getValue().getLevel());
                firepower.append((researches.getDamage() + 10) * combatant.getFirePower() / 10);
                shields.append((researches.getShield() + 10) * combatant.getShields() / 10);
                structure.append((researches.getStructure() + 10) * combatant.getStructure() / 10);
            }
            panel.add(getMultiColumnComponent(type.toString(), number.toString(),
                      firepower.toString(), shields.toString(), structure.toString()));
        }
        else {
            panel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.destroyed")));
        }
        panel.add(getSingleLineComponent("\n"));
    }

    private void roundFooter( JPanel panel, RoundDataBean data ) {
        if(data.getAgressor() != null && data.getDefender() != null) {
            long a_shield = data.getAgressor().getShield();
            long a_damage = data.getAgressor().getDamage();
            long d_shield = data.getDefender().getShield();
            long d_damage = data.getDefender().getDamage();
            if(a_damage > 0) {
                panel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.attackMessage", dataManager.getI18nText( "BattleReport.attacking"), a_damage)));
                panel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.shieldMessage", dataManager.getI18nText( "BattleReport.defending"), Math.min(a_damage, d_shield))));
            }
            if(d_damage > 0) {
                panel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.attackMessage", dataManager.getI18nText( "BattleReport.defending"), d_damage)));
                panel.add(getSingleLineComponent(dataManager.getI18nText( "BattleReport.shieldMessage", dataManager.getI18nText( "BattleReport.attacking"), Math.min(d_damage, a_shield))));
            }
            panel.add(getSingleLineComponent("\n"));
        }
    }

    @Override
    protected void updateDb()
    {
        try
        {
            dataManager.save( report );
            dataManager.commit();
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        mainFrame.updatePane();
    }

}
