/****************************************
 *  COPYRIGHT (C) 2012
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import siarchive.MainFrame;
import siarchive.components.Priority;
import siarchive.persistence.Report;


/**
 * @author graf
 *
 */
public abstract class ReportDao<T extends Report<T>> extends BaseDao<T>
{
    public final static String table = "report";

    protected final static String ORDERBYCREATIONTIMEASC = " ORDER BY CREATETIME ASC";
    protected final static String ORDERBYCREATIONTIMEDESC = " ORDER BY CREATETIME DESC";

    public ReportDao()
    {
        super( 500 );
    }

    private final String COUNT = "SELECT COUNT(*) FROM " + getTable() + " WHERE KIND = ?";
    public int count() throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNT );
        count.setString( 1, getKind() );
        int records = 0;
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String COUNTBYACCOUNT = "SELECT COUNT(*) FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ?";
    public int countByAccount(Long account) throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNTBYACCOUNT );
        int records = 0;
        count.setLong( 1, account );
        count.setString( 2, getKind() );
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String FINDBYACCOUNT = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ?" + ORDERBYCREATIONTIMEASC;
    public List<T> findByAccount(Long account) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYACCOUNT, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYACCOUNTANDCREATIONTIME = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND CREATETIME = ?" + ORDERBYCREATIONTIMEASC;
    public List<T> findByCreationTime(Long account, long creationTime) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYACCOUNTANDCREATIONTIME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setLong( 3, creationTime );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYACCOUNTPOSITIONANDCREATIONTIME = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND POSITION = ? AND CREATETIME = ?" + ORDERBYCREATIONTIMEASC;
    public T find(Long account, int position, long creationTime) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYACCOUNTPOSITIONANDCREATIONTIME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setInt( 3, position );
        find.setLong( 4, creationTime );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBETWEENPOSITION = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND POSITION >= ? AND POSITION <= ?" + ORDERBYCREATIONTIMEASC;
    public List<T> findByPosition(Long account, int startPosition, int stopPosition) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBETWEENPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setInt( 3, startPosition );
        find.setInt( 4, stopPosition );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBETWEENPOSITIONANDPRIORITY = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND CREATETIME >= ? AND CREATETIME <= ? AND PRIORITY = ?" + ORDERBYCREATIONTIMEASC;
    public List<T> findByPositionAndPriority(Long account, int startPosition, int stopPosition, Priority priority) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBETWEENPOSITIONANDPRIORITY, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setLong( 3, startPosition );
        find.setLong( 4, stopPosition );
        find.setString( 5, priority.name() );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYTIME = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND CREATETIME >= ? AND CREATETIME <= ?" + ORDERBYCREATIONTIMEASC;
    public List<T> findByTime(Long account, long startTime, long stopTime) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYTIME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setLong( 3, startTime );
        find.setLong( 4, stopTime );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYTIMEANDPRIORITY = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND CREATETIME >= ? AND CREATETIME <= ? AND PRIORITY = ?" + ORDERBYCREATIONTIMEASC;
    public List<T> findByTimeAndPriority(Long account, long startTime, long stopTime, Priority priority) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYTIMEANDPRIORITY, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setLong( 3, startTime );
        find.setLong( 4, stopTime );
        find.setString( 5, priority.name() );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOSITION = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND POSITION = ? AND ISASTEROID = ?" + ORDERBYCREATIONTIMEDESC;
    public List<T> find(Long account, int position, boolean isAsteroid)  throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setInt( 3, position );
        find.setBoolean( 4, isAsteroid );
        ResultSet resultSet = find.executeQuery();
        List<T> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    protected T fillBaseDTO(T report, ResultSet resultSet) throws SQLException
    {
        report.setId( resultSet.getLong( "ID" ) );
        report.setAccount( resultSet.getLong( "ACCOUNT" ) );
        report.setCreationTime( resultSet.getLong( "CREATETIME" ) );
        report.setPosition( resultSet.getInt( "POSITION" ) );
        report.setAsteroid( resultSet.getBoolean( "ISASTEROID" ) );
        report.setFlag( MainFrame.parsePriorityAttribute( resultSet.getString( "PRIORITY" ) ) );
        report.setData( resultSet.getString( "REPORTDATA" ) );
        report.setNotes(getNote( resultSet.getString( "NOTES" ) ) );
        report.setPlanetName( resultSet.getString( "PLANET" ) );
        return report;
    }

    @Override
    protected String getTable()
    {
        return table;
    }

    @Override
    protected T findUnique(T report) throws SQLException
    {
        return find(report.getAccount(), report.getPosition(), report.getCreationTime());
    }

    protected abstract String getKind();
}
