/****************************************
 *  COPYRIGHT (C) 2012, 2024
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import siarchive.MainFrame;
import siarchive.components.Status;
import siarchive.persistence.Alliance;
import siarchive.persistence.DatabaseException;
import siarchive.persistence.DatabaseFactory;
import siarchive.persistence.Player;

/**
 * @author graf
 *
 */
public class PlayerDao extends IdentifierDao<Player>
{
    public final static String table = "player";

    public PlayerDao()
    {
        super( 5000 );
    }

    private final String COUNTBYALLIANCEID = "SELECT COUNT(*) FROM " + table + " WHERE ALLIANCE = ?";
    public int countByAllianceId(long id) throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNTBYALLIANCEID );
        count.setLong( 1, id );
        int records = 0;
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String CLEANUP = "DELETE FROM " + table + " WHERE ( SELECT COUNT(" + PlanetDao.table + ".OWNER) FROM " + PlanetDao.table + " WHERE " + table + ".id = " + PlanetDao.table + ".OWNER ) = 0";
    public int cleanupOrphanedPlayers() throws SQLException
    {
        PreparedStatement cleanup = getConnection().prepareStatement( CLEANUP );
        int records = cleanup.executeUpdate();
        cleanup.close();
        if( records != 0 )
        {
            cleanupCache();
        }
        return records;
    }

    private final String FIND = "SELECT * FROM " + table + " WHERE ACCOUNT = ?";
    public List<Player> find(Long account) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FIND, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAME = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND NAME = ?";
    protected Player find(Long account, String name) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, name );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBYPLAYERID = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND PLAYERID = ?";
    public Player find(Long account, Long playerId) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPLAYERID, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, playerId );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBYALLIANCE = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND ALLIANCE = ?";
    public List<Player> findByAlliance(Long account, Alliance alliance) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYALLIANCE, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        Long id = (alliance != null) ? alliance.getId() : null;
        find.setObject( 2, id );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYALLIANCENAME = "SELECT " + table +".* FROM " + table + ", " + AllianceDao.table + " WHERE " + AllianceDao.table + ".ACCOUNT = ? AND LOWER(" + AllianceDao.table + ".NAME) LIKE ? and " + table + ".ALLIANCE = " + AllianceDao.table + ".ID";
    public List<Player> findByAllianceName(Long account, String allianceNameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYALLIANCENAME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( allianceNameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDIDBYALLIANCENAME = "SELECT " + table +".ID FROM " + table + ", " + AllianceDao.table + " WHERE " + AllianceDao.table + ".ACCOUNT = ? AND LOWER(" + AllianceDao.table + ".NAME) LIKE ? and " + table + ".ALLIANCE = " + AllianceDao.table + ".ID";
    public List<Long> findIdByAllianceName(Long account, String allianceNameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDIDBYALLIANCENAME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( allianceNameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Long> list = new ArrayList<Long>();
        while(resultSet.next())
        {
            list.add( resultSet.getLong( "ID" ) );
        }
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDIDBYNAMEXPR = "SELECT ID FROM " + table + " WHERE ACCOUNT = ? AND LOWER(NAME) LIKE ?";
    public List<Long> findIdByNameExpr(Long account, String nameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDIDBYNAMEXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( nameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Long> list = new ArrayList<Long>();
        while(resultSet.next())
        {
            list.add( resultSet.getLong( "ID" ) );
        }
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAMEXPR = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND LOWER(NAME) LIKE ?";
    public List<Player> findByNameExpr(Long account, String nameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAMEXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( nameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAMEXPRWITHSTATUS = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND LOWER(NAME) LIKE ? AND (STATUS = ? OR SECSTATUS = ?)";
    public List<Player> findByNameExpr(Long account, String nameExpr, Status status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAMEXPRWITHSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( nameExpr ).toLowerCase() );
        find.setString( 3, status.name() );
        find.setString( 4, status.name() );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOINTS = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POINTS >= ? AND POINTS <= ?";
    public List<Player> findByPoints(Long account, long startPoints, long endPoints) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOINTS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, startPoints );
        find.setLong( 3, endPoints );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOINTSWITHSTATUS = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POINTS >= ? AND POINTS <= ? AND (STATUS = ? OR SECSTATUS = ?)";
    public List<Player> findByPoints(Long account, long startPoints, long endPoints, Status status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOINTSWITHSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, startPoints );
        find.setLong( 3, endPoints );
        find.setString( 4, status.name() );
        find.setString( 5, status.name() );
        ResultSet resultSet = find.executeQuery();
        List<Player> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    @Override
    protected String getTable()
    {
        return table;
    }

    @Override
    protected String identifierColumn()
    {
        return "PLAYERID";
    }

	@Override
	protected Player findUnique(Player dbObject) throws SQLException {
        Player player;
        if( dbObject.getPlayerId() >= 0)
        {
            player = find(dbObject.getAccount(), dbObject.getPlayerId());
        }
        else
        {
            // compatibility fallback
            player = find(dbObject.getAccount(), dbObject.getName());
        }
        return player;
	}

    private final String CREATE = "INSERT INTO " + table + " (ACCOUNT, NAME, TITLE, PLAYERID, ALLIANCE, SPECIALIZATION, POINTS, SPICE, UPDATETIME, STATUS, SECSTATUS, NOTES) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
	@Override
	protected PreparedStatement createStatement(Player player) throws SQLException
    {
	    Long id;
        PreparedStatement create = getConnection().prepareStatement( CREATE, Statement.RETURN_GENERATED_KEYS );
        create.setLong( 1, player.getAccount() );
        create.setString( 2, player.getName() );
        create.setString( 3, player.getTitle() );
        id = player.getPlayerId();
        if( id < 0)
        {
            id = minId( player.getAccount() );
            player.setPlayerId( id );
        }
        create.setLong( 4, id );
        if (player.getAlliance() == null)
        {
            throw new DatabaseException( "Invalid Alliance reference for Player '" + player.getName() + "'" );
        }
        create.setObject( 5, player.getAlliance().getId() );
        create.setString( 6, player.getSpecialization() );
        create.setLong( 7, player.getPoints() );
        create.setLong( 8, player.getSpiceValue() );
        create.setLong( 9, player.getUpdateTime() );
        create.setString( 10, getStatus( player.getStatus() ) );
        create.setString( 11, getStatus( player.getSecondaryStatus() ) );
        create.setString( 12, getNote( player.getNotes() ) );
        return create;
    }

    private final String UPDATE = "UPDATE " + table + " SET ACCOUNT = ?, NAME = ?, TITLE = ?, PLAYERID = ?, ALLIANCE = ?, SPECIALIZATION = ?, POINTS = ?, SPICE = ?, UPDATETIME = ?, STATUS = ?, SECSTATUS = ?, NOTES = ? WHERE ID = ?";
	@Override
	protected PreparedStatement updateStatement(Player player) throws SQLException
    {
    	PreparedStatement update;
        update = getConnection().prepareStatement( UPDATE );
        update.setLong( 1, player.getAccount() );
        update.setString( 2, player.getName() );
        update.setString( 3, player.getTitle() );
        update.setLong( 4, player.getPlayerId() );
        if (player.getAlliance() == null)
        {
            throw new DatabaseException( "Invalid Alliance reference for Player '" + player.getName() + "'" );
        }
        update.setObject( 5, player.getAlliance().getId() );
        update.setString( 6, player.getSpecialization() );
        update.setLong( 7, player.getPoints() );
        update.setLong( 8, player.getSpiceValue() );
        update.setLong( 9, player.getUpdateTime() );
        update.setString( 10, getStatus( player.getStatus() ) );
        update.setString( 11, getStatus( player.getSecondaryStatus() ) );
        update.setString( 12, getNote( player.getNotes() ) );
        update.setLong( 13, player.getId() );
        return update;
    }

    @Override
    protected List<Player> createDTO( ResultSet resultSet ) throws SQLException
    {
        List<Player> list = new ArrayList<Player>();
        while(resultSet.next())
        {
            Long id = resultSet.getLong( "ID" );
            Player player = readCache(id);
            if(player == null )
            {
                player = new Player();
                player.setId( resultSet.getLong( "ID" ) );
                player.setAccount( resultSet.getLong( "ACCOUNT" ) );
                player.setName( resultSet.getString( "NAME" ) );
                player.setTitle( resultSet.getString( "TITLE" ) );
                player.setPlayerId(resultSet.getLong( "PLAYERID" ) );
                player.setAlliance( DatabaseFactory.getDao( AllianceDao.class ).get( resultSet.getLong( "ALLIANCE" ) ) );
                player.setSpecialization( resultSet.getString( "SPECIALIZATION" ) );
                player.setPoints( resultSet.getLong( "POINTS" ) );
                player.setSpiceValue( resultSet.getLong( "SPICE" ) );
                player.setUpdateTime( resultSet.getLong( "UPDATETIME" ) );
                player.setStatus( getStatus( resultSet.getString( "STATUS" ) ) );
                player.setSecondaryStatus( getStatus( resultSet.getString( "SECSTATUS" ) ) );
                player.setNotes(getNote( resultSet.getString( "NOTES" ) ) );
            }
            list.add(player);
            updateCache( player );
        }
        return list;
    }

    private String getStatus( Status value )
    {
        String status = null;
        if(value != null)
        {
            status = value.name();
        }
        return status;
    }

    private Status getStatus( String value )
    {
        Status status = null;
        if(value != null)
        {
            status = MainFrame.parseStatusAttribute( value );
        }
        return status;
    }

}
