/****************************************
 *  COPYRIGHT (C) 2012, 2021
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import siarchive.components.Cost;
import siarchive.components.Status;
import siarchive.persistence.DatabaseFactory;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;

/**
 * @author graf
 *
 */
public class PlanetDao extends BaseDao<Planet>
{
    public final static String table = "planet";
    // update times are now located on planets and black holes, the query is redirected to a view comprising both
    public final static String view = "updatetimes";

    private final static String ORDERBYPOSITIONASC = " ORDER BY POSITION ASC";

    public PlanetDao()
    {
        super( 10000 );
    }

    private final String COUNTBYOWNERID = "SELECT COUNT(*) FROM " + table + " WHERE OWNER = ?";
    public int countByOwnerId(long id) throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNTBYOWNERID );
        count.setLong( 1, id );
        int records = 0;
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String COUNTBYSYSTEM = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POSITION >= ? AND POSITION <= ?";
    public int count(Long account, int startPosition, int endPosition) throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNTBYSYSTEM, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        count.setLong( 1, account );
        count.setInt( 2, startPosition );
        count.setInt( 3, endPosition );
        int records = 0;
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String FINDBYALLIANCENAME = "SELECT * FROM " + table + " WHERE " + table + ".OWNER IN "
                    + "( SELECT " + PlayerDao.table + ".ID FROM " + PlayerDao.table + " WHERE " + PlayerDao.table + ".ALLIANCE in "
                    + "( SELECT " + AllianceDao.table + ".ID FROM " + AllianceDao.table + " where " + AllianceDao.table + ".ACCOUNT = ? AND LOWER( " + AllianceDao.table + ".NAME) LIKE ? ) )" + ORDERBYPOSITIONASC;
    public List<Planet> findByAllianceName(Long account, String allianceNameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYALLIANCENAME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( allianceNameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAMEXPR = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND LOWER(NAME) LIKE ?" + ORDERBYPOSITIONASC;
    public List<Planet> findByNameExpr(Long account, String nameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAMEXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( nameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAMEXPRWITHSTATUS = "SELECT * FROM " + table + ", " + PlayerDao.table + " WHERE " + table + ".ACCOUNT = ? AND LOWER(" + table + ".NAME) LIKE ? AND " + table + ".OWNER = " + PlayerDao.table + ".ID AND (" + PlayerDao.table + ".STATUS = ? OR " + PlayerDao.table + ".SECSTATUS = ?)" + ORDERBYPOSITIONASC;
    public List<Planet> findByNameExpr(Long account, String nameExpr, Status status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAMEXPRWITHSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( nameExpr ).toLowerCase() );
        find.setString( 3, status.name() );
        find.setString( 4, status.name() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FIND = "SELECT * FROM " + table + " WHERE ACCOUNT = ?" + ORDERBYPOSITIONASC;
    public List<Planet> find(Long account) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FIND, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAME = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND NAME = ?" + ORDERBYPOSITIONASC;
    public List<Planet> find(Long account, String name) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, name );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYOWNER = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND OWNER = ?" + ORDERBYPOSITIONASC;
    public List<Planet> find(Long account, Player owner) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYOWNER, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, owner.getId() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDPOSITIONSBYOWNERXPR = "SELECT " + table +".POSITION FROM " + table + ", " + PlayerDao.table + " WHERE " + PlayerDao.table + ".ACCOUNT = ? AND LOWER(" + PlayerDao.table + ".NAME) LIKE ? and " + table + ".OWNER = " + PlayerDao.table + ".ID" + ORDERBYPOSITIONASC;
    public List<Integer> findPositionsByOwnerExpr(Long account, String ownerExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDPOSITIONSBYOWNERXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( ownerExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Integer> list = new ArrayList<Integer>();
        while(resultSet.next())
        {
            list.add( resultSet.getInt( "POSITION" ) );
        }
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYOWNERXPR = "SELECT " + table +".* FROM " + table + ", " + PlayerDao.table + " WHERE " + PlayerDao.table + ".ACCOUNT = ? AND LOWER(" + PlayerDao.table + ".NAME) LIKE ? and " + table + ".OWNER = " + PlayerDao.table + ".ID" + ORDERBYPOSITIONASC;
    public List<Planet> findByOwnerExpr(Long account, String ownerExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYOWNERXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( ownerExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYOWNERXPRWITHSTATUS = "SELECT " + table +".* FROM " + table + ", " + PlayerDao.table + " WHERE " + PlayerDao.table + ".ACCOUNT = ? AND LOWER(" + PlayerDao.table + ".NAME) LIKE ? and " + table + ".OWNER = " + PlayerDao.table + ".ID AND (" + PlayerDao.table + ".STATUS = ? OR " + PlayerDao.table + ".SECSTATUS = ?)" + ORDERBYPOSITIONASC;
    public List<Planet> findByOwnerExpr(Long account, String ownerExpr, Status status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYOWNERXPRWITHSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( ownerExpr ).toLowerCase() );
        find.setString( 3, status.name() );
        find.setString( 4, status.name() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOINTS = "SELECT * FROM " + table + " WHERE " + table + ".OWNER IN "
                    + "( SELECT " + PlayerDao.table + ".ID FROM " + PlayerDao.table + " WHERE " + PlayerDao.table + ".ACCOUNT = ? AND " + PlayerDao.table + ".POINTS >= ? AND " + PlayerDao.table + ".POINTS <= ? )" + ORDERBYPOSITIONASC;
    public List<Planet> findByPoints(Long account, long startPoints, long endPoints) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOINTS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, startPoints );
        find.setLong( 3, endPoints );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOINTSWITHSTATUS = "SELECT * FROM " + table + " WHERE " + table + ".OWNER IN "
                    + "( SELECT " + PlayerDao.table + ".ID FROM " + PlayerDao.table + " WHERE " + PlayerDao.table + ".ACCOUNT = ? AND " + PlayerDao.table + ".POINTS >= ? AND " + PlayerDao.table + ".POINTS <= ? AND ( " + PlayerDao.table + ".STATUS = ? OR " + PlayerDao.table + ".SECSTATUS = ? ) )" + ORDERBYPOSITIONASC;
    public List<Planet> findByPoints(Long account, long startPoints, long endPoints, Status status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOINTSWITHSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, startPoints );
        find.setLong( 3, endPoints );
        find.setString( 4, status.name() );
        find.setString( 5, status.name() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOSITION = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POSITION = ?";
    public Planet find(Long account, int position) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, position );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBETWEENPOSITION = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POSITION >= ? AND POSITION <= ?" + ORDERBYPOSITIONASC;
    public List<Planet> findByPosition(Long account, int startPosition, int endPosition) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBETWEENPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, startPosition );
        find.setInt( 3, endPosition );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBETWEENPOSITIONWITHSTATUS = "SELECT * FROM " + table + ", " + PlayerDao.table + " WHERE " + table + ".ACCOUNT = ? AND POSITION >= ? AND POSITION <= ? AND " + table + ".OWNER = " + PlayerDao.table + ".ID AND (" + PlayerDao.table + ".STATUS = ? OR " + PlayerDao.table + ".SECSTATUS = ?)" + ORDERBYPOSITIONASC;
    public List<Planet> findByPosition(Long account, int startPosition, int endPosition, Status status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBETWEENPOSITIONWITHSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, startPosition );
        find.setInt( 3, endPosition );
        find.setString( 4, status.name() );
        find.setString( 5, status.name() );
        ResultSet resultSet = find.executeQuery();
        List<Planet> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDOWNERIDBETWEENPOSITION = "SELECT DISTINCT OWNER FROM " + table + " WHERE ACCOUNT = ? AND POSITION >= ? AND POSITION <= ?";
    public List<Long> findOwnerIdsByPosition(Long account, int startPosition, int endPosition) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDOWNERIDBETWEENPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, startPosition );
        find.setInt( 3, endPosition );
        ResultSet resultSet = find.executeQuery();
        List<Long> list = new ArrayList<Long>();
        while(resultSet.next())
        {
            list.add( resultSet.getLong( "OWNER" ) );
        }
        resultSet.close();
        find.close();
        return list;
    }

    private final String MINTIMEBETWEENPOSITION = "SELECT MIN(UPDATETIME) AS MINTIME FROM " + view + " WHERE ACCOUNT = ? AND POSITION >= ? AND POSITION <= ?";
    public Long minTimeByPosition(Long account, int startPosition, int endPosition) throws SQLException
    {
        Long result = null;
        if(account != null)
        {
            PreparedStatement find = getConnection().prepareStatement( MINTIMEBETWEENPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
            find.setLong( 1, account );
            find.setInt( 2, startPosition );
            find.setInt( 3, endPosition );
            ResultSet resultSet = find.executeQuery();
            if(resultSet.next())
            {
                result = (Long)resultSet.getObject( "MINTIME" );
            }
            resultSet.close();
            find.close();
        }
        return result;
    }

    private final String MINTIMES = "SELECT POSITION, UPDATETIME FROM " + view + " WHERE ACCOUNT = ? " + ORDERBYPOSITIONASC;
    public Map<Integer, Long> minTimeByPosition(Long account) throws SQLException
    {
        Map<Integer, Long> result = new HashMap<Integer, Long>();
        if(account != null)
        {
            PreparedStatement find = getConnection().prepareStatement( MINTIMES, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
            find.setLong( 1, account );
            ResultSet resultSet = find.executeQuery();
            while( resultSet.next() )
            {
                int position = resultSet.getInt( "POSITION" );
                // remove planet from position
                position -= ( position % 100 );
                long time = resultSet.getLong( "UPDATETIME" );
                Integer key = Integer.valueOf( position );
                Long minTime = result.get( key );
                if( minTime == null || minTime.longValue() > time )
                {
                    result.put( key, Long.valueOf( time ) );
                }
            }
            resultSet.close();
            find.close();
        }
        return result;
    }

    @Override
    protected String getTable()
    {
        return table;
    }

	@Override
	protected Planet findUnique(Planet planet) throws SQLException
	{
		return find(planet.getAccount(), planet.getPosition());
	}

    private final String CREATE = "INSERT INTO " + table + " (ACCOUNT, NAME, POSITION, HASASTEROID, EVENT, OWNER, UPDATETIME, NOTES, IRON, METAL, KRYPTONITE, SPICE) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    protected PreparedStatement createStatement( Planet planet ) throws SQLException
    {
        PreparedStatement create = getConnection().prepareStatement( CREATE, Statement.RETURN_GENERATED_KEYS );
        create.setLong( 1, planet.getAccount() );
        create.setString( 2, planet.getName() );
        create.setInt( 3, planet.getPosition() );
        create.setBoolean( 4, planet.hasAsteroid() );
        create.setBoolean( 5, planet.isEvent() );
        create.setObject( 6, (planet.getOwner() != null) ? planet.getOwner().getId() : null, Types.BIGINT );
        create.setLong( 7, planet.getUpdateTime() );
        create.setString( 8, getNote( planet.getNotes() ) );
        Cost debris = planet.getDebris();
        if(debris != null)
        {
            create.setLong( 9, debris.getIron() );
            create.setLong( 10, debris.getMetal() );
            create.setLong( 11, debris.getKryptonite() );
            create.setLong( 12, debris.getSpice() );
        }
        else
        {
            create.setLong( 9, 0 );
            create.setLong( 10, 0 );
            create.setLong( 11, 0 );
            create.setLong( 12, 0 );
        }
        return create;
    }

    private final String UPDATE = "UPDATE " + table + " SET ACCOUNT = ?, NAME = ?, POSITION = ?, HASASTEROID = ?, EVENT = ?, OWNER = ?, UPDATETIME = ?, IRON = ?, METAL = ?, KRYPTONITE = ?, SPICE = ?, NOTES = ? WHERE ID = ?";
    protected PreparedStatement updateStatement( Planet planet ) throws SQLException
    {
    	PreparedStatement update;
        update = getConnection().prepareStatement( UPDATE );
    	update.setLong( 1, planet.getAccount() );
    	update.setString( 2, planet.getName() );
    	update.setInt( 3, planet.getPosition() );
        update.setBoolean( 4, planet.hasAsteroid() );
        update.setBoolean( 5, planet.isEvent() );
        update.setObject( 6, (planet.getOwner() != null) ? planet.getOwner().getId() : null, Types.BIGINT );
        update.setLong( 7, planet.getUpdateTime() );
        Cost debris = planet.getDebris();
        if(debris != null)
        {
            update.setLong( 8, debris.getIron() );
            update.setLong( 9, debris.getMetal() );
            update.setLong( 10, debris.getKryptonite() );
            update.setLong( 11, debris.getSpice() );
        }
        else
        {
            update.setLong( 8, 0 );
            update.setLong( 9, 0 );
            update.setLong( 10, 0 );
            update.setLong( 11, 0 );
        }
        update.setString( 12, getNote( planet.getNotes() ) );
        update.setLong( 13, planet.getId() );
        return update;
    }

    @Override
    protected List<Planet> createDTO( ResultSet resultSet ) throws SQLException
    {
        List<Planet> list = new ArrayList<Planet>();
        while(resultSet.next())
        {
            Long id = resultSet.getLong( "ID" );
            Planet planet = readCache(id);
            if(planet == null )
            {
                planet = new Planet();
                planet.setId( resultSet.getLong( "ID" ) );
                planet.setAccount( resultSet.getLong( "ACCOUNT" ) );
                planet.setName( resultSet.getString( "NAME" ) );
                planet.setPosition( resultSet.getInt( "POSITION" ) );
                planet.setAsteroid( resultSet.getBoolean( "HASASTEROID" ) );
                planet.setEvent( resultSet.getBoolean( "EVENT" ) );
                planet.setOwner( DatabaseFactory.getDao( PlayerDao.class ).get( resultSet.getLong( "OWNER" ) ) );
                planet.setUpdateTime( resultSet.getLong( "UPDATETIME" ) );
                planet.setNotes(getNote( resultSet.getString( "NOTES" ) ) );
                long iron = resultSet.getLong( "IRON" );
                long metal = resultSet.getLong( "METAL" );
                long kryptonite = resultSet.getLong( "KRYPTONITE" );
                long spice = resultSet.getLong( "SPICE" );
                planet.setDebris( new Cost(iron, metal, kryptonite, spice) );
            }
            list.add(planet);
            updateCache( planet );
        }
        return list;
    }

}
