/****************************************
 *  COPYRIGHT (C) 2021
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import siarchive.persistence.BlackHole;

/**
 * @author graf
 *
 */
public class BlackHoleDao extends BaseDao<BlackHole>
{
    public final static String table = "blackhole";

    private final static String ORDERBYPOSITIONASC = " ORDER BY POSITION ASC";

    public BlackHoleDao()
    {
        super( 4000 );
    }

    private final String COUNTBYSYSTEM = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POSITION >= ? AND POSITION <= ?";
    public int count(Long account, int startPosition, int endPosition) throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNTBYSYSTEM, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        count.setLong( 1, account );
        count.setInt( 2, startPosition );
        count.setInt( 3, endPosition );
        int records = 0;
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String FIND = "SELECT * FROM " + table + " WHERE ACCOUNT = ?" + ORDERBYPOSITIONASC;
    public List<BlackHole> find(Long account) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FIND, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        ResultSet resultSet = find.executeQuery();
        List<BlackHole> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYPOSITION = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POSITION = ?";
    public BlackHole find(Long account, int position) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, position );
        ResultSet resultSet = find.executeQuery();
        List<BlackHole> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBETWEENPOSITION = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND POSITION >= ? AND POSITION <= ?" + ORDERBYPOSITIONASC;
    public List<BlackHole> findByPosition(long account, int startPosition, int endPosition) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBETWEENPOSITION, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, startPosition );
        find.setInt( 3, endPosition );
        ResultSet resultSet = find.executeQuery();
        List<BlackHole> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYATTACKSTATUS = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND ATTACKED = ?" + ORDERBYPOSITIONASC;
    public List<BlackHole> findByAttackStatus(long account, boolean status) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYATTACKSTATUS, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setBoolean( 2, status );
        ResultSet resultSet = find.executeQuery();
        List<BlackHole> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYLEVEL = "SELECT * FROM " + table + " WHERE ACCOUNT = ? AND LEVEL = ?" + ORDERBYPOSITIONASC;
    public List<BlackHole> findByLevel(long account, int level) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYLEVEL, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setInt( 2, level );
        ResultSet resultSet = find.executeQuery();
        List<BlackHole> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    @Override
    protected String getTable()
    {
        return table;
    }

	@Override
	protected BlackHole findUnique(BlackHole blackHole) throws SQLException
	{
		return find(blackHole.getAccount(), blackHole.getPosition());
	}

    private final String CREATE = "INSERT INTO " + table + " (ACCOUNT, POSITION, UPDATETIME, SETUP, LEVEL, ATTACKED, UPGRADE, POWER, SHIPS, SPICE, EMBLEMS, STARS) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    protected PreparedStatement createStatement( BlackHole blackHole ) throws SQLException
    {
        PreparedStatement create = getConnection().prepareStatement( CREATE, Statement.RETURN_GENERATED_KEYS );
        create.setLong( 1, blackHole.getAccount() );
        create.setInt( 2, blackHole.getPosition() );
        create.setLong( 3, blackHole.getUpdateTime() );
        create.setBoolean( 4, blackHole.isSetup() );
        create.setInt( 5, blackHole.getLevel() );
        create.setBoolean( 6, blackHole.isAttacked());
        create.setLong( 7, blackHole.getUpgradeTime() );
        create.setLong( 8, blackHole.getPower() );
        create.setString( 9, blackHole.getShipsAsJson() );
        create.setLong( 10, blackHole.getSpice() );
        create.setLong( 11, blackHole.getEmblems() );
        create.setLong( 12, blackHole.getStars() );
        return create;
    }

    private final String UPDATE = "UPDATE " + table + " SET ACCOUNT = ?, POSITION = ?, UPDATETIME = ?, SETUP = ?, LEVEL = ?, ATTACKED = ?, UPGRADE = ?, POWER = ?, SHIPS = ?, SPICE = ?, EMBLEMS = ?, STARS = ? WHERE ID = ?";
    protected PreparedStatement updateStatement( BlackHole blackHole ) throws SQLException
    {
    	PreparedStatement update;
        update = getConnection().prepareStatement( UPDATE );
        update.setLong( 1, blackHole.getAccount() );
        update.setInt( 2, blackHole.getPosition() );
        update.setLong( 3, blackHole.getUpdateTime() );
        update.setBoolean( 4, blackHole.isSetup() );
        update.setInt( 5, blackHole.getLevel() );
        update.setBoolean( 6, blackHole.isAttacked());
        update.setLong( 7, blackHole.getUpgradeTime() );
        update.setLong( 8, blackHole.getPower() );
        update.setString( 9, blackHole.getShipsAsJson() );
        update.setLong( 10, blackHole.getSpice() );
        update.setLong( 11, blackHole.getEmblems() );
        update.setLong( 12, blackHole.getStars() );
        update.setLong( 13, blackHole.getId() );
        return update;
    }

    @Override
    protected List<BlackHole> createDTO( ResultSet resultSet ) throws SQLException
    {
        List<BlackHole> list = new ArrayList<>();
        while(resultSet.next())
        {
            Long id = resultSet.getLong( "ID" );
            BlackHole blackHole = readCache(id);
            if(blackHole == null )
            {
                blackHole = new BlackHole();
                blackHole.setId( resultSet.getLong( "ID" ) );
                blackHole.setAccount( resultSet.getLong( "ACCOUNT" ) );
                blackHole.setPosition( resultSet.getInt( "POSITION" ) );
                blackHole.setUpdateTime( resultSet.getLong( "UPDATETIME" ) );
                blackHole.setSetup( resultSet.getBoolean( "SETUP" ) );
                blackHole.setLevel( resultSet.getInt( "LEVEL" ) );
                blackHole.setAttacked( resultSet.getBoolean( "ATTACKED" ) );
                blackHole.setUpgradeTime( resultSet.getLong( "UPGRADE" ) );
                blackHole.setPower( resultSet.getInt( "POWER" ) );
                blackHole.setShips( resultSet.getString( "SHIPS" ) );
                blackHole.setSpice( resultSet.getLong( "SPICE" ) );
                blackHole.setEmblems( resultSet.getLong( "EMBLEMS" ) );
                blackHole.setStars( resultSet.getLong( "STARS" ) );
            }
            list.add(blackHole);
            updateCache( blackHole );
        }
        return list;
    }

}
