/****************************************
 *  COPYRIGHT (C) 2012
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import siarchive.components.Priority;
import siarchive.persistence.BattleReport;


/**
 * @author graf
 *
 */
public class BattleReportDao extends ReportDao<BattleReport>
{
    private final String FINDBYATTACKER = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND LOWER(ATTACKER) LIKE ?" + ORDERBYCREATIONTIMEASC;
    public List<BattleReport> findByAttacker(Long account, String attackerNameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYATTACKER, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setString( 3, sqlSubstitute( attackerNameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<BattleReport> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYATTACKERANDPRIORITY = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND LOWER(ATTACKER) LIKE ? AND PRIORITY = ?" + ORDERBYCREATIONTIMEASC;
    public List<BattleReport> findByAttackerAndPriority(Long account, String attackerNameExpr, Priority priority) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYATTACKERANDPRIORITY, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setString( 3, sqlSubstitute( attackerNameExpr ).toLowerCase() );
        find.setString( 4, priority.name() );
        ResultSet resultSet = find.executeQuery();
        List<BattleReport> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYDEFENDER = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND LOWER(DEFENDER) LIKE ?" + ORDERBYCREATIONTIMEASC;
    public List<BattleReport> findByDefender(Long account, String defenderNameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYDEFENDER, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setString( 3, sqlSubstitute( defenderNameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<BattleReport> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYDEFENDERANDPRIORITY = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND KIND = ? AND LOWER(DEFENDER) LIKE ? AND PRIORITY = ?" + ORDERBYCREATIONTIMEASC;
    public List<BattleReport> findByDefenderAndPriority(Long account, String defenderNameExpr, Priority priority) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYDEFENDERANDPRIORITY, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, getKind() );
        find.setString( 3, sqlSubstitute( defenderNameExpr ).toLowerCase() );
        find.setString( 4, priority.name() );
        ResultSet resultSet = find.executeQuery();
        List<BattleReport> list = createDTO( resultSet );
        resultSet.close();
        find.close();
        return list;
    }

    @Override
    protected List<BattleReport> createDTO( ResultSet resultSet ) throws SQLException
    {
        List<BattleReport> list = new ArrayList<BattleReport>();
        while(resultSet.next())
        {
            Long id = resultSet.getLong( "ID" );
            BattleReport report = readCache(id);
            if(report == null )
            {
                report = new BattleReport();
                fillBaseDTO( report, resultSet );
                report.setAttacker( resultSet.getString( "ATTACKER" ) );
                report.setDefender( resultSet.getString( "DEFENDER" ) );
            }
            list.add(report);
            updateCache( report );
        }
        return list;
    }

    @Override
    protected String getKind()
    {
        return "battle";
    }

    private final String CREATE = "INSERT INTO " + getTable() + " (KIND, ACCOUNT, CREATETIME, POSITION, ISASTEROID, PRIORITY, REPORTDATA, NOTES, ATTACKER, DEFENDER, PLANET) values ('" + getKind() + "', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    @Override
    protected PreparedStatement createStatement( BattleReport report ) throws SQLException
    {
        PreparedStatement create = getConnection().prepareStatement( CREATE, Statement.RETURN_GENERATED_KEYS );
        int pos = 1;
        create.setLong( pos++, report.getAccount() );
        create.setLong( pos++, report.getCreationTime() );
        create.setInt( pos++, report.getPosition() );
        create.setBoolean( pos++, report.isAsteroid() );
        String flag = (report.getFlag() != null) ? report.getFlag().name() : null;
        create.setString( pos++, flag );
        create.setString( pos++, report.getData() );
        create.setString( pos++, getNote( report.getNotes() ) );
        create.setString( pos++, report.getAttacker() );
        create.setString( pos++, report.getDefender() );
        create.setString( pos++, report.getPlanetName() );
        return create;
    }

    private final String UPDATE = "UPDATE " + getTable() + " SET ACCOUNT = ?, CREATETIME = ?, POSITION = ?, ISASTEROID = ?, PRIORITY = ?, REPORTDATA = ?, NOTES = ?, ATTACKER = ?, DEFENDER = ?, PLANET = ? WHERE ID = ? AND KIND =  '" + getKind() + "'";
    @Override
    protected PreparedStatement updateStatement( BattleReport report ) throws SQLException
    {
        PreparedStatement update;
        int pos = 1;
        update = getConnection().prepareStatement( UPDATE );
        update.setLong( pos++, report.getAccount() );
        update.setLong( pos++, report.getCreationTime() );
        update.setInt( pos++, report.getPosition() );
        update.setBoolean( pos++, report.isAsteroid() );
        String flag = (report.getFlag() != null) ? report.getFlag().name() : null;
        update.setString( pos++, flag );
        update.setString( pos++, report.getData() );
        update.setString( pos++, getNote( report.getNotes() ) );
        update.setString( pos++, report.getAttacker() );
        update.setString( pos++, report.getDefender() );
        update.setString( pos++, report.getPlanetName() );
        update.setLong( pos++, report.getId() );
        return update;
    }

}
