/****************************************
 *  COPYRIGHT (C) 2012
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import siarchive.MainFrame;
import siarchive.persistence.DatabaseException;
import siarchive.persistence.DatabaseFactory;
import siarchive.persistence.DbObject;
import siarchive.persistence.LRUCache;
import siarchive.persistence.Notable;
import siarchive.persistence.Note;

/**
 * @author graf
 *
 */
public abstract class BaseDao<T extends DbObject<T>>
{
    private LRUCache<T> cache;

    protected final static long maxLowId = -1;

    public BaseDao( int cacheSize )
    {
        cache = new LRUCache<T>( cacheSize );
    }

    private final String COUNT = "SELECT COUNT(*) FROM " + getTable();
    public int count() throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNT );
        int records = 0;
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    private final String COUNTBYACCOUNT = "SELECT COUNT(*) FROM " + getTable() + " WHERE ACCOUNT = ?";
    public int countByAccount(Long account) throws SQLException
    {
        PreparedStatement count = getConnection().prepareStatement( COUNTBYACCOUNT );
        int records = 0;
        count.setLong( 1, account );
        ResultSet rs = count.executeQuery();
        if (rs.next())
        {
            records = rs.getInt(1);
        }
        rs.close();
        count.close();
        return records;
    }

    @SuppressWarnings("unchecked")
	protected T create( T dbObject ) throws SQLException
    {
        long key = -1;
        PreparedStatement create = createStatement(dbObject);
        create.executeUpdate();
        ResultSet results = create.getGeneratedKeys();
        if (results.next()) {
            key = results.getInt(1);
            dbObject.setId( key );
        }
        results.close();
        create.close();
        if(dbObject instanceof Notable<?>)
        {
        	// retainNotes is only a marker on filter objects
        	((Notable<T>)dbObject).setRetainNotes(false);
        }
        return dbObject;
    }

    protected abstract PreparedStatement createStatement( T dbObject ) throws SQLException;

    private final String DELETE = "DELETE FROM "+ getTable() + " WHERE ID=?";
    public final void delete(long key) throws SQLException
    {
        PreparedStatement delete = getConnection().prepareStatement( DELETE );
        delete.setLong( 1, key );
        delete.execute();
        delete.close();
        cache.remove( key );
    }

    public final void delete(T dbObject) throws SQLException
    {
    	delete( dbObject.getId() );
    }

    private final String FIND = "SELECT * FROM " + getTable();
    public List<T> find()  throws SQLException
    {
        PreparedStatement get = getConnection().prepareStatement( FIND );
        ResultSet resultSet = get.executeQuery();
        List<T> results = createDTO( resultSet );
        resultSet.close();
        get.close();
        return results;
    }

    private final String FINDID = "SELECT ID FROM " + getTable();
    public List<Long> findId() throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDID, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        ResultSet resultSet = find.executeQuery();
        List<Long> list = new ArrayList<Long>();
        while(resultSet.next())
        {
            list.add( resultSet.getLong( "ID" ) );
        }
        resultSet.close();
        find.close();
        return list;
    }

    protected abstract T findUnique(T dbObject) throws SQLException;

    private final String GET = "SELECT * FROM " + getTable() + " WHERE ID=?";
    public T get(Long key) throws SQLException
    {
        T result = null;
        if( key != null)
        {
            result = cache.get( key );
            if(result == null)
            {
                PreparedStatement get = getConnection().prepareStatement( GET );
                get.setLong( 1, key );
                ResultSet resultSet = get.executeQuery();
                List<T> results = createDTO( resultSet );
                resultSet.close();
                // there can be at most 1
                result = getFirst( results );
                get.close();
            }
            updateCache(result);
        }
        return result;
    }

    protected void cleanupCache()
    {
        cache.clear();
    }

    protected T readCache( Long id )
    {
        return cache.get( id );
    }

    protected void updateCache( T result )
    {
        if(result != null) {
            cache.put( result.getId(), result );
        }
    }

    @SuppressWarnings("unchecked")
	public T save(T dbObject) throws SQLException
    {
        if(dbObject.getId() != null)
        {
        	dbObject = update(dbObject);
        }
        else
        {
        	T stored = findUnique(dbObject);
            if(stored != null)
            {
            	stored.update( dbObject );
            	dbObject.setId(stored.getId());
            	if( dbObject instanceof Notable )
            	{
            		setNote((Notable<T>)stored, (Notable<T>)dbObject);
            	}
                dbObject = update(stored);
            }
            else
            {
                dbObject = create(dbObject);
            }
        }
        updateCache( dbObject );
        return dbObject;
    }



	protected Note getNote(String text)
    {
        return MainFrame.parseNoteAttribute( text );
    }

    protected abstract List<T> createDTO(ResultSet resultSet) throws SQLException;

    protected Connection getConnection() throws DatabaseException
    {
        return DatabaseFactory.getConnection();
    }

    protected abstract String getTable();

    protected T getFirst(List<T> result)
    {
        return (result.size() > 0) ? result.get( 0 ) : null;
    }

    protected String getNote( Note note )
    {
        String result = null;
        if( note != null )
        {
            result = note.getText();
        }
        return result;
    }

    protected String sqlSubstitute(String filterExpr)
    {
        // replace wildcards with SQL wildcards
        String rv = filterExpr.replace( '*', '%' ).replace( '?', '_' );
        // change back escaped wildcards
        rv = rv.replace( "\\%", "*" ).replace( "\\_", "?" );
        return rv;
    }

    protected T update( T dbObject ) throws SQLException
    {
    	PreparedStatement update = updateStatement(dbObject);
        update.executeUpdate();
        update.close();
        return dbObject;
    }

    protected abstract PreparedStatement updateStatement( T dbObject ) throws SQLException;;

    private Note addToNote( Note dbNote, Note addNote )
    {
        Note result = new Note();
        if( dbNote != null )
        {
            result.setText( dbNote.getText() );
        }
        if( addNote != null && addNote.getText().trim().length() > 0 )
        {
            String noteText = addNote.getText().trim();
            String text = (dbNote != null) ? dbNote.getText() : "";
            if( text.trim().length() == 0 )
            {
                text = noteText;
            }
            else
            {
                if( text.length() >= noteText.length() )
                {
                    if( text.indexOf( noteText ) == -1 )
                    {
                        text += "\n" + noteText;
                    }
                }
                else
                {
                    if( noteText.indexOf( text ) == -1 )
                    {
                        text += "\n" + noteText;
                    }
                }
            }
            result.setText( text );
        }
        return (result.getText().length() > 0 ) ? result : null;
    }

    private void setNote(Notable<T> stored, Notable<T> dbObject)
    {
    	if(dbObject.isRetainNotes())
    	{
    	    // add notes together
    	    stored.setNotes( addToNote( stored.getNotes(), dbObject.getNotes() ) );
    	}
	}

}
