/****************************************
 *  COPYRIGHT (C) 2012
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import siarchive.persistence.Alliance;

/**
 * @author graf
 *
 */
public class AllianceDao extends IdentifierDao<Alliance>
{
    public final static String table = "alliance";

    public AllianceDao()
    {
        super( 1000 );
    }

    private final String CLEANUP = "DELETE FROM " + table + " WHERE ( SELECT COUNT(" + PlayerDao.table + ".ALLIANCE) FROM " + PlayerDao.table + " WHERE " + table + ".id = " + PlayerDao.table + ".ALLIANCE ) = 0";
    public int cleanupOrphanedAlliances() throws SQLException
    {
        PreparedStatement cleanup = getConnection().prepareStatement( CLEANUP );
        int records = cleanup.executeUpdate();
        cleanup.close();
        if( records != 0 )
        {
            cleanupCache();
        }
        return records;
    }
    
    private final String FIND = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ?";
    public List<Alliance> find(Long account) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FIND, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        ResultSet resultSet = find.executeQuery();
        List<Alliance> list = createDTO( resultSet ); 
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYNAME = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND NAME = ?";
    protected Alliance find(Long account, String name) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAME, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, name );
        ResultSet resultSet = find.executeQuery();
        List<Alliance> list = createDTO( resultSet ); 
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBYALLIANCEID = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND ALLIANCEID = ?";
    public Alliance find(Long account, Long allianceId) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYALLIANCEID, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setLong( 2, allianceId );
        ResultSet resultSet = find.executeQuery();
        List<Alliance> list = createDTO( resultSet ); 
        resultSet.close();
        find.close();
        return getFirst( list );
    }

    private final String FINDBYNAMEEXPR = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND LOWER(NAME) LIKE ?";
    public List<Alliance> findByNameExpr(Long account, String nameExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYNAMEEXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( nameExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Alliance> list = createDTO( resultSet ); 
        resultSet.close();
        find.close();
        return list;
    }

    private final String FINDBYTAGEXPR = "SELECT * FROM " + getTable() + " WHERE ACCOUNT = ? AND LOWER(TAG) LIKE ?";
    public List<Alliance> findByTagExpr(Long account, String tagExpr) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( FINDBYTAGEXPR, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setLong( 1, account );
        find.setString( 2, sqlSubstitute( tagExpr ).toLowerCase() );
        ResultSet resultSet = find.executeQuery();
        List<Alliance> list = createDTO( resultSet ); 
        resultSet.close();
        find.close();
        return list;
    }

    @Override
    protected String identifierColumn()
    {
        return "ALLIANCEID";
    }

    @Override
	protected Alliance findUnique(Alliance dbObject) throws SQLException 
	{
        Alliance alliance;
        if( dbObject.getAllianceId() >= 0)
        {
            alliance = find(dbObject.getAccount(), dbObject.getAllianceId());
        }
        else
        {
            // compatibility fallback
            alliance = find(dbObject.getAccount(), dbObject.getName());
        }
		return alliance;
	}

	@Override
    protected String getTable()
    {
        return table;
    }

    private final String CREATE = "INSERT INTO " + getTable() + " (ACCOUNT, ALLIANCEID, NAME, TAG, UPDATETIME, NOTES) values (?, ?, ?, ?, ?, ?)";
    protected PreparedStatement createStatement( Alliance alliance ) throws SQLException
    {
        Long id;
        PreparedStatement create = getConnection().prepareStatement( CREATE, Statement.RETURN_GENERATED_KEYS );
        create.setLong( 1, alliance.getAccount() );
        id = alliance.getAllianceId();
        if( id < 0)
        {
            id = minId( alliance.getAccount() );
            alliance.setAllianceId( id );
        }
        create.setLong( 2, id );
        create.setString( 3, alliance.getName() );
        create.setString( 4, alliance.getTag() );
        create.setLong( 5, alliance.getUpdateTime() );
        create.setString( 6, getNote( alliance.getNotes() ) );
        return create;
    }

    private final String UPDATE = "UPDATE " + getTable() + " SET ACCOUNT = ?, ALLIANCEID = ?, NAME = ?, TAG = ?, UPDATETIME = ?, NOTES = ? WHERE ID = ?";
    protected PreparedStatement updateStatement( Alliance alliance ) throws SQLException
    {
    	PreparedStatement update;
        update = getConnection().prepareStatement( UPDATE );
        update.setLong( 1, alliance.getAccount() );
        update.setLong( 2, alliance.getAllianceId() );
        update.setString( 3, alliance.getName() );
        update.setString( 4, alliance.getTag() );
        update.setLong( 5, alliance.getUpdateTime() );
        update.setString( 6, getNote( alliance.getNotes() ) );
        update.setLong( 7, alliance.getId() );
        return update;
    }

    @Override
    protected List<Alliance> createDTO( ResultSet resultSet ) throws SQLException
    {
        List<Alliance> list = new ArrayList<Alliance>(); 
        while(resultSet.next())
        {
            Long id = resultSet.getLong( "ID" );
            Alliance alliance = readCache(id);
            if(alliance == null )
            {
                alliance = new Alliance();
                alliance.setId( resultSet.getLong( "ID" ) );
                alliance.setAccount( resultSet.getLong( "ACCOUNT" ) );
                alliance.setAllianceId( resultSet.getLong( "ALLIANCEID" ) );
                alliance.setName( resultSet.getString( "NAME" ) );
                alliance.setTag( resultSet.getString( "TAG" ) );
                alliance.setUpdateTime( resultSet.getLong( "UPDATETIME" ) );
                alliance.setNotes(getNote( resultSet.getString( "NOTES" ) ) );
            }
            list.add(alliance);
            updateCache( alliance );
        }
        return list;
    }

}
