/****************************************
 *  COPYRIGHT (C) 2012
 *  Holger Graf
 ****************************************/
package siarchive.persistence.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import siarchive.MainFrame;
import siarchive.persistence.Account;

/**
 * @author graf
 *
 */
public class AccountDao extends BaseDao<Account>
{
    public AccountDao()
    {
        super( 10 );
    }

    public final static String table = "account";

    public List<Account> find()  throws SQLException
    {
        PreparedStatement get = getConnection().prepareStatement( "SELECT * FROM " + getTable() + " ORDER BY SEQUENCE ASC");
        ResultSet resultSet = get.executeQuery();
        List<Account> results = createDTO( resultSet );
        resultSet.close();
        get.close();
        return results;
    }

    public Account find(String name) throws SQLException
    {
        PreparedStatement find = getConnection().prepareStatement( "SELECT * FROM " + getTable() + " WHERE NAME=? ORDER BY SEQUENCE ASC", ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        find.setString( 1, name );
        ResultSet resultSet = find.executeQuery();
        List<Account> result = createDTO( resultSet );
        resultSet.close();
        find.close();
        // there can be at most 1
        return getFirst( result );
    }

    @Override
	protected Account findUnique(Account dbObject) throws SQLException
	{
		return find(dbObject.getName());
	}

	@Override
    protected List<Account> createDTO( ResultSet resultSet ) throws SQLException
    {
        List<Account> list = new ArrayList<Account>();
        while(resultSet.next())
        {
            Long id = resultSet.getLong( "ID" );
            Account account = readCache(id);
            if(account == null )
            {
                account = new Account();
                account.setId( id );
                account.setName( resultSet.getString( "NAME" ) );
                account.setDefault( resultSet.getBoolean( "ISDEFAULT" ) );
                account.setIndex( resultSet.getInt( "SEQUENCE" ) );
                account.setLanguage( MainFrame.parseLanguageAttribute( resultSet.getString( "LANGUAGE" ) ) );
                account.setFlavor( MainFrame.parseFlavorAttribute( resultSet.getString( "FLAVOR" ) ) );
                account.setOutpost( resultSet.getBoolean( "ISOUTPOST" ) );
            }
            list.add(account);
            updateCache( account );
        }
        return list;
    }

    @Override
    protected String getTable()
    {
        return table;
    }

    protected PreparedStatement createStatement( Account account ) throws SQLException
    {
        PreparedStatement create = getConnection().prepareStatement( "INSERT INTO " + getTable() + " (NAME, ISDEFAULT, SEQUENCE, LANGUAGE, FLAVOR, ISOUTPOST) values (?, ?, ?, ?, ?, ?)", Statement.RETURN_GENERATED_KEYS );
        create.setString( 1, account.getName() );
        create.setBoolean( 2, account.isDefault() );
        create.setInt( 3, account.getIndex() );
        create.setString( 4, account.getLanguage().name() );
        create.setString( 5, account.getFlavor().name() );
        create.setBoolean( 6, account.isOutpost() );
        return create;
    }

    protected PreparedStatement updateStatement( Account account ) throws SQLException
    {
        PreparedStatement update = getConnection().prepareStatement( "UPDATE " + getTable() + " SET NAME = ?, ISDEFAULT = ?, SEQUENCE = ?, LANGUAGE = ?, FLAVOR = ?, ISOUTPOST = ? WHERE ID = ?" );
        update.setString( 1, account.getName() );
        update.setBoolean( 2, account.isDefault() );
        update.setInt( 3, account.getIndex() );
        update.setString( 4, account.getLanguage().name() );
        update.setString( 5, account.getFlavor().name() );
        update.setBoolean( 6, account.isOutpost() );
        update.setLong( 7, account.getId() );
        return update;
    }

}
