/****************************************
 *  COPYRIGHT (C) 2011, 2017
 *  Holger Graf
 ****************************************/
package siarchive.persistence;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import siarchive.persistence.dao.BaseDao;

/**
 * @author graf
 *
 */
public class DatabaseFactory
{
    private static final String fileName = "datasource.properties";
    private static final String derby = "jdbc:derby:";
    private static final String sqlite = "jdbc:sqlite:";
    private static final Pattern variablePattern = Pattern.compile("\\$\\{([^\\}]+)\\}");

    private static String configFile = "<undefined>";
    private static Connection connection = null;
    private static Properties properties = null;
    private static Map<Class<?>, BaseDao<?>> daos = new HashMap<Class<?>, BaseDao<?>>();

    // suppress derby.log
    public static final OutputStream NO_DERBY_LOG = new OutputStream()
    {
        public void write(int b) { }
    };


    public static Connection getConnection()
    {
        return connection;
    }

    public static void close()
    {
        if(connection != null)
        {
            try
            {
                connection.close();
            }
            catch( SQLException e )
            {
                e.printStackTrace();
            }
            connection = null;
        }
    }

    public static void commit() throws SQLException
    {
        if(connection != null)
        {
            connection.commit();
        }
    }

    public static void rollback() throws SQLException
    {
        if(connection != null)
        {
            connection.rollback();
        }
    }

    public static Properties getProperties()
    {
        return properties;
    }

    public static String getConnectionName()
    {
        String dbName = null;
        if( properties != null )
        {
            dbName = properties.getProperty( "database.URL" );
        }
        return dbName;
    }

    public static String getConfigFile()
    {
        return configFile;
    }

    public static <T extends BaseDao<?>> T getDao(Class<T> daoType)
    {
        @SuppressWarnings("unchecked")
        T dao = (T)daos.get( daoType );
        if( dao == null )
        {
            try
            {
                dao = daoType.newInstance();
            }
            catch( InstantiationException e )
            {
                e.printStackTrace();
            }
            catch( IllegalAccessException e )
            {
                e.printStackTrace();
            }
            daos.put( daoType, dao );
        }
        return dao;
    }

    public static void open() throws DatabaseException
    {
        String database = "";
        try
        {
            // suppress derby.log
            System.setProperty( "derby.stream.error.field", DatabaseFactory.class.getCanonicalName() + ".NO_DERBY_LOG" );
            // read properties
            Properties properties = loadProperties();
            Driver driver = loadDriver( properties );
            database = getDataBase( properties );
            connection = driver.connect(database, properties);
            System.out.println("Connected to database " + database);

            connection.setAutoCommit(false);
            connection.setTransactionIsolation( Connection.TRANSACTION_READ_UNCOMMITTED );
        }
        catch( FileNotFoundException e )
        {
            throw new DatabaseException("cannot find datasource.properties", e);
        }
        catch( IOException e )
        {
            throw new DatabaseException("cannot read datasource.properties", e);
        }
        catch( DatabaseException e )
        {
            throw e;
        }
        catch( SQLException e )
        {
            throw new DatabaseException("could not connect to database " + database, e);
        }

    }

    public static String expandVariables( String path )
    {
    	StringBuffer buffer = new StringBuffer();
    	// find variables ${var} in path and replace with the system property var if one can be found
    	Matcher matcher = variablePattern.matcher(path);
    	while( matcher.find() )
    	{
    		// retrieve variable name
    		String orig = matcher.group();
    		String var = matcher.group(1);
    		if( var != null )
    		{
    			// get value or variable expression if no such property exists
    			String value = System.getProperty( var, orig );
    			// escape special chars
    			value = value.replace( "\\", "\\\\" ).replace( "$", "\\$" );
    			matcher.appendReplacement(buffer, value);
    		}
    	}
    	matcher.appendTail(buffer);
    	// tack on a final '/' if not already there
    	int last = buffer.length() - 1;
        if(buffer.charAt( last ) != File.separatorChar && buffer.charAt( last )  != '/' )
        {
            buffer.append( File.separator );
        }
    	return buffer.toString();
    }

    private static String getDataBase( Properties properties )
    {
        String database = properties.getProperty( "database.URL" );
        // adapt for embedded DB
        final String prefix = (database.indexOf( derby ) > -1) ? derby : (database.indexOf( sqlite ) > -1) ? sqlite : "";
        if(prefix.length() > 0)
        {
            String dir = properties.getProperty( "database.path" );
            if(dir != null && dir.trim().length() > 0)
            {
                dir = prefix + expandVariables(dir);
                database = database.replace( prefix, dir );
                properties.setProperty( "database.URL", database );
            }
        }

        return database;
    }

    private static Properties loadProperties() throws IOException
    {
        InputStream inStream = new FileInputStream( findPropertyFile() );
        properties = new Properties();
        properties.load( inStream );
        return properties;
    }

    private static File findPropertyFile()
    {
        File file;
        // find a suitable config file
        // start with command line option
        String configDir = System.getProperty( "siarchive.configdir" );
        file = new File( configDir + File.separator + fileName  );

        if( !file.exists() )
        {
            // try current dir (USB stick use case)
            configDir = System.getProperty( "user.dir" );
            file = new File( configDir + File.separator + fileName  );
        }

        if( !file.exists() )
        {
            // finally user home
            configDir = System.getProperty( "user.home" );
            file = new File( configDir + File.separator + fileName  );
        }
        configFile = file.getAbsolutePath();
        System.out.print( "properties file: " );
        System.out.println( configFile );
        return file;
    }

    private static Driver loadDriver(Properties properties) throws DatabaseException {
        String driver = properties.getProperty( "database.Driver" );
        if( driver == null )
        {
            throw new DatabaseException("database.Driver not defined");
        }
        String library = properties.getProperty( "database.Library" );
        final URL[] urls;
        if(library != null)
        {
            System.out.println("using library " + library);
            File libFile = new File(library);
            if(!libFile.exists())
            {
                String relativeLib = System.getProperty( "user.dir" ) + File.separator + library;
                libFile = new File( relativeLib  );
                if(!libFile.exists())
                {
                    throw new DatabaseException("Could not find library " + library + " or " + libFile);
                }
            }
            try
            {
                urls = new URL[] { libFile.toURI().toURL() };
            }
            catch (MalformedURLException e)
            {
                throw new DatabaseException("Classpath element '" + libFile + "' could not be used to create a valid file system URL");
            }
        }
        else
        {
            urls = new URL[0];
        }
        try
        {
            final URLClassLoader cl = new URLClassLoader(urls, ClassLoader.getSystemClassLoader());
            return (Driver)Class.forName(driver, true, cl).newInstance();
        }
        catch (ClassNotFoundException cnfe)
        {
            throw new DatabaseException("Unable to load the JDBC driver '" + driver + "'. Please check your CLASSPATH.", cnfe);
        }
        catch (InstantiationException ie)
        {
            throw new DatabaseException("Unable to instantiate the JDBC driver '" + driver + "'.", ie);
        }
        catch (IllegalAccessException iae)
        {
            throw new DatabaseException("Not allowed to access the JDBC driver '" + driver + "'.", iae);
        }
        catch (ClassCastException cce)
        {
            throw new DatabaseException("Not a JDBC driver: '" + driver + "'.", cce);
        }
    }
}
