/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/
package siarchive.persistence;

import java.util.AbstractSet;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

/**
 * @author graf
 * 
 */
public class DataSet<E> extends AbstractSet<E> implements Set<E>
{
    private HashMap<E, E> map;

    /**
     * Constructs a new, empty set; the backing <tt>HashMap</tt> instance has
     * default initial capacity (16) and load factor (0.75).
     */
    public DataSet()
    {
        map = new HashMap<E, E>();
    }

    /**
     * Constructs a new set containing the elements in the specified collection.
     * The <tt>HashMap</tt> is created with default load factor (0.75) and an
     * initial capacity sufficient to contain the elements in the specified
     * collection.
     * 
     * @param c
     *            the collection whose elements are to be placed into this set
     * @throws NullPointerException
     *             if the specified collection is null
     */
    public DataSet( Collection<? extends E> c )
    {
        map = new HashMap<E, E>( Math.max( ( int )( c.size() / .75f ) + 1, 16 ) );
        addAll( c );
    }

    public boolean add( E e )
    {
        return ( map.put( e, e ) == null );
    }

    /**
     * Returns an iterator over the elements in this set. The elements are
     * returned in no particular order.
     * 
     * @return an Iterator over the elements in this set
     * @see ConcurrentModificationException
     */
    public Iterator<E> iterator()
    {
        return map.keySet().iterator();
    }

    /**
     * Returns the number of elements in this set (its cardinality).
     * 
     * @return the number of elements in this set (its cardinality)
     */
    public int size()
    {
        return map.size();
    }

    /**
     * Returns <tt>true</tt> if this set contains no elements.
     * 
     * @return <tt>true</tt> if this set contains no elements
     */
    public boolean isEmpty()
    {
        return map.isEmpty();
    }

    /**
     * Returns <tt>true</tt> if this set contains the specified element. More
     * formally, returns <tt>true</tt> if and only if this set contains an
     * element <tt>e</tt> such that
     * <tt>(o==null&nbsp;?&nbsp;e==null&nbsp;:&nbsp;o.equals(e))</tt>.
     * 
     * @param o
     *            element whose presence in this set is to be tested
     * @return <tt>true</tt> if this set contains the specified element
     */
    public boolean contains( Object o )
    {
        return map.containsKey( o );
    }

    /**
     * Removes the specified element from this set if it is present.
     *
     * @param o object to be removed from this set, if present
     * @return <tt>true</tt> if the set contained the specified element
     */
    public boolean remove(Object o) 
    {
        return ( map.remove( o ) != null );
    }

    /**
     * Removes all of the elements from this set.
     * The set will be empty after this call returns.
     */
    public void clear() 
    {
        map.clear();
    }
    
    public E get( Object o )
    {
        return map.get( o );
    }
}
