/****************************************
 *  COPYRIGHT (C) 2011
 *  Holger Graf
 ****************************************/
package siarchive.init;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.table.AbstractTableModel;

import siarchive.MainFrame;
import siarchive.Resources;
import siarchive.persistence.DatabaseException;
import siarchive.persistence.DatabaseFactory;
import siarchive.persistence.SQLParser;
import siarchive.table.TooltipTable;

/**
 * @author graf
 *
 */
public class InitGui  extends JFrame
{
    private static final long serialVersionUID = -5569107566589870498L;
    private static final String info = "info";
    private static final String message = "message";
    private static final String subdir = "sql";

    private CardLayout layout;
    private JTextField propertyFile = new JTextField();
    private JComboBox<String> sqlSources = new JComboBox<>();
    private PropertyTableModel propertyListModel = new PropertyTableModel();
    private JButton infoDefault;
    private JLabel messageTitle = new JLabel();
    private JTextArea messageArea = new JTextArea();
    private JButton messageDefault;
    private JButton messageBack;

    /**
     * @param args
     */
    public static void main( String[] args )
    {
        InitGui init = new InitGui();
        init.validate();
        init.setVisible( true );
        try
        {
            DatabaseFactory.open();
        }
        catch (DatabaseException ex)
        {
            init.showErrorInfo(ex);
            init.actionExit();
        }
        init.actionShowInfo();
    }

    public InitGui() throws HeadlessException
    {
        super();
        this.setIconImage( Resources.getIcon( "/symbol.png" ).getImage() );
        enableEvents( AWTEvent.WINDOW_EVENT_MASK );

        JPanel contentPane = ( JPanel )this.getContentPane();
        setTitle( "SI Archive Initialization");

        layout = new CardLayout();
        contentPane.setLayout( layout );
        JPanel infoPanel = createInfoPanel();
        JPanel messagePanel = createMessagePanel();
        contentPane.add( infoPanel, info );
        contentPane.add( messagePanel, message );
        setSize( 640, 480 );
    }

    private JPanel createInfoPanel()
    {
        JPanel panel = new JPanel( new BorderLayout() );

        JPanel center = new JPanel();
        center.setBorder( BorderFactory.createEmptyBorder(10, 10, 10, 10));
        GridBagLayout gridBagLayout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        center.setLayout( gridBagLayout );

        // ComboBoxes are slightly higher than TextFields, adjust
        propertyFile.setPreferredSize(sqlSources.getPreferredSize());
        propertyFile.setEditable(false);
        propertyFile.setText( DatabaseFactory.getConfigFile() );

        constraints.fill = GridBagConstraints.HORIZONTAL;
        constraints.gridwidth = 1;
        constraints.weightx = 0.1;
        center.add( new JLabel( "Properties" ), constraints );
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.weightx = 1.0;
        center.add( propertyFile, constraints );

        JScrollPane scrollPane = new JScrollPane( new TooltipTable(propertyListModel)
        {

            private static final long serialVersionUID = -5570614492994721767L;

            @Override
            public Dimension getPreferredScrollableViewportSize() {
                Dimension d = super.getPreferredSize();
                d.height = 300;
                return d;
            }

        }, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        scrollPane.setBorder( BorderFactory.createEmptyBorder(10, 0, 10, 0) );
        center.add( scrollPane, constraints );

        constraints.gridwidth = 1;
        constraints.weightx = 0.1;
        center.add( new JLabel( "SQL File" ), constraints );
        constraints.weightx = 1.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        center.add( sqlSources, constraints );
        sqlSources.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(ItemEvent e)
            {
                infoDefault.setEnabled( sqlSources.getSelectedItem() != null );
            }
        });

        JPanel lower = new JPanel( new FlowLayout( FlowLayout.CENTER, 4, 4 ) );
        JButton button = new JButton( "Next >>" );
        button.addActionListener( new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                layout.next( InitGui.this.getContentPane() );
                actionShowMessage();
            }

        } );
        lower.add( button );
        infoDefault = button;
        infoDefault.setEnabled( false );

        panel.add( center, BorderLayout.NORTH );
        panel.add( lower, BorderLayout.SOUTH );

        return panel;
    }

    private JPanel createMessagePanel()
    {
        JPanel panel = new JPanel( new BorderLayout() );
        messageTitle.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        messageTitle.setText( DatabaseFactory.getConfigFile() );
        panel.add( messageTitle, BorderLayout.NORTH );
        JPanel center = new JPanel( new BorderLayout());
        center.setBorder( BorderFactory.createEtchedBorder() );
        messageArea.setEditable( false );
        JScrollPane scrollPane = new JScrollPane( messageArea,
                        JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                        JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        center.add( scrollPane );

        JPanel lower = new JPanel( new FlowLayout( FlowLayout.CENTER, 4, 4 ) );
        messageBack = new JButton( "<< Back" );
        messageBack.addActionListener( new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                layout.first( InitGui.this.getContentPane() );
                actionShowInfo();
            }

        } );
        lower.add( messageBack );
        messageBack.setEnabled( false );

        messageDefault = new JButton( "Finish (x)" );
        messageDefault.addActionListener( new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                actionExit();
            }
        } );
        lower.add( messageDefault );
        getRootPane().setDefaultButton( messageDefault );

        messageDefault.setEnabled( false );

        panel.add( center, BorderLayout.CENTER );
        panel.add( lower, BorderLayout.SOUTH );

        return panel;
    }

    private void actionShowInfo()
    {
        propertyFile.setText( DatabaseFactory.getConfigFile() );
        infoDefault.setEnabled( sqlSources.getSelectedItem() != null );
        getRootPane().setDefaultButton( infoDefault );

        List<File> sqlDirs = new ArrayList<File>();
        String user_dir = System.getProperty("user.dir");
        sqlDirs.add( new File( user_dir ) );
        String user_home = System.getProperty("user.home");
        sqlDirs.add( new File( user_home ) );
        Properties dbProperties = DatabaseFactory.getProperties();
        String dbType = null;
        if( dbProperties != null )
        {
            propertyListModel.setData( dbProperties );
            String sql_dir = dbProperties.getProperty("sql.dir");
            if( sql_dir != null )
            {
                sql_dir = DatabaseFactory.expandVariables( sql_dir );
                sqlDirs.add( new File( sql_dir ) );
            }
            String db_url = dbProperties.getProperty("database.URL");
            if(db_url != null) {
                String[] parts = db_url.split(":", 3);
                if(parts.length > 2) {
                    dbType = parts[1].toLowerCase();
                }
            }
        }
        Collection<String> sqlFiles = getSqlFiles(sqlDirs);

        String current = null;
        sqlSources.removeAllItems();
        for( String sqlFile : sqlFiles )
        {
            sqlSources.addItem( sqlFile );
            if(dbType != null && sqlFile.toLowerCase().contains(dbType)) {
                current = sqlFile;
            }
        }
        if( current != null )
        {
            sqlSources.setSelectedItem( current );
        }
        else if( sqlSources.getItemCount() > 0)
        {
            sqlSources.setSelectedIndex( 0 );
        }

    }

    private void actionShowMessage()
    {
        messageArea.setText("");
        getRootPane().setDefaultButton( messageDefault );
        messageBack.setEnabled( false );
        messageDefault.setEnabled( false );

        String sqlFile = (String)sqlSources.getSelectedItem();
        messageTitle.setText( sqlFile );

        // trigger sql execution in separate thread
        new Thread( new MessageAppender( sqlFile ) ).start();
    }

    private void appendText(String sql)
    {
        messageArea.append( sql );
        messageArea.append( "\n" );
        messageArea.setCaretPosition( messageArea.getDocument().getLength() );
        messageArea.repaint();
    }

    private Collection<String> getSqlFiles( List<File> searchDirs )
    {
        // suppress double entries
        Set<String> sqlFiles = new TreeSet<String>();
        for( File dir : searchDirs )
        {
            sqlFiles.addAll( getSqlFilesInDir( dir ) );
            sqlFiles.addAll( getSqlFilesInDir( new File( dir, subdir ) ) );
        }
        return sqlFiles;
    }

    private Collection<String> getSqlFilesInDir(File dir)
    {
        List<String> sqlFiles = new ArrayList<String>();
        if( dir.exists() && dir.isDirectory() )
        {
            File[] files = dir.listFiles(new FilenameFilter()
            {
                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(".sql") && new File(dir, name).isFile();
                }
            });
            if( files != null && files.length > 0 )
            {
                for( File file : files )
                {
                    sqlFiles.add( file.getAbsolutePath() );
                }
            }
        }
        return sqlFiles;
    }

    private void showErrorInfo( Exception ex )
    {
        MainFrame.showErrorInfo(this, "Warning", ex);
    }

    // Overridden so we can exit when window is closed
    protected void processWindowEvent( WindowEvent e )
    {
        super.processWindowEvent( e );
        if( e.getID() == WindowEvent.WINDOW_CLOSING )
        {
            actionExit();
        }
    }

    private void actionExit()
    {
        try
        {
            DatabaseFactory.commit();
        }
        catch( SQLException ex )
        {
            try
            {
                DatabaseFactory.rollback();
            }
            catch( SQLException ex1 )
            {
                ex1.printStackTrace();
            }
            showErrorInfo( ex );
        }
        finally
        {
            // close DB connection
            DatabaseFactory.close();
        }
        System.exit( 0 );
    }

    class MessageAppender implements Runnable
    {
        private String sqlFile;
        public MessageAppender( String sqlFile )
        {
            this.sqlFile = sqlFile;
        }

        @Override
        public void run()
        {
            Connection connection = DatabaseFactory.getConnection();

            try
            {
                Statement statement = connection.createStatement();

                SQLParser parser = new SQLParser( sqlFile );
                String sql = null;
                while( (sql = parser.getNextStatement()) != null )
                {
                    appendText( sql );
                    try
                    {
                        statement.execute( sql );
                    }
                    catch( SQLException ex)
                    {
                        DatabaseFactory.rollback();
                        throw ex;
                    }
                }

                // make it real
                DatabaseFactory.commit();

                messageDefault.setEnabled( true );
            }
            catch( SQLException ex)
            {
                showErrorInfo(ex);
            }
            catch( FileNotFoundException ex )
            {
                showErrorInfo(ex);
            }
            catch( IOException ex )
            {
                showErrorInfo(ex);
            }

            // only now enable back button
            messageBack.setEnabled( true );
        }

    }

    class PropertyTableModel extends AbstractTableModel
    {
        private static final long serialVersionUID = -2821249160388159034L;
        private List<Map.Entry<Object, Object>> properties = new ArrayList<Map.Entry<Object,Object>>();

        @Override
        public int getRowCount()
        {
            return properties.size();
        }

        @Override
        public int getColumnCount()
        {
            return 2;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex)
        {
            Object value = "";
            switch( columnIndex )
            {
                case 0:
                    value = properties.get(rowIndex).getKey();
                    break;
                case 1:
                    value = properties.get(rowIndex).getValue();
                    break;

            }
            return value;
        }

        @Override
        public String getColumnName(int column)
        {
            return "";
        }

        @Override
        public Class<?> getColumnClass(int columnIndex)
        {
            return Object.class;
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex)
        {
            return false;
        }

        public void setData( Properties properties)
        {
            this.properties.clear();
            this.properties.addAll( properties.entrySet() );
            Collections.sort( this.properties, new Comparator<Map.Entry<Object, Object>>()
            {
                @Override
                public int compare(Entry<Object, Object> o1, Entry<Object, Object> o2)
                {
                    // Properties is a Hashtable and has no 'null' keys
                    return o1.getKey().toString().compareTo( o2.getKey().toString() );
                }
            } );
            fireTableDataChanged();
        }
    }
}
