/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/

package siarchive.imports;

import java.sql.SQLException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.jr.ob.JSON;

import siarchive.DataManager;
import siarchive.components.Position;
import siarchive.datatransfer.JsonScanner;
import siarchive.json.JSONTags;
import siarchive.json.spyreport.SpyReportBean;
import siarchive.persistence.SpyReport;

/**
 * @author graf
 *
 */
public class SpyReportImporter
{
    private DataManager dataManager;
    private Pattern positionPattern;

    public SpyReportImporter( DataManager dataManager )
    {
        this.dataManager = dataManager;
    }

    public void update()
    {
        positionPattern = Pattern.compile( dataManager.getI18nText( "ImportFrame.spyMarker" ) + "\\s+(\\S*.*)\\[(\\d+):(\\d+):(\\d+)\\s*\\]" );
    }

    public void scan( String input ) throws SQLException
    {
        SpyReport report;
        if(input.indexOf(JSONTags.reportType_SpyReport) >= 0) {
            report = scanSpyReportJSON(input);
        }
        else {
            report = scanSpyReport(input);
        }

        dataManager.save( report );
        dataManager.commit();
    }

    private SpyReport scanSpyReport( String input )
    {
        SpyReport report = new SpyReport();
        report.setAccount( dataManager.getActiveAccountId() );
        Matcher m;
        m = DateParser.datePattern.matcher( input );
        if( m.find() )
        {
            long date = DateParser.parse( m.group() );
            report.setCreationTime( date );
        }
        m = positionPattern.matcher( input );
        if( m.find() )
        {
            // get planet
            String name = m.group( 1 ).trim();
            report.setPlanetName( name );
            int gal = Integer.parseInt( m.group( 2 ) );
            int sys = Integer.parseInt( m.group( 3 ) );
            int pl = Integer.parseInt( m.group( 4 ) );
            report.setPosition( Position.createId( gal, sys, pl ) );
            boolean astro = dataManager.getI18nText( "Asteroid" ).equals( name );
            report.setAsteroid( astro );
        }
        report.setData( input );
        report.setRetainNotes( true );

        return report;
    }

    private SpyReport scanSpyReportJSON( String input )
    {
        SpyReport report = new SpyReport();
        report.setAccount( dataManager.getActiveAccountId() );
        try {
            SpyReportBean reportBean = JSON.std.beanFrom(SpyReportBean.class, input);
            JsonScanner.scanSpyReportJSON(report, reportBean, dataManager);
        }
        catch( Exception e1 ) {
            e1.printStackTrace();
        }
        report.setData( input );
        report.setRetainNotes( true );

        return report;
    }
}
