/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/

package siarchive.imports;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.FlavorEvent;
import java.awt.datatransfer.FlavorListener;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.sql.SQLException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.border.EtchedBorder;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.components.ModulePane;
import siarchive.components.Position;
import siarchive.json.JSONTags;

/**
 * @author graf
 *
 */
public class ImportPane extends JPanel implements ModulePane
{
    private static final long serialVersionUID = -2664457786424424808L;

    private JTextArea textArea = new JTextArea();
    private JButton button;

    private GalaxyImporterNT galaxyImporter;
    private BattleReportImporter battleReportImporter;
    private SpyReportImporter spyReportImporter;

    private Pattern spyPattern;
    private Pattern battlePattern;
    private Pattern galaxyPatternNT = Pattern.compile(JSONTags.reportType_GalaxyOverview);
;

    private MainFrame parent;
    private DataManager dataManager;
    private FlavorListener flavorListener;
    private final Transferable imageSelection;

    public ImportPane( MainFrame parent )
    {
        this.parent = parent;
        this.dataManager = parent.getDataManager();

        galaxyImporter = new GalaxyImporterNT( dataManager );
        battleReportImporter = new BattleReportImporter( dataManager );
        spyReportImporter = new SpyReportImporter( dataManager );

        setLayout( new BorderLayout() );
        setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );

        JPanel center = createCenterPanel();

        JPanel lower = new JPanel( new FlowLayout( FlowLayout.CENTER, 4, 4 ) );
        button = new JButton( dataManager.getI18nText( "Import" ) );
        button.addActionListener( new ActionListener()
        {
            public void actionPerformed(ActionEvent e)
            {
                actionImport();
            }
        } );
        lower.add( button );

        add( center, BorderLayout.CENTER );
        add( lower, BorderLayout.EAST );

        imageSelection = new Transferable() {
            @Override
            public boolean isDataFlavorSupported(DataFlavor flavor) {
                return DataFlavor.imageFlavor.equals( flavor );
            }

            @Override
            public DataFlavor[] getTransferDataFlavors() {
                return new DataFlavor[] { DataFlavor.imageFlavor };
            }

            @Override
            public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
                return null;
            }
        };

        flavorListener = new FlavorListener()
        {
            public void flavorsChanged( FlavorEvent e )
            {
                Clipboard clipboard = ( Clipboard )e.getSource();
                if( inputFromClipBoard( clipboard ) )
                {
                    actionImport();
                    // this is somewhat ugly
                    // to be notified on the next text input, we'll put an image in
                    // to change the data flavor to something else
                    boolean contentSet= false;
                    while(!contentSet) {
                        try {
                            clipboard.setContents( imageSelection, null );
                            contentSet = true;
                        }
                        catch(IllegalStateException ex) {
                            contentSet = false;
                            try {
                                Thread.sleep(200);
                            }
                            catch(InterruptedException e1) {}
                        }
                    }
                }
            }
        };

        // by popular demand, add paste clipboard to RMB
        MouseAdapter l = new MouseAdapter()
        {

            @Override
            public void mouseClicked( MouseEvent e )
            {
                if( SwingUtilities.isRightMouseButton( e ) )
                {
                    Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                    textArea.setText( "" );
                    inputFromClipBoard( clipboard );
                }
            }

        };

        textArea.addMouseListener( l );
    }

    public void setGrabClipboard( boolean isEnabled )
    {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        if( isEnabled )
        {
            // bootstrap, s.a.
            clipboard.setContents( imageSelection, null );
            clipboard.addFlavorListener( flavorListener );
        }
        else
        {
            clipboard.removeFlavorListener( flavorListener );
        }
    }

    protected boolean inputFromClipBoard( Clipboard clipboard )
    {
        {
            Transferable contents = clipboard.getContents( null );
            boolean hasTransferableText = ( ( contents != null ) && contents.isDataFlavorSupported( DataFlavor.stringFlavor ) );
            if( hasTransferableText )
            {
                String input = "";
                try
                {
                    input = ( String )contents.getTransferData( DataFlavor.stringFlavor );
                }
                catch( UnsupportedFlavorException ex )
                {
                    System.out.println( ex );
                    ex.printStackTrace();
                }
                catch( IOException ex )
                {
                    System.out.println( ex );
                    ex.printStackTrace();
                }
                textArea.setText( input );
            }
            return hasTransferableText;
        }
    }

    protected void actionImport()
    {
        String input = textArea.getText();
        Matcher m;
        parent.dbOn();
        try
        {
            m = spyPattern.matcher( input );
            if( m.find() )
            {
                spyReportImporter.scan( input );
                textArea.setText( "" );
                if( parent.isVisibleGalaxyView() )
                {
                    parent.updatePane();
                }
            }
            else
            {
                m = battlePattern.matcher( input );
                if( m.find() )
                {
                    battleReportImporter.scan( input );
                    textArea.setText( "" );
                    if( parent.isVisibleGalaxyView() )
                    {
                        parent.updatePane();
                    }
                }
                else
                {
                    m = galaxyPatternNT.matcher( input );
                    if( m.find() )
                    {
                        Position position = galaxyImporter.scan( input );
                        textArea.setText( "" );
                        if( position.isRealPosition(dataManager.getFlavor()) )
                        {
                            parent.showGalaxyView( position.getGalaxy(), position.getSystem(), parent.isVisibleGalaxyView() );
                        }
                    }
                }
            }
        }
        catch( Exception ex )
        {
            try
            {
                dataManager.rollback();
            }
            catch( SQLException e )
            {
                e.printStackTrace();
            }
            parent.showErrorInfo( ex );
        }

        parent.dbOff();
        //parent.updatePane();
    }

    protected JPanel createCenterPanel()
    {
        JPanel center = new JPanel( new BorderLayout() );
        center.setBorder( BorderFactory.createEmptyBorder( 5, 10, 5, 10) );

        JScrollPane scrollPane = new JScrollPane( textArea,
                        JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                        JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS );
        center.add( scrollPane, BorderLayout.CENTER );
        scrollPane.setBorder( null );
        return center;
    }

    @Override
    public Dimension getPreferredSize()
    {
        Dimension size = super.getPreferredSize();
        if( size.height < 70 )
        {
            size.height = 70;
        }
        return size;
    }

    public void update()
    {
        boolean isEnabled = ( dataManager.getActiveAccount() != null );
        button.setEnabled( isEnabled );
        textArea.setEnabled( isEnabled );
        textArea.setEditable( isEnabled );

        spyPattern = Pattern.compile( "(" + dataManager.getI18nText( "ImportFrame.spyMarker" ) + ")|(" + JSONTags.reportType_SpyReport + ")", Pattern.DOTALL );
        battlePattern = Pattern.compile( "(" + dataManager.getI18nText( "ImportFrame.battleMarker" ) + ")|(" + JSONTags.reportType_BattleReport + ")", Pattern.DOTALL );

        battleReportImporter.update();;
        spyReportImporter.update();;

        updateLabel();
    }

    private void updateLabel()
    {
        button.setText( dataManager.getI18nText( "Import" ) );
    }
}
