/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/

package siarchive.imports;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.jr.ob.JSON;

import siarchive.DataManager;
import siarchive.components.Position;
import siarchive.datatransfer.JsonScanner;
import siarchive.json.JSONTags;
import siarchive.json.battlereport.BattleReportBean;
import siarchive.persistence.BattleReport;

/**
 * @author graf
 *
 */
public class BattleReportImporter
{
	private DataManager dataManager;
    private Pattern attackerPattern;
    private Pattern defenderPattern;

	public BattleReportImporter( DataManager dataManager )
	{
		this.dataManager = dataManager;
	}

	public void update()
	{
        attackerPattern = Pattern.compile( dataManager.getI18nText( "BattleReport.attacker" ) + "\\s+(\\S*.*)\\[(\\d+):(\\d+):(\\d+)\\s*\\]" );
        defenderPattern = Pattern.compile( dataManager.getI18nText( "BattleReport.defender" ) + "\\s+(\\S*.*)\\[(\\d+):(\\d+):(\\d+)\\s*\\]" );
	}

	public void scan( String input ) throws SQLException
	{
	    BattleReport report;
	    if(input.indexOf(JSONTags.reportType_BattleReport) >= 0) {
	        report = scanBattleReportJSON(input);
	    }
	    else {
	        report = scanBattleReport(input);
	    }

        dataManager.save( report );
        dataManager.commit();
	}

    private BattleReport scanBattleReport( String input )
    {
        BattleReport report = new BattleReport();
        report.setAccount( dataManager.getActiveAccountId() );
        Matcher m;
        m = DateParser.datePattern.matcher( input );
        if( m.find() )
        {
            long date = DateParser.parse( m.group() );
            report.setCreationTime( date );
        }
        List<String> attacker = new ArrayList<String>();
        m = attackerPattern.matcher( input );
        while( m.find() )
        {
            // get planet
            String name = m.group( 1 ).trim();
            if( !attacker.contains( name ) )
            {
                attacker.add( name );
            }
        }
        StringBuilder sb = new StringBuilder();
        for( int i = 0; i < attacker.size(); i++ )
        {
            if( i > 0 )
            {
                sb.append( ", " );
            }
            sb.append( attacker.get( i ) );
        }
        report.setAttacker( sb.toString() );

        m = defenderPattern.matcher( input );
        if( m.find() )
        {
            // get planet
            String name = m.group( 1 ).trim();
            report.setDefender( name );
            int gal = Integer.parseInt( m.group( 2 ) );
            int sys = Integer.parseInt( m.group( 3 ) );
            int pl = Integer.parseInt( m.group( 4 ) );
            report.setPosition( Position.createId( gal, sys, pl ) );
        }
        // there is no way to identify the battle report from an asteroid
        report.setAsteroid( false );
        report.setData( input );
        report.setRetainNotes( true );

        return report;
    }

    private BattleReport scanBattleReportJSON( String input ) {
        final BattleReport report = new BattleReport();
        report.setAccount( dataManager.getActiveAccountId() );
        try {
            BattleReportBean reportBean = JSON.std.beanFrom(BattleReportBean.class, input);
            JsonScanner.scanBattleReportJSON(report, reportBean, dataManager);
        }
        catch( IOException e1 ) {
        }
        report.setData( input );
        report.setRetainNotes(true);

        return report;
    }
}
