/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/

package siarchive.galaxy;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.sql.SQLException;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.ToolTipManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.Resources;
import siarchive.components.ModulePane;
import siarchive.components.PlanetarySystem;
import siarchive.components.Position;
import siarchive.components.PositionField;

/**
 * @author graf
 *
 */
public class GalaxyView extends JFrame implements ModulePane
{
    private static final long serialVersionUID = 9100127657367061025L;

    private DataManager dataManager;
    private MainFrame parent;
    private GalaxyComponent galaxy;
    private GalaxyOverview galaxyOverview;

    private PositionField positionField;
    private JLabel updateLabel = new JLabel();
    private JLabel updateTime = new JLabel();

    public GalaxyView( MainFrame parent ) throws HeadlessException
    {
        super();
        this.parent = parent;
        this.dataManager = parent.getDataManager();
        this.galaxy = new GalaxyComponent( this );
        this.galaxyOverview = new GalaxyOverview( this );

        this.setIconImage( Resources.getIcon( "/symbol.png" ).getImage() );
        enableEvents( AWTEvent.WINDOW_EVENT_MASK );

        // customize ToolTip
        ToolTipManager.sharedInstance().setInitialDelay( 350 );
        ToolTipManager.sharedInstance().setDismissDelay( 120000 );
        ToolTipManager.sharedInstance().setReshowDelay( 200 );

        JPanel contentPane = ( JPanel )this.getContentPane();
        contentPane.setLayout( new BorderLayout() );

        JPanel centerPanel = createCenterPanel();
        contentPane.add( centerPanel, BorderLayout.CENTER );
        JPanel southPanel = new JPanel( new FlowLayout( FlowLayout.CENTER, 2, 5 ) );
        southPanel.add( galaxyOverview );
        contentPane.add( southPanel, BorderLayout.SOUTH );

        pack();
        setResizable( false );
    }

    public DataManager getDataManager()
    {
        return dataManager;
    }

    public MainFrame getMainFrame()
    {
        return parent;
    }

    protected void processWindowEvent( WindowEvent e )
    {
        if( e.getID() == WindowEvent.WINDOW_CLOSING )
        {
            parent.closeGalaxyView();
        }
        super.processWindowEvent( e );
    }

    protected JPanel createCenterPanel()
    {
        JPanel centerPanel = new JPanel( new BorderLayout() );
        JPanel northPanel = new JPanel( new BorderLayout() );
        JPanel buttonPanel = new JPanel( new FlowLayout( FlowLayout.LEFT, 5, 5 ) );
        positionField = new PositionField( false );
        positionField.addChangeListener( new ChangeListener()
        {
            public void stateChanged(ChangeEvent e)
            {
                actionShow();
            }
        });
        buttonPanel.add( positionField );
        JButton goButton = new JButton( "Go" );
        goButton.addActionListener( new ActionListener()
        {
            public void actionPerformed(ActionEvent e)
            {
                actionShow();
            }
        });
        buttonPanel.add( goButton );
        northPanel.add( buttonPanel, BorderLayout.CENTER );
        JPanel timePanel = new JPanel( new GridLayout( 1, 2 ) );
        timePanel.setBorder( BorderFactory.createEmptyBorder( 5, 0, 5, 10 ) );
        updateLabel.setBorder( BorderFactory.createEtchedBorder() );
        updateTime.setBorder( BorderFactory.createEtchedBorder() );
        timePanel.add( updateLabel );
        timePanel.add( updateTime );
        northPanel.add( timePanel, BorderLayout.EAST );

        centerPanel.add( northPanel, BorderLayout.NORTH );
        centerPanel.add( galaxy, BorderLayout.CENTER );

        return centerPanel;
    }

    protected void actionShow()
    {
        Position position = positionField.getPosition();
        galaxy.setSystem( position.getGalaxy(), position.getSystem() );
        galaxy.repaint();
        galaxyOverview.repaint();
    }

    protected Position getPosition()
    {
        return positionField.getPosition();
    }

    public void setSystem( int galaxy, int system )
    {
        int id = Position.createId( galaxy, system, 1 );
        positionField.setPosition( Position.createFromId( id ) );
        galaxyOverview.update(galaxy, system);
        //        this.galaxy.setSystem( galaxy, system );
        //        this.galaxy.repaint();
    }

    @Override
    public void setVisible(boolean b)
    {
        super.setVisible(b);
        if( b )
        {
            galaxy.requestFocus();
        }
    }

    protected void updateLabel() throws SQLException
    {
        updateLabel.setText( dataManager.getI18nText( "GalaxyView.lastUpdated" ) );
        Position position = getPosition();
        PlanetarySystem system = dataManager.getSystem( position.getGalaxy(), position.getSystem() );
        String updateText;
        if( system != null && system.isValidTime() )
        {
            updateText = DataManager.getDateFormat().format( system.getTime() );
        }
        else
        {
            updateText = "                   ";
        }
        updateTime.setText( updateText );

    }

    public void update()
    {
        setTitle( dataManager.getI18nText( "MainFrame.menu.tools.galaxy" ) );

        positionField.setMinGalaxies(Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        positionField.setMaxGalaxies(Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()));
        positionField.setMaxSystems(Position.getUpperSystemBound(dataManager.getFlavor()));
        galaxyOverview.update();
        galaxy.update();
        pack();
    }

}
