/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/

package siarchive.galaxy;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.border.BevelBorder;

import siarchive.DataManager;
import siarchive.components.Flavor;
import siarchive.components.Position;

/**
 * @author graf
 *
 */
public class GalaxyOverview extends JComponent
{
    private static final long serialVersionUID = 7673774522871792355L;
    private static int lineHeight = 5;
    private int lineWidth;

    private static Color backGround = Color.white;
    private static Color current = Color.blue.darker();

    private static Color[] colors = new Color[] {
                    new Color( 0, 190, 20 ), new Color( 30, 190, 30 ), new Color( 60, 190, 30 ),
                    new Color( 90, 190, 40 ), new Color( 120, 190, 40 ), new Color( 150, 190, 40 ),
                    new Color( 180, 190, 40 ), new Color( 200, 190, 40 ), new Color( 210, 195, 40 ),
                    new Color( 220, 200, 30 ), new Color( 225, 205, 30 ), new Color( 235, 220, 30 ),
                    new Color( 240, 230, 30 ), new Color( 250, 225, 30 ), new Color( 255, 210, 30 ),
                    new Color( 255, 200, 30 ), new Color( 255, 190, 20 ), new Color( 250, 170, 20 ),
                    new Color( 245, 150, 30 ), new Color( 240, 140, 20 ), new Color( 230, 130, 20 ),
                    new Color( 230, 110, 30 ), new Color( 230, 95, 20 ), new Color( 220, 85, 20 ),
                    new Color( 220, 75, 20 ), new Color( 210, 65, 20 ), new Color( 195, 60, 20 ),
                    new Color( 190, 50, 20 )
    };

    private static int colorsPerDay = 2;
    private static int days = colors.length / colorsPerDay;
    private static long msPerDay = 24 * 60 * 60 * 1000L;
    private static final long cutOff = days * msPerDay;

    private DataManager dataManager;
    private GalaxyView parent;

    // time cache for color display
    private Map<Integer, Long> times = new HashMap<Integer, Long>();

    public GalaxyOverview( GalaxyView parent )
    {
        super();
        this.parent = parent;
        this.dataManager = parent.getDataManager();
        lineWidth = calculateLineWidth();

        setOpaque( true );
        setBorder( BorderFactory.createBevelBorder( BevelBorder.LOWERED ) );
        setToolTipText( " " );
        addMouseListener( new MouseAdapter()
        {
            public void mouseClicked( MouseEvent event )
            {
                int galaxy = ( event.getY() > 0 ) ? ( event.getY() - 1 ) / lineHeight + 1 : 1;
                int system = ( event.getX() > 0 ) ? ( event.getX() - 1 ) / lineWidth + 1 : 1;
                actionSetSystem( galaxy, system );
            }
        } );
    }

    protected void actionSetSystem( int galaxy, int system )
    {
        int lowerGalaxyBound = Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost());
        int upperGalaxyBound = Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost());
        galaxy = ( galaxy > lowerGalaxyBound ) ? galaxy : lowerGalaxyBound;
        galaxy = ( galaxy < upperGalaxyBound ) ? galaxy : upperGalaxyBound;
        system = ( system > Position.lowerSystemBound ) ? system : Position.lowerSystemBound;
        system = ( system < Position.getUpperSystemBound(dataManager.getFlavor()) ) ? system : Position.getUpperSystemBound(dataManager.getFlavor());
        parent.setSystem( galaxy, system );
    }

    protected void paintComponent(Graphics g)
    {
        super.paintComponent(g);

        int width = getWidth();
        int height = getHeight();
        int x;
        int y;
        long now = System.currentTimeMillis();
        // prepare background
        g.setColor( backGround );
        g.drawRect( 0, 0, width, height );
        g.fillRect( 0, 0, width, height );

        // show active systems
        boolean isOutpost = dataManager.isOutpost();
        Flavor flavor = dataManager.getFlavor();
        int lowerBound = Position.getLowerGalaxyBound(flavor, isOutpost);
        int upperBound = Position.getUpperGalaxyBound(flavor, isOutpost);
        for( int gal = lowerBound; gal <= upperBound; gal++ )
        {
            for( int sys = 1; sys <= Position.getUpperSystemBound(dataManager.getFlavor()); sys++ )
            {
                Integer position = Position.createId(gal, sys, 0);
                Long time = times.get(position);
                if( time != null  )
                {
                    long updateInterval = now - time.longValue();
                    g.setColor( getColor( updateInterval ) );
                    x = lineWidth * ( sys - 1) + 2;
                    y = lineHeight * ( gal - lowerBound ) + 2;
                    g.drawRect( x, y, lineWidth, lineHeight - 1 );
                    g.fillRect( x, y, lineWidth, lineHeight );
                }
            }
        }


        g.setColor( current );
        Position position = parent.getPosition();
        x = lineWidth * (position.getSystem() - 1) + 2;
        y = lineHeight * ( position.getGalaxy() - lowerBound ) + 2;
        g.drawRect( x, y, lineWidth, lineHeight - 1 );

    }

    @Override
    public String getToolTipText( MouseEvent event )
    {
        int lowerGalaxyBound = Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost());
        int upperGalaxyBound = Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost());
        int galaxy = ( event.getY() > 0 ) ? ( event.getY() - 1 ) / lineHeight + lowerGalaxyBound : lowerGalaxyBound;
        int system = ( event.getX() > 0 ) ? ( event.getX() - 1 ) / lineWidth + 1 : 1;
        if( galaxy < lowerGalaxyBound ) {
            galaxy = upperGalaxyBound;
        } else if( galaxy > upperGalaxyBound ) {
            galaxy = upperGalaxyBound;
        }
        if( system > Position.getUpperSystemBound(dataManager.getFlavor()) ) {
            system = Position.getUpperSystemBound(dataManager.getFlavor());
        }

        StringBuilder sb = new StringBuilder();
        sb.append('[');
        sb.append( galaxy );
        sb.append(':');
        sb.append( system );
        sb.append(']');
        return sb.toString();
    }

    protected static  Color getColor( long updateInterval )
    {
        if( updateInterval > cutOff )
        {
            updateInterval = cutOff;
        }
        int colorIndex = (int)( colorsPerDay * updateInterval / msPerDay );
        if( colorIndex >= colors.length )
        {
            colorIndex = colors.length - 1;
        }
        return colors[colorIndex];
    }

    public void update()
    {
        // time cache update
        //System.out.print("GalaxyOverview update... ");
        //long now = System.currentTimeMillis();
        try
        {
            times.clear();
            times.putAll( dataManager.findMinTimes() );
        }
        catch (SQLException ex)
        {
            parent.getMainFrame().showErrorInfo(ex);
        }
        //System.out.println( "finished " + (System.currentTimeMillis() - now) + "ms" );
        // and repaint()
        lineWidth = calculateLineWidth();
        revalidate();
        repaint();
    }

    public void update( int galaxy, int system )
    {
        try
        {
            updateCache( galaxy, system );
        }
        catch (SQLException ex)
        {
            parent.getMainFrame().showErrorInfo(ex);
        }
    }

    @Override
    public Dimension getPreferredSize() {
        Flavor flavor = dataManager.getFlavor();
        boolean isOutpost = dataManager.isOutpost();
        int galaxies = Position.getUpperGalaxyBound(flavor, isOutpost) - Position.getLowerGalaxyBound(flavor, isOutpost) + 1;
        Dimension size = new Dimension( lineWidth * ( Position.getUpperSystemBound(flavor) + 1 ) + Position.getUpperSystemBound(flavor) / 100 - 1, lineHeight * galaxies + 4 );
        return size;
    }

    private void updateCache(int galaxy, int system) throws SQLException
    {
        int startPosition = Position.createId(galaxy, system, 0);
        int endPosition = Position.createId(galaxy, system, Position.upperPlanetBound);
        Long time = dataManager.findMinTimeByPosition(startPosition, endPosition);
        times.put(startPosition, time);
    }

    private int calculateLineWidth() {
        return 2 * 400 / Position.getUpperSystemBound(dataManager.getFlavor());
    }
}
