/****************************************
 *  COPYRIGHT (C) 2010, 2024
 *  Holger Graf
 ****************************************/
package siarchive.galaxy;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.sql.SQLException;
import java.util.Map;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.CellRendererPane;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.JToolTip;
import javax.swing.plaf.basic.BasicToolTipUI;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import siarchive.DataManager;
import siarchive.components.Cost;
import siarchive.components.Position;
import siarchive.components.Resource;
import siarchive.components.Ship;
import siarchive.components.Status;
import siarchive.persistence.BlackHole;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.table.AlternateLineTableCellRenderer;

/**
 * @author graf
 *
 */
public class GalaxyComponentToolTipUI extends BasicToolTipUI
{
    static private GalaxyComponentToolTipUI sharedInstance = new GalaxyComponentToolTipUI();
    protected CellRendererPane rendererPane;

    private JTable toolTipArea ;
    private DataManager dataManager = null;
    private Pattern positionPattern = Pattern.compile( "\\[(\\d+):(\\d+):(\\w+)\\]" );
    private static final Color bgColor = new Color( 30, 55, 85 );
    private AlternateLineTableCellRenderer cellRenderer;

    public static GalaxyComponentToolTipUI createUI(JComponent c)
    {
        return sharedInstance;
    }

    public GalaxyComponentToolTipUI()
    {
        super();
        cellRenderer = new AlternateLineTableCellRenderer();
        cellRenderer.setForeground( Color.white );
    }

    public void installUI(JComponent c)
    {
        super.installUI(c);
        rendererPane = new CellRendererPane();
        c.add(rendererPane);
    }

    public void uninstallUI(JComponent c)
    {
        super.uninstallUI(c);

        c.remove(rendererPane);
        rendererPane = null;
    }

    public void paint(Graphics g, JComponent c)
    {
        Dimension size = c.getSize();
        g.setColor(c.getBackground());
        g.fillRect(0, 0, c.getWidth(), c.getHeight());
        rendererPane.paintComponent(g, toolTipArea, c,
                                    1, 1,
                                    size.width - 1, size.height - 1,
                                    true);
    }

    public Dimension getMinimumSize(JComponent c)
    {
        return getPreferredSize(c);
    }

    public Dimension getMaximumSize(JComponent c)
    {
        return getPreferredSize(c);
    }

    public Dimension getPreferredSize(JComponent c)
    {
        Dimension dim = new Dimension(0,0);
        String tipText = ((JToolTip)c).getTipText();

        if( tipText != null && dataManager != null )
        {
            Matcher m = positionPattern.matcher( tipText );
            if( m.find() )
            {
                int gal = Integer.parseInt( m.group( 1 ) );
                int sys = Integer.parseInt( m.group( 2 ) );
                int pla = (m.group( 3 ).indexOf('x') > -1) ? 0 : Integer.parseInt( m.group( 3 ) );

                // column titles are not shown
                Vector<String> columnTitles = new Vector<>();
                columnTitles.add( "A" );
                columnTitles.add( "B" );
                Vector<Vector<String>> tableData = null;
                try
                {
                    if(pla == 0) {
                        tableData = getBlackHoleTableData(gal, sys);
                    }
                    else {
                        tableData = getPlanetTableData(gal, sys, pla);
                    }
                }
                catch( SQLException e )
                {
                    e.printStackTrace();
                }

                if(tableData != null) {
                    toolTipArea = new JTable( tableData, columnTitles );
                    toolTipArea.setDefaultRenderer( Object.class, cellRenderer );
                    toolTipArea.setBackground( bgColor );
                    TableColumnModel columnModel = toolTipArea.getColumnModel();
                    TableColumn column;
                    column = columnModel.getColumn( 0 );
                    int width = 105;
                    column.setMinWidth( width );
                    column.setMaxWidth( width );
                    column.setPreferredWidth( width );
                    column = columnModel.getColumn( 1 );
                    width = 160;
                    column.setMinWidth( width );
                    column.setMaxWidth( width );
                    column.setPreferredWidth( width );

                    dim = toolTipArea.getPreferredSize();
                    dim.height += 2;
                    dim.width += 2;
                }
            }
        }
        return dim;
    }

    private Vector<Vector<String>> getPlanetTableData(int gal, int sys, int pla) throws SQLException {
        Vector<Vector<String>> tableData = new Vector<>();
        Vector<String> rowData;
        Planet planet;
        Position position = new Position( gal, sys, pla );

        planet = dataManager.getPlanetHere( position.getId() );
        String temperature = Integer.toString( SystemData.getPlanetData( dataManager.getFlavor(), new Position(gal, sys, pla) ).getTemperature() ) + "°C";
        if( planet == null )
        {
            rowData = new Vector<String>();
            rowData.add( position.toString() );
            rowData.add( dataManager.getI18nText( "GalaxyView.uninhabited" ) );
            tableData.add( rowData );
            rowData = new Vector<String>();
            rowData.add( dataManager.getI18nText( "GalaxyView.temperature" ) );
            rowData.add( temperature );
            tableData.add( rowData );
        }
        else
        {
            rowData = new Vector<String>();
            rowData.add( position.toString() );
            if( planet.getOwner() != null ) {
                rowData.add( planet.getName() );
            }
            else {
                rowData.add( dataManager.getI18nText( "GalaxyView.uninhabited" ) );
            }
            tableData.add( rowData );
            if( planet.getOwner() != null )
            {
                Player owner = planet.getOwner();
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "GalaxyView.owner" ) );
                rowData.add(GalaxyComponent.updateTitle(owner.getTitle(),owner.getName()));
                tableData.add( rowData );
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "GalaxyView.specialization" ) );
                rowData.add( owner.getSpecialization() );
                tableData.add( rowData );
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "GalaxyView.points" ) );
                rowData.add( DataManager.format( owner.getPoints() ) );
                tableData.add( rowData );
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "GalaxyView.spiceValue" ) );
                rowData.add( DataManager.format( owner.getSpiceValue() ) );
                tableData.add( rowData );
            }
            rowData = new Vector<String>();
            rowData.add( dataManager.getI18nText( "GalaxyView.position" ) );
            rowData.add( Position.createFromId( Position.createId( gal, sys, pla ) ).toString() );
            tableData.add( rowData );
            rowData = new Vector<String>();
            rowData.add( dataManager.getI18nText( "GalaxyView.temperature" ) );
            rowData.add( temperature );
            tableData.add( rowData );
            if( planet.getOwner() != null )
            {
                Player owner = planet.getOwner();
                if( owner.getAlliance() != null )
                {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( "GalaxyView.alliance" ) );
                    rowData.add( owner.getAlliance().getName() );
                    tableData.add( rowData );
                }
                if( owner.getStatus() != null && owner.getStatus() != Status.own)
                {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( "GalaxyView.status" ) );
                    rowData.add( dataManager.getI18nText( owner.getStatus().name() ) );
                    tableData.add( rowData );
                }
            }
            Cost debris = planet.getDebris();
            if( debris != null && !debris.isEmpty() )
            {
                rowData = new Vector<String>();
                rowData.add( "" );
                rowData.add( dataManager.getI18nText( "GalaxyView.debris" ) );
                tableData.add( rowData );
                if( debris.getIron() > 0 )
                {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( Resource.iron.name() ) );
                    rowData.add( DataManager.format( debris.getIron() ) );
                    tableData.add( rowData );
                }
                if( debris.getMetal() > 0 )
                {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( Resource.metal.name() ) );
                    rowData.add( DataManager.format( debris.getMetal() ) );
                    tableData.add( rowData );
                }
                if( debris.getKryptonite() > 0 )
                {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( Resource.kryptonite.name() ) );
                    rowData.add( DataManager.format( debris.getKryptonite() ) );
                    tableData.add( rowData );
                }
                if( debris.getSpice() > 0 )
                {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( Resource.spice.name() ) );
                    rowData.add( DataManager.format( debris.getSpice() ) );
                    tableData.add( rowData );
                }
            }
            if( planet.getUpdateTime() > 0 )
            {
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "GalaxyView.lastUpdated" ) );
                rowData.add( DataManager.getDateFormat().format( planet.getUpdateTime() ) );
                tableData.add( rowData );
            }
        }

        return tableData;
    }

    private Vector<Vector<String>> getBlackHoleTableData(int gal, int sys) throws SQLException {
        Vector<Vector<String>> tableData = null;
        BlackHole blackHole = dataManager.getBlackHole(Position.createId(gal, sys, 0));

        if(blackHole != null && blackHole.isSetup()) {
            tableData = new Vector<>();
            Vector<String> rowData;
            if( blackHole.getPower() > 0 ) {
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "BlackHole.power" ) );
                rowData.add( DataManager.format( blackHole.getPower() ) );
                tableData.add( rowData );
            }
            if( blackHole.getShips().size() > 0 ) {
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "BlackHole.ships" ) );
                rowData.add( "" );
                tableData.add( rowData );
                for(Map.Entry<Ship, Long> entry : blackHole.getShips().entrySet()) {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( entry.getKey().name() ) );
                    rowData.add( DataManager.format( entry.getValue() ) );
                    tableData.add( rowData );
                }
            }
            if( blackHole.getStars() + blackHole.getEmblems() + blackHole.getSpice() > 0 ) {
                rowData = new Vector<String>();
                rowData.add( dataManager.getI18nText( "BlackHole.rewards" ) );
                rowData.add( "" );
                tableData.add( rowData );
                if( blackHole.getSpice() > 0 ) {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( Resource.spice.name() ) );
                    rowData.add( DataManager.format( blackHole.getSpice() ) );
                    tableData.add( rowData );
                }
                if( blackHole.getEmblems() > 0 ) {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( "BlackHole.rewards.emblems" ) );
                    rowData.add( DataManager.format( blackHole.getEmblems() ) );
                    tableData.add( rowData );
                }
                if( blackHole.getStars() > 0 ) {
                    rowData = new Vector<String>();
                    rowData.add( dataManager.getI18nText( "BlackHole.rewards.stars" ) );
                    rowData.add( DataManager.format( blackHole.getStars() ) );
                    tableData.add( rowData );
                }
            }
        }

        return tableData;
    }

    protected void setDataManager( DataManager dataManager )
    {
        this.dataManager = dataManager;
    }
}
