/****************************************
 *  COPYRIGHT (C) 2010-2025
 *  Holger Graf
 ****************************************/

package siarchive.galaxy;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.sql.SQLException;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTabbedPane;
import javax.swing.JToolTip;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.border.BevelBorder;

import siarchive.DataManager;
import siarchive.Resources;
import siarchive.components.Building;
import siarchive.components.Defense;
import siarchive.components.Flavor;
import siarchive.components.NoteBox;
import siarchive.components.PlanetData;
import siarchive.components.PlanetarySystem;
import siarchive.components.Position;
import siarchive.components.Ship;
import siarchive.components.Status;
import siarchive.persistence.BattleReport;
import siarchive.persistence.BlackHole;
import siarchive.persistence.Notable;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.persistence.SpyReport;
import siarchive.reports.BattleReportView;
import siarchive.reports.SpyReportParser;
import siarchive.reports.SpyReportView;
import siarchive.search.BattleReportSummaryPane;
import siarchive.search.PlanetSummaryPane;
import siarchive.search.SpyReportSummaryPane;

/**
 * @author graf
 *
 */
public class GalaxyComponent extends JComponent
{
    private static final long serialVersionUID = 4025295210952509026L;

    private static final int toolTipWidth = 40;

    protected GalaxyView parent;
    protected DataManager dataManager;

    private int galaxy = 1;
    private int system = 1;
    private Font baseFont;
    private Font boldFont;
    private Font dialogFont;

    private JMenuItem popupAlliance = new JMenuItem();
    private JMenuItem popupPlayer = new JMenuItem();
    private JMenuItem popupSystem = new JMenuItem();
    private JMenuItem popupSpyReports = new JMenuItem();
    private JMenuItem popupBattleReports = new JMenuItem();
    private Planet menuPlanet = null;

    private Map<Rectangle, SpyReport> spyReports = new LinkedHashMap<Rectangle, SpyReport>();
    private Map<Rectangle, BattleReport> battleReports = new LinkedHashMap<Rectangle, BattleReport>();
    private Map<Rectangle, Notable<Planet>> notes = new LinkedHashMap<Rectangle, Notable<Planet>>();

    // to correct the planet shadow to true vertical
    private static final double corr = 11 * Math.PI / 180;
    private static final int display_x_offset = 20;
    private static final int display_y_offset = 15;


    public GalaxyComponent( final GalaxyView parent )
    {
        super();
        initFont(11.0f);
        this.parent = parent;
        this.dataManager = parent.getDataManager();

        setOpaque( true );

        setBorder( BorderFactory.createBevelBorder( BevelBorder.LOWERED ) );
        setToolTipText( " " );

        this.addKeyListener( new KeyAdapter()
        {
            public void keyPressed( KeyEvent event )
            {
                eventKeyPressed( event );
            }
        } );

        popupAlliance.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowAlliance( e );
            }
        } );
        popupPlayer.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowPlayer( e );
            }
        } );
        popupSystem.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowSystem( e );
            }
        } );
        popupBattleReports.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowBattleReports( e );
            }
        } );
        popupSpyReports.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionShowSpyReports( e );
            }
        } );

        this.addMouseListener( new MouseAdapter()
        {
            public void mouseClicked(MouseEvent e)
            {
                boolean focusRequest = true;
                if( spyReports.size() > 0 )
                {
                    for( Map.Entry<Rectangle, SpyReport> entry : spyReports.entrySet() )
                    {
                        if( entry.getKey().contains( e.getX(), e.getY() ) )
                        {
                            SpyReportView spyReportViewer = new SpyReportView( entry.getValue(), parent.getMainFrame() );
                            spyReportViewer.setVisible( true );
                            focusRequest = false;
                        }
                    }
                }
                if( battleReports.size() > 0 )
                {
                    for( Map.Entry<Rectangle, BattleReport> entry : battleReports.entrySet() )
                    {
                        if( entry.getKey().contains( e.getX(), e.getY() ) )
                        {
                            BattleReportView battleReportViewer = new BattleReportView( entry.getValue(), parent.getMainFrame() );
                            battleReportViewer.setVisible( true );
                            focusRequest = false;
                        }
                    }
                }
                if( notes.size() > 0 )
                {
                    for( Map.Entry<Rectangle, Notable<Planet>> entry : notes.entrySet() )
                    {
                        if( entry.getKey().contains( e.getX(), e.getY() ) )
                        {
                            NoteBox<Planet> noteBox = new NoteBox<Planet>( parent, parent.getMainFrame(), entry.getValue() );
                            noteBox.setVisible( true );
                            focusRequest = false;
                        }
                    }
                }

                if( e.getClickCount() > 1 )
                {
                    Position position = getPosition( e );
                    // hit on planet?
                    if( position != null )
                    {
                        Planet planet = null;
                        try
                        {
                            planet = dataManager.getPlanetHere( position.getId() );
                        }
                        catch( SQLException e1 )
                        {
                            e1.printStackTrace();
                        }
                        if( planet != null )
                        {
                            NoteBox<Planet> noteBox = new NoteBox<Planet>( parent, parent.getMainFrame(), planet );
                            noteBox.setVisible( true );
                            focusRequest = false;
                        }
                    }
                }

                if( focusRequest )
                {
                    GalaxyComponent.this.requestFocusInWindow();
                }
            }

            @Override
            public void mousePressed( MouseEvent event )
            {
                if( event.isPopupTrigger() )
                {
                    showPopup( event );
                }
            }

            @Override
            public void mouseReleased( MouseEvent event )
            {
                if( event.isPopupTrigger() )
                {
                    showPopup( event );
                }
            }

            private void showPopup( MouseEvent event )
            {
                // hit on planet?
                Position position = getPosition( event );
                if( position != null )
                {
                    menuPlanet = null;
                    try
                    {
                        menuPlanet = dataManager.getPlanetHere( position.getId() );
                    }
                    catch( SQLException e )
                    {
                        e.printStackTrace();
                    }
                    if( menuPlanet != null )
                    {
                        JPopupMenu popup = new JPopupMenu();
                        if( menuPlanet.getOwner() != null )
                        {
                            popup.add( popupPlayer );
                            if( menuPlanet.getOwner().getAlliance() != null )
                            {
                                popup.add( popupAlliance );
                            }
                        }
                        popup.add( popupSystem );
                        popup.add( popupSpyReports );
                        popup.add( popupBattleReports );
                        popup.pack();
                        updateLabel();

                        popup.show(event.getComponent(), event.getX(), event.getY() );
                    }
                }
            }
        });
    }

    @Override
    public JToolTip createToolTip()
    {
        return new JToolTip()
        {
            private static final long serialVersionUID = 4237963755955028539L;

            public void updateUI()
            {
                GalaxyComponentToolTipUI ui = GalaxyComponentToolTipUI.createUI( this );
                ui.setDataManager( dataManager );
                setUI( ui );
            }
        };
    }


    @Override
    public String getToolTipText( MouseEvent event )
    {
        String toolTipText = null;
        Position position = getPosition( event );
        if( position != null )
        {
            toolTipText = position.toString();
        }
        return toolTipText;
    }

    private static Dimension galaxySize = new Dimension(805, 630);
    @Override
    public Dimension getPreferredSize() {
        return galaxySize;
    }

    private Position getPosition( MouseEvent event )
    {
        Position position = null;
        int mouseX = event.getX();
        int mouseY = event.getY();
        for( int planet = Position.lowerPlanetBound; planet <= Position.upperPlanetBound; planet++ )
        {
            Position p = new Position(galaxy, system, planet );
            PlanetData data = SystemData.getPlanetData( dataManager.getFlavor(), p );

            int baseX = data.getX() + display_x_offset;
            int baseY = data.getY() + display_y_offset;
            ImageIcon icon = getPlanetIcon(planet);
            int w2 = icon.getIconWidth() / 2;
            int h2 = icon.getIconHeight() / 2;

            if( ( mouseX >= baseX - w2 ) && ( mouseX <= baseX + w2 )
             && ( mouseY >= baseY - h2 ) && ( mouseY <= baseY + h2 )
            )
            {
                position = p;
                break;
            }
        }
        // black hole position?
        if(position == null) {
            Position p = new Position(galaxy, system, 0);
            PlanetData data = SystemData.getSunData( dataManager.getFlavor(), p );
            int baseX = data.getX() + display_x_offset - toolTipWidth / 2;
            int baseY = data.getY() + display_y_offset - 20 - toolTipWidth / 2;
            if( mouseX >= baseX && mouseX <= baseX + toolTipWidth
             && mouseY >= baseY && mouseY <= baseY + toolTipWidth ) {
                position = p;
            }
        }
        return position;
    }

    protected void actionShowSpyReports( ActionEvent e )
    {
        JTabbedPane pane = parent.getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( dataManager.getI18nText( "MainFrame.spyReports" ) );
        if( tabIndex >= 0 )
        {
            SpyReportSummaryPane summaryPane = ( SpyReportSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowSystem( ActionEvent e )
    {
        JTabbedPane pane = parent.getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( dataManager.getI18nText( "MainFrame.planets" ) );
        if( tabIndex >= 0 )
        {
            PlanetSummaryPane summaryPane = ( PlanetSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( Position.createFromId( menuPlanet.getPosition() ) );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowPlayer( ActionEvent e )
    {
        JTabbedPane pane = parent.getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( dataManager.getI18nText( "MainFrame.planets" ) );
        if( tabIndex >= 0 )
        {
            PlanetSummaryPane summaryPane = ( PlanetSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet.getOwner() );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowAlliance( ActionEvent e )
    {
        JTabbedPane pane = parent.getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( dataManager.getI18nText( "MainFrame.planets" ) );
        if( tabIndex >= 0 )
        {
            PlanetSummaryPane summaryPane = ( PlanetSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet.getOwner().getAlliance() );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void actionShowBattleReports( ActionEvent e )
    {
        JTabbedPane pane = parent.getMainFrame().getTabbedPane();
        int tabIndex = pane.indexOfTab( dataManager.getI18nText( "MainFrame.battleReports" ) );
        if( tabIndex >= 0 )
        {
            BattleReportSummaryPane summaryPane = ( BattleReportSummaryPane )pane.getComponentAt( tabIndex );
            summaryPane.actionNew( menuPlanet );
            pane.setSelectedIndex( tabIndex );
        }
    }

    protected void eventKeyPressed( KeyEvent event )
    {
        Position position = parent.getPosition();
        int galaxy = position.getGalaxy();
        int system = position.getSystem();
        switch( event.getKeyCode() )
        {
            case KeyEvent.VK_LEFT:
                if( system > Position.lowerSystemBound )
                {
                    system--;
                }
                break;
            case KeyEvent.VK_RIGHT:
                if( system < Position.getUpperSystemBound(dataManager.getFlavor()) )
                {
                    system++;
                }
                break;
            case KeyEvent.VK_DOWN:
                if( galaxy > Position.getLowerGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()) )
                {
                    galaxy--;
                }
                break;
            case KeyEvent.VK_UP:
                if( galaxy < Position.getUpperGalaxyBound(dataManager.getFlavor(), dataManager.isOutpost()) )
                {
                    galaxy++;
                }
                break;
        }
        if( galaxy != position.getGalaxy() || system != position.getSystem() )
        {
            parent.setSystem( galaxy, system );
        }
    }

    protected void setSystem( int galaxy, int system )
    {
        this.galaxy = galaxy;
        this.system = system;
    }


    private void initFont(float size) {
        UIDefaults uiDefaults = UIManager.getDefaults();
        String key = "Panel.font";
        Font font = (Font)uiDefaults.get( key );
        baseFont = font.deriveFont(size);
        boldFont = baseFont.deriveFont( Font.BOLD );
        dialogFont = Font.decode(null).deriveFont(Font.BOLD, size - 1);
    }

    protected void paintComponent(Graphics g)
    {
        super.paintComponent(g);
        if( dataManager.getActiveAccount() == null )
        {
            return;
        }
        Graphics2D g2 = (Graphics2D)g;
        g2.setRenderingHint(
                        RenderingHints.KEY_INTERPOLATION,
                        RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        g2.setRenderingHint(
                        RenderingHints.KEY_ANTIALIASING,
                        RenderingHints.VALUE_ANTIALIAS_ON);

        // prepare background
        StringBuilder path = new StringBuilder();
        path.append( "/galaxy" );
        if( galaxy < 10 )
        {
            path.append( '0' );

        }
        path.append( galaxy );
        path.append( ".jpg" );
        ImageIcon icon = Resources.getIcon( path.toString() );
        g.drawImage( icon.getImage(), 0, 0, null );

        try
        {
            // draw sun
            drawSun( g2 );
            // draw planets
            drawPlanets( g2 );
            // decorate planets
            decoratePlanets( g2 );
            // update time indicator
            parent.updateLabel();
        }
        catch (SQLException ex) {
            parent.getMainFrame().showErrorInfo( ex );
        }
    }

    private void drawSun( Graphics2D g ) throws SQLException
    {
        // save transform
        AffineTransform t2 = (AffineTransform)g.getTransform().clone();

        Position position = new Position(galaxy, system, 0);
        PlanetData data = SystemData.getSunData( dataManager.getFlavor(), position );
        BlackHole blackHole = dataManager.getBlackHole(position.getId());
        // center transform, the sun appears higher than the coordinates suggest
        int baseX = data.getX() + display_x_offset;
        int baseY = data.getY() + display_y_offset - 20;
        g.translate(baseX, baseY);

        ImageIcon icon;
        String path;
        if(blackHole != null) {
            path = new StringBuilder()
                    .append("/blackhole")
                    .append(blackHole.getLevel())
                    .append(".png")
                    .toString();
        }
        else {
            path = "/sun.png";
        }
        icon = Resources.getIcon( path );
        // image offset upper left corner
        int w2 = icon.getIconWidth() / 2;
        int h2 = icon.getIconHeight() / 2;
        g.drawImage( icon.getImage(), -w2, -h2, null );
        String sun = dataManager.getI18nText( "GalaxyView.sun" );
        g.setColor( Color.white );
        g.setFont( boldFont );
        w2 = g.getFontMetrics().stringWidth(sun) / 2;
        // assume the same height for the sun and the black hole core
        g.drawString( sun, -w2 + 2, 29 );
        // reset transform
        g.setTransform(t2);
    }

    private void drawPlanets( Graphics2D g ) throws SQLException
    {
        // save transform
        AffineTransform t2 = (AffineTransform)g.getTransform().clone();

        Flavor flavor = dataManager.getFlavor();
        PositionSet localSystem;
        if(flavor.hasMissingPlanets()) {
            localSystem = SystemData.getVisiblePlanets(flavor, new Position(galaxy, system, 0));
        }
        else {
            localSystem = new PositionSet(galaxy, system);
        }

        g.setFont(dialogFont);
        for( Position position : localSystem ) {
            int planet = position.getPlanet();
            PlanetData data = SystemData.getPlanetData( flavor, position );

            // center transform
            int baseX = data.getX() + display_x_offset;
            int baseY = data.getY() + display_y_offset;
            g.translate(baseX, baseY);

            ImageIcon icon = getPlanetIcon(planet);
            // image offset upper left corner
            int w2 = icon.getIconWidth() / 2;
            int h2 = icon.getIconHeight() / 2;
            double theta = getRotation(position);
            g.rotate(theta);
            g.drawImage( icon.getImage(), -w2, -h2, null );
            g.rotate(-theta);

            // planet number lower right corner
            g.setColor( Color.white );
            g.drawString( Integer.toString( planet ), w2 + 2, h2 );

            // add symbols for spy and battle reports
            Planet dummy = new Planet();
            dummy.setAccount( dataManager.getActiveAccount().getId() );
            dummy.setPosition( Position.createId( galaxy, system, planet ) );
            Collection<SpyReport> spyReports = dataManager.getSpyReportsForPlanet( dummy );
            if( spyReports.size() > 0 )
            {
                icon = Resources.getIcon( "/scroll.png" );
                // lower left corner
                g.drawImage( icon.getImage(), -w2 - 8, h2 - 10, null );
                SpyReport spyReport = spyReports.iterator().next();
                // save for popup
                this.spyReports.put( new Rectangle( baseX - w2 - 8, baseY + h2 - 10, 12, 12 ), spyReport );
                Map<?, Long> scanResults  = SpyReportParser.scanSpyReport( spyReport, dataManager );
                if( hasFleet( scanResults ) )
                {
                    // upper right corner, beside debris
                    icon = Resources.getIcon( "/fleet.png" );
                    g.drawImage( icon.getImage(), w2 + 10, -h2, null );
                }
                if( hasDefense( scanResults ) )
                {
                    g.setColor( Color.white );
                    g.drawOval( -w2 - 2 , -h2 - 3, 2 * w2 + 4, 2* h2 + 4 );
                }
                switch( industryLevel( scanResults ) )
                {
                    case 0:
                    case 1:
                        break;
                    case 2:
                        icon = Resources.getIcon( "/silver.png" );
                        g.drawImage( icon.getImage(), -icon.getIconWidth() / 2, -icon.getIconHeight() / 2, null );
                        break;
                    case 3:
                    default:
                        icon = Resources.getIcon( "/gold.png" );
                        g.drawImage( icon.getImage(), -icon.getIconWidth() / 2, -icon.getIconHeight() / 2, null );
                        break;
                }
            }
            Collection<BattleReport> battleReports = dataManager.getBattleReportsForPlanet( dummy );
            if( battleReports.size() > 0 )
            {
                // lower left corner, beside spy report
                icon = Resources.getIcon( "/battle.png" );
                g.drawImage( icon.getImage(), -w2 + 4, h2 - 10, null );
                // save for popup
                this.battleReports.put( new Rectangle( baseX - w2 + 4, baseY + h2 - 10, 12, 12 ), battleReports.iterator().next() );
            }
            // reset transform
            g.setTransform(t2);
        }
    }

    // while drawPlanets() draws all planets that can possibly exist in the current system,
    // decoratePlanets() will only decorate the planets that actually exist in the database
    private void decoratePlanets( Graphics2D g ) throws SQLException
    {
        // save transform
        AffineTransform t2 = (AffineTransform)g.getTransform().clone();
        Flavor flavor = dataManager.getFlavor();
        PlanetarySystem system = dataManager.getSystem( this.galaxy, this.system );
        if( system != null ) {
            long now = System.currentTimeMillis();
            Collection<Planet> planets = system.getPlanets();
            Iterator<Planet> it = planets.iterator();
            FontMetrics metrics = g.getFontMetrics( baseFont );
            FontMetrics boldMetrics = g.getFontMetrics( boldFont );
            while( it.hasNext() ) {
                Planet planet = it.next();
                int pl = planet.getPosition() % 100;
                PlanetData data = SystemData.getPlanetData( flavor, new Position( this.galaxy, this.system, pl ) );
                int baseX = data.getX() + display_x_offset;
                int baseY = data.getY() + display_y_offset;
                // center transform
                g.translate(baseX, baseY);
                ImageIcon icon = getPlanetIcon(pl);
                // image offset upper left corner
                int w2 = icon.getIconWidth() / 2;
                int h2 = icon.getIconHeight() / 2;

                //overlay old planets
                if(planet.isEvent()) {
                    icon = Resources.getIcon( "/planet_event.png" );
                    g.drawImage( icon.getImage(), -icon.getIconWidth() / 2, -icon.getIconHeight() / 2, null );
                }
                if( planet.getUpdateTime() < system.getHighTime() )
                {
                    long updateInterval = now - planet.getUpdateTime();
                    Color base = GalaxyOverview.getColor( updateInterval );
                    Color color = new Color( base.getRed(), base.getGreen(), base.getBlue(), 75 );
                    g.setColor( color );
                    g.fillOval( -w2 - 4, -h2 -4 , 2* w2 + 4, 2 * h2 +4 );
                }

                Player owner = planet.getOwner();
                Status status = Status.none;
                if( owner != null && owner.getStatus() != null ) {
                    status = owner.getStatus();
                }
                if( planet.hasAsteroid() ) {
                    decorateAsteroid(g, planet, baseX, baseY, w2, h2, status);
                }
                if( planet.getDebris() != null ) {
                    icon = Resources.getIcon( "/debris.png" );
                    // upper right
                    g.drawImage( icon.getImage(), w2 / 2, -h2 - 5, null );
                }
                if( planet.getNotes() != null && planet.getNotes().getText().trim().length() > 0 ) {
                    icon = Resources.getIcon( "/notes-small.png" );
                    g.drawImage( icon.getImage(), -w2 + 16, h2 - 10, null );
                    // save for popup
                    this.notes.put( new Rectangle( baseX - w2 + 16, baseY + h2 - 10, 12, 12 ), planet );
                }

                int posX = -w2 - 2;
                int posY = h2 + 12;
                if( owner != null ) {
                    // set color according to status
                    Color statusColor = status.getColor();
                    // extra consideration for 'locked' players
                    if( status == Status.locked )
                    {
                        Status secondaryStatus = owner.getSecondaryStatus();
                        if( secondaryStatus != null && secondaryStatus != Status.none ) {
                            statusColor = secondaryStatus.getColor();
                        }
                    }
                    g.setColor( statusColor );
                    // planet name is bold
                    g.setFont( boldFont );
                    g.drawString( planet.getName(), posX, posY );
                    int height = boldMetrics.getMaxAscent() + boldMetrics.getMaxDescent() + 1;
                    posY += height;
                    // name is in white parentheses
                    g.setColor( Color.white );
                    g.setFont( baseFont );
                    g.drawString( "(", posX, posY );
                    g.setColor( statusColor );
                    posX += metrics.stringWidth( "(" );
                    //g.setFont( boldFont );
                    final String title = updateTitle(owner.getTitle(), owner.getName());
                    g.drawString( title, posX, posY );
                    posX += metrics.stringWidth( title );
                    g.setColor( Color.white );
                    //g.setFont( baseFont );
                    g.drawString( ")", posX, posY );
                    // add points in normal font
                    g.setColor( statusColor );
                    posX = -w2;
                    posY += height - 2;
                    g.drawString( DataManager.format( owner.getPoints() ), posX, posY );
                }
                // reset transform
                g.setTransform(t2);
            }
        }
    }

    private void decorateAsteroid( Graphics2D g, Planet planet, int baseX, int baseY, int w2, int h2, Status status ) throws SQLException {
        // save transform
        AffineTransform t2 = (AffineTransform)g.getTransform().clone();

        ImageIcon icon = Resources.getIcon( "/moon.png" );
        int astroW = icon.getIconWidth();
        int astroH = icon.getIconHeight();
        // upper left corner
        int astroX = -astroW / 2;
        int astroY = -astroH / 2;

        // re-center on astro
        g.translate(-w2, -h2);
        double theta = getRotation(Position.createFromId(planet.getPosition()));
        g.rotate(theta);
        g.drawImage( icon.getImage(), astroX, astroY, null );
        g.rotate(-theta);
        // notify unfriendly asteroids
        switch( status )
        {
            case enemy:
            case noob:
            case strong:
                g.setColor( Status.enemy.getColor() );
                g.drawOval( astroX - 2, astroY - 2, astroW + 1, astroH + 1 );
            default:
                break;
        }
        Collection<SpyReport> spyReports = dataManager.getSpyReportsForAsteroid( planet );
        if( spyReports.size() > 0 ) {
            icon = Resources.getIcon( "/scroll.png" );
            g.drawImage( icon.getImage(), astroX - 10, astroY + 6, null );
            SpyReport spyReport = spyReports.iterator().next();
            // save for popup in absolute position
            this.spyReports.put( new Rectangle( baseX + astroX - w2 - 11,  baseY + astroY - h2 + 5, 13, 13 ), spyReport );

            Map<?, Long> scanResults  = SpyReportParser.scanSpyReport( spyReport, dataManager );
            if( hasFleet( scanResults ) ) {
                icon = Resources.getIcon( "/fleet.png" );
                g.drawImage( icon.getImage(), astroX + 10, astroY - 8, null );
            }
            if( hasDefense( scanResults ) ) {
                g.setColor( Color.white );
                g.drawOval( astroX - 2, astroY - 2, astroW + 5, astroH + 5 );
            }
        }
        Collection<BattleReport> battleReports = dataManager.getBattleReportsForAsteroid(planet);
        if( battleReports.size() > 0 )
        {
            // lower left corner, beside spy report
            icon = Resources.getIcon( "/battle.png" );
            g.drawImage( icon.getImage(), astroX + 2, astroY + 6, null );
            // save for popup
            this.battleReports.put( new Rectangle( baseX + astroX - w2 + 1,  baseY + astroY - h2 + 5, 13, 13 ), battleReports.iterator().next() );
        }
        // reset transform
        g.setTransform(t2);
    }

    private ImageIcon getPlanetIcon(int planet) {
        StringBuilder path = new StringBuilder();
        path.append( "/planet" );
        if( planet < 10 )
        {
            path.append( '0' );

        }
        path.append( planet );
        path.append( ".png" );
        return Resources.getIcon( path.toString() );
    }

    private double getRotation( Position position ) {
        PlanetData sun = SystemData.getSunData( dataManager.getFlavor(), position );
        PlanetData planet = SystemData.getPlanetData( dataManager.getFlavor(), position );
        // shadow left
        double dx = sun.getX() - planet.getX();
        // inverted y axis
        double dy = sun.getY() - planet.getY();
        double theta = Math.atan2(dy, dx);
        return theta + corr;
    }

    private boolean hasFleet( Map<?, Long> scanResults )
    {
        boolean rv = false;

        for( Ship ship : Ship.ships() )
        {
            if( scanResults.containsKey( ship ) )
            {
                rv = true;
                break;
            }
        }

        return rv;
    }

    private boolean hasDefense( Map<?, Long> scanResults )
    {
        boolean rv = false;

        for( Defense defense : Defense.values() )
        {
            if( scanResults.containsKey( defense ) )
            {
                rv = true;
                break;
            }
        }

        return rv;
    }

    private int industryLevel( Map<?, Long> scanResults )
    {
        int rv = 0;

        for( Building building : EnumSet.range( Building.ironMine, Building.spiceMine ) )
        {
            Long level = scanResults.get( building );
            if( level != null )
            {
                int value = (level.intValue() - 1) / 10;
                if( rv < value)
                {
                    rv = value;
                }
            }
        }
        return rv;
    }

    private void updateLabel()
    {
        popupAlliance.setText( dataManager.getI18nText( "Popup.showAlliance" ) );
        popupPlayer.setText( dataManager.getI18nText( "Popup.showPlayer" ) );
        popupSystem.setText( dataManager.getI18nText( "Popup.showSystem" ) );
        popupSpyReports.setText( dataManager.getI18nText( "Popup.spyReports" ) );
        popupBattleReports.setText( dataManager.getI18nText( "Popup.battleReports" ) );
    }

    static String updateTitle(String title, String name) {
        if(title.isEmpty() || title.contains("<div")) {
            title = "%name%";
        }
        return title.replace("%name%", name);
    }

    public void update() {
        revalidate();
        repaint();
    }


}
