/****************************************
 *  COPYRIGHT (C) 2021, 2023-2024
 *  Holger Graf
 ****************************************/
package siarchive.datatransfer;

import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import siarchive.DataManager;
import siarchive.components.Cost;
import siarchive.components.Position;
import siarchive.components.Ship;
import siarchive.components.Status;
import siarchive.imports.DateParser;
import siarchive.json.PlanetType;
import siarchive.json.battlereport.BattleReportBean;
import siarchive.json.galaxy.BlackHoleBean;
import siarchive.json.galaxy.GalaxyBean;
import siarchive.json.galaxy.PlanetBean;
import siarchive.json.galaxy.RewardsBean;
import siarchive.json.galaxy.RubbleFieldBean;
import siarchive.json.galaxy.TaskBean;
import siarchive.json.galaxy.UserBean;
import siarchive.json.spyreport.SpyReportBean;
import siarchive.persistence.Alliance;
import siarchive.persistence.BattleReport;
import siarchive.persistence.BlackHole;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.persistence.SpyReport;

public class JsonScanner {

    public JsonScanner() {}

    public static void scanBattleReportJSON( BattleReport report, BattleReportBean reportBean, DataManager dataManager ) {
        long date = DateParser.parse(reportBean.getInsertDate());
        report.setCreationTime(date);
        String attackerNames = reportBean.getAllianceAttackMember();
        if(attackerNames == null || attackerNames.length() == 0) {
            attackerNames = reportBean.getAttacker().getNickname();
        }
        report.setAttacker(attackerNames);
        report.setDefender(reportBean.getDefender().getNickname());
        report.setPosition(reportBean.getDestinationPlanet().getCoords());
        report.setPlanetName(reportBean.getDestinationPlanet().getName());

        report.setAsteroid(reportBean.getDestinationPlanet().getType() == PlanetType.MOON);
    }

    public static void scanSpyReportJSON( SpyReport report, SpyReportBean reportBean, DataManager dataManager ) {
        long date = DateParser.parse(reportBean.getInsertDate());
        report.setCreationTime(date);
        report.setPosition(reportBean.getPlanet().getCoords());
        report.setPlanetName(reportBean.getPlanet().getName());
        report.setAsteroid(reportBean.getPlanet().getType() == PlanetType.MOON);
    }

    public static Position scanGalaxyJSON( GalaxyBean galaxyBean, DataManager dataManager ) throws SQLException {
        int galaxy = galaxyBean.getPosition().getGalaxy();
        int system = galaxyBean.getPosition().getSystem();

        if(!(dataManager.isOutpost() ^ galaxy == dataManager.getFlavor().getOutpostGalaxy())) {
            // collect old data for orphan check
            Set<Player> owners = new HashSet<Player>();
            Set<Alliance> alliances = new HashSet<Alliance>();
            if( dataManager.getSystem( galaxy, system ) != null ) {
                for( Planet planet : dataManager.getSystem( galaxy, system ).getPlanets() ) {
                    Player owner = planet.getOwner();
                    if( owner != null ) {
                        owners.add( owner );
                        Alliance alliance = owner.getAlliance();
                        if( alliance != null ) {
                            alliances.add( alliance );
                        }
                    }
                }
            }

            long timeStamp = galaxyBean.getInfo().getTs();
            // save black hole
            saveBlackHole( dataManager, galaxy, system, galaxyBean.getHole(), timeStamp);
            // get recognized planets and remove from oldPlanets
            savePlanets( dataManager, galaxyBean.getPlanets().values(), timeStamp );

            // remove orphans
            dataManager.removeOrphanedPlayers( owners );
            dataManager.removeOrphanedAlliances(alliances);
        }
        else {
            galaxy = 0;
            system = 0;
        }
        return new Position(galaxy, system, 1);
    }

    private static void saveBlackHole( DataManager dataManager, int galaxy, int system, BlackHoleBean blackHoleBean, long timeStamp ) throws SQLException {
        final int position = Position.createId(galaxy, system, 0);
        if(blackHoleBean != null) {
            BlackHole blackHole = new BlackHole();
            blackHole.setAccount(dataManager.getActiveAccountId());
            blackHole.setPosition(position);
            blackHole.setUpdateTime(timeStamp);
            blackHole.setSetup(blackHoleBean.isSetup());
            blackHole.setLevel(blackHoleBean.getLevel());
            blackHole.setAttacked(blackHoleBean.isAttacked());
            blackHole.setUpgradeTime(blackHoleBean.getUpgrade());
            RewardsBean rewards = blackHoleBean.getRewards();
            if(rewards != null) {
                blackHole.setSpice(rewards.getSpice());
                blackHole.setStars(rewards.getStars());
                blackHole.setEmblems(rewards.getEmblems());
            }
            TaskBean tasks = blackHoleBean.getTasks();
            if(tasks != null) {
                blackHole.setPower(tasks.getPower());
                Map<Ship, Long> ships = new HashMap<>();
                for(Map.Entry<String, Long> entry : tasks.getShips().entrySet()) {
                    Ship ship = Ship.fromId(entry.getKey());
                    ships.put(ship, entry.getValue());
                }
                blackHole.setShips(ships);
            }
            dataManager.save(blackHole);
        }
        else {
            BlackHole previous = dataManager.getBlackHole(position);
            if(previous != null) {
                dataManager.delete(previous);
            }
        }
    }

    private static void savePlanets( DataManager dataManager, Collection<PlanetBean> planets, long timeStamp ) throws SQLException {
        for(PlanetBean planetBean : planets) {
            final UserBean userBean = planetBean.getUser();
            final int position = Position.createId(planetBean.getCoords());
            final Long account = dataManager.getActiveAccountId();
            Planet planet = new Planet();
            planet.setAccount(account);
            planet.setPosition(position);
            planet.setEvent(planetBean.isIsEventPlanet());
            RubbleFieldBean rubble = planetBean.getRubblefield();
            if(rubble != null ) {
                planet.setDebris(new Cost(rubble.getIron(), rubble.getMetal(), rubble.getKryptonite(), rubble.getSpice()));
            }
            planet.setUpdateTime( timeStamp );
            if(userBean != null) {
                planet.setName(planetBean.getName());
                planet.setAsteroid(planetBean.isHasAstro());
                // do not overwrite existing notes
                planet.setRetainNotes( true );

                Player owner = new Player();
                owner.setAccount( account );
                owner.setName(userBean.getNickname());
                owner.setTitle(userBean.getTitle());
                owner.setPlayerId(userBean.getId());
                // admins don't have ranking
                owner.setSpecialization(userBean.getSpec().getSpecialization());
                owner.setPoints(userBean.getRanking());
                owner.setSpiceValue(userBean.getSpice());
                owner.setStatus(convertStatus(userBean.getStatus()));
                owner.setSecondaryStatus((userBean.getStatus().getSecondary() != null)
                    ? convertStatus(userBean.getStatus().getSecondary())
                    : Status.none);

                // add alliance
                Alliance alliance = new Alliance();
                alliance.setAccount( account );
                // the empty alliance will get ID 0
                alliance.setAllianceId( 0 );
                alliance.setName( "" );
                alliance.setUpdateTime( timeStamp );
                // do not overwrite existing notes
                alliance.setRetainNotes( true );
                if(userBean.getAlliance() != null) {
                    alliance.setAllianceId(userBean.getAlliance().getId());
                    alliance.setName(userBean.getAlliance().getName());
                    alliance.setTag(userBean.getAlliance().getTag());
                }
                // persist intermediate objects to add persistent references to planet
                // so they won't be doubled after restart
                alliance = dataManager.save( alliance );
                owner.setAlliance( alliance );
                owner.setUpdateTime( timeStamp );
                // do not overwrite existing notes
                owner.setRetainNotes( true );
                // persist intermediate objects to add persistent references to planet
                // so they won't be doubled after restart
                owner = dataManager.save( owner );

                planet.setOwner( owner );
                // save recognized planets
                dataManager.save( planet);
            }
            else {
                if(planet.getDebris() != null || planet.isEvent()) {
                    dataManager.save( planet);
                }
                else {
                    planet = dataManager.getPlanetHere(position);
                    if(planet != null) {
                        dataManager.delete(planet);
                    }
                }
            }
        }
    }

    private static siarchive.components.Status convertStatus(siarchive.json.galaxy.Status statusJson) {
        siarchive.components.Status status = siarchive.components.Status.none;;
        switch(statusJson) {
            case locked:
                status = siarchive.components.Status.locked;
                break;
            case admin:
                status = siarchive.components.Status.operator;
                break;
            case friendly:
                status = siarchive.components.Status.friendly;
                break;
            case vacationMode:
                status = siarchive.components.Status.vacation;
                break;
            case inactive:
                status = siarchive.components.Status.inactive;
                break;
            case enemy:
                status = siarchive.components.Status.enemy;
                break;
            case bnd:
                status = siarchive.components.Status.bnd;
                break;
            case nap:
                status = siarchive.components.Status.nap;
                break;
            case wing:
                status = siarchive.components.Status.wing;
                break;
            case noob:
                status = siarchive.components.Status.noob;
                break;
            case strong:
                status = siarchive.components.Status.strong;
                break;
            case own:
                status = siarchive.components.Status.own;
                break;
            case spectral:
                status = siarchive.components.Status.spectral;
                break;
            case noUser:
            default:
                status = siarchive.components.Status.none;
                break;
        }
        return status;
    }
}
