/****************************************
 *  COPYRIGHT (C) 2021
 *  Holger Graf
 ****************************************/
package siarchive.datatransfer;

import java.io.IOException;
import java.sql.SQLException;
import javax.swing.SwingUtilities;

import com.fasterxml.jackson.jr.ob.JSON;

import datatransfer.RequestType;
import datatransfer.json.ResponseBean;
import siarchive.AboutBox;
import siarchive.DataManager;
import siarchive.MainFrame;
import siarchive.components.Position;
import siarchive.json.battlereport.BattleReportBean;
import siarchive.json.galaxy.GalaxyBean;
import siarchive.json.spyreport.SpyReportBean;
import siarchive.persistence.Account;
import siarchive.persistence.BattleReport;
import siarchive.persistence.SpyReport;

/**
 * @author graf
 *
 */
public class DataServer extends datatransfer.DataServer {

    private final MainFrame parent;
    // this is the minimum time (in ms) for the data indicator to visibly blink
    private final long minBlink = 35L;

    public DataServer(final MainFrame parent) {
        this.parent = parent;
    }

    @Override
    protected void updateResponseBean(ResponseBean responseBean, RequestType type, String json) throws IOException  {
        // call base to process reequest types we are not interested in
        super.updateResponseBean(responseBean, type, json);
        switch(type) {
            case battlereport:
                {
                    BattleReportBean reportBean = JSON.std.beanFrom(BattleReportBean.class, json);
                    if(checkAccess(reportBean.getInfo())) {
                        importData(reportBean, json);
                    }
                }
                break;
            case galaxyOverview:
                {
                    GalaxyBean galaxyBean = JSON.std.beanFrom(GalaxyBean.class, json);
                    if(checkAccess(galaxyBean.getInfo())) {
                        importData(galaxyBean);
                    }
                }
                break;
            case spyreport:
                {
                    SpyReportBean reportBean = JSON.std.beanFrom(SpyReportBean.class, json);
                    if(checkAccess(reportBean.getInfo())) {
                        importData(reportBean, json);
                    }
                }
                break;
            default:
                break;
        }
    }

    private void importData(final BattleReportBean reportBean, final String input) {
        // move to the  event-dispatch thread
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                long s = System.currentTimeMillis();
                parent.dbOn();
                DataManager dataManager = parent.getDataManager();
                try {
                    final BattleReport report = new BattleReport();
                    report.setData( input );
                    report.setRetainNotes(true);
                    report.setAccount( dataManager.getActiveAccountId() );
                    JsonScanner.scanBattleReportJSON(report, reportBean, dataManager);
                    dataManager.save( report );
                    dataManager.commit();
                    if( parent.isVisibleGalaxyView() ) {
                        parent.updatePane();
                    }
                }
                catch( Exception ex ) {
                    try {
                        dataManager.rollback();
                    }
                    catch( SQLException e ) {
                        e.printStackTrace();
                    }
                    parent.showErrorInfo( ex );
                }
                long diff = System.currentTimeMillis() - s;
                if(diff < minBlink) {
                    try { Thread.sleep(minBlink - diff); } catch(InterruptedException e) {}
                }
                parent.dbOff();
            }
        });
    }

    private void importData(final SpyReportBean reportBean, final String input) {
        // move to the  event-dispatch thread
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                long s = System.currentTimeMillis();
                parent.dbOn();
                DataManager dataManager = parent.getDataManager();
                try {
                    final SpyReport report = new SpyReport();
                    report.setData( input );
                    report.setRetainNotes(true);
                    report.setAccount( dataManager.getActiveAccountId() );
                    JsonScanner.scanSpyReportJSON(report, reportBean, dataManager);
                    dataManager.save( report );
                    dataManager.commit();
                    if( parent.isVisibleGalaxyView() ) {
                        parent.updatePane();
                    }
                }
                catch( Exception ex ) {
                    try {
                        dataManager.rollback();
                    }
                    catch( SQLException e ) {
                        e.printStackTrace();
                    }
                    parent.showErrorInfo( ex );
                }
                long diff = System.currentTimeMillis() - s;
                if(diff < minBlink) {
                    try { Thread.sleep(minBlink - diff); } catch(InterruptedException e) {}
                }
                parent.dbOff();
            }
        });
    }

    private void importData(final GalaxyBean galaxyBean) {
        // move to the  event-dispatch thread
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                long s = System.currentTimeMillis();
                parent.dbOn();
                DataManager dataManager = parent.getDataManager();
                try {
                    Position position = JsonScanner.scanGalaxyJSON(galaxyBean, dataManager);
                    dataManager.commit();
                    if( position.isRealPosition(dataManager.getFlavor()) ) {
                        parent.showGalaxyView( position.getGalaxy(), position.getSystem(), parent.isVisibleGalaxyView() );
                    }
                }
                catch( Exception ex ) {
                    try {
                        dataManager.rollback();
                    }
                    catch( SQLException e ) {
                        e.printStackTrace();
                    }
                    parent.showErrorInfo( ex );
                }
                long diff = System.currentTimeMillis() - s;
                if(diff < minBlink) {
                    try { Thread.sleep(minBlink - diff); } catch(InterruptedException e) {}
                }
                parent.dbOff();
            }
        });
    }

    @Override
    protected Integer getAccountVersionID() {
        Account account = parent.getDataManager().getActiveAccount();
        return account != null ? account.getFlavor().getId() : null;
    }

    @Override
    protected String getIdentifier() {
        return AboutBox.getIdentifier();
    }

    @Override
    protected String getProductName() {
        return AboutBox.getProductName();
    }

    @Override
    protected String getProductVersion() {
        return AboutBox.getVersion().toShortString();
    }

}
