/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/
package siarchive.components;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Position implements Comparable<Position>
{
    public final static int lowerPlanetBound = 1;
    public final static int lowerSystemBound = 1;
    private final static int lowerGalaxyBound = 1;
    public final static int upperPlanetBound = 16;
    private static Pattern positionPattern = Pattern.compile( "(\\d+):(\\d+):(\\d+)" );

    private int id;

    private Position( int id )
    {
        this.id = id;
    }

    public Position( int galaxy, int system, int planet )
    {
        this( createId(galaxy, system, planet) );
    }

    public Position( Position that )
    {
        this.id = that.id;
    }

    public static int createId( int galaxy, int system, int planet )
    {
        return 100 * (1000 * galaxy + system ) + planet;
    }

    public static int createId( String coords )
    {
        Matcher m = positionPattern.matcher( coords );
        int position = 0;
        if( m.find() )
        {
            int galaxy = Integer.parseInt( m.group( 1 ) );
            int system = Integer.parseInt( m.group( 2 ) );
            int planet = Integer.parseInt( m.group( 3 ) );
            position = Position.createId(galaxy, system, planet);
        }
        return position;
    }

    public static Position createFromId( int id )
    {
        return new Position( id );
    }

    public static Position createSystemFromId( int id )
    {
        return new Position( 100 * ( id / 100 ) );
    }

    public static int getLowerGalaxyBound(final Flavor flavor, final boolean isOutpost) {
        return (isOutpost) ? flavor.getOutpostGalaxy() : lowerGalaxyBound;
    }

    public static int getUpperGalaxyBound(final Flavor flavor, final boolean isOutpost) {
        return (isOutpost) ? flavor.getOutpostGalaxy() : flavor.getUpperGalaxyBound();
    }

    public static int getUpperSystemBound(final Flavor flavor) {
        return flavor.getUpperSystemBound();
    }

    public int getGalaxy()
    {
        return id / 100000;
    }

    public int getSystem()
    {
        return ( id % 100000 ) / 100;
    }

    public int getPlanet()
    {
        return id % 100;
    }

    public int getId()
    {
        return id;
    }

    public boolean isRealPosition(Flavor flavor)
    {
        boolean rv;
        boolean isOutpost = getGalaxy() == flavor.getOutpostGalaxy();

        rv = ( getGalaxy() >= getLowerGalaxyBound(flavor, isOutpost) ) && ( getGalaxy() <= getUpperGalaxyBound(flavor, isOutpost) );
        rv &= ( getSystem() >= lowerSystemBound ) && ( getSystem() <= getUpperSystemBound(flavor) );
        rv &= ( getPlanet() >= lowerPlanetBound ) && ( getPlanet() <= upperPlanetBound );

        return rv;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals( Object obj )
    {
        boolean rv = false;

        if( obj instanceof Position )
        {
            Position that = ( Position )obj;
            rv = ( this.id == that.id );
        }
        return rv;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode()
    {
        return id;
    }

    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        if( id == 0 )
        {
            return sb.toString();
        }
        sb.append( '[' );
        sb.append( pad( getGalaxy() ) );
        sb.append( ':' );
        sb.append( pad( getSystem() ) );
        sb.append( ':' );
        sb.append( pad( getPlanet() ) );
        sb.append( ']' );
        return sb.toString();
    }

    private String pad( int value )
    {
        StringBuilder sb = new StringBuilder();
        if( value != 0 )
        {
            if( value < 10 )
            {
                sb.append( '0' );
            }
            sb.append( value );
        }
        else
        {
            sb.append( 'x' );
        }
        return sb.toString();
    }

    public int compareTo(Position that)
    {
        return Long.signum(this.id - that.id);
    }

    //	public static void main( String args[] )
    //	{
    //		for( int gal = 0; gal < 17; gal += 2 )
    //		{
    //			for( int sys = 0; sys < 500; sys += 80)
    //			{
    //				for( int pla = 0; pla < 20; pla += 3 )
    //				{
    //					Position a = new Position( gal, sys, pla );
    //					System.out.print( a ); System.out.println( "  " + a.getId() + " " + a.isRealPosition() );
    //				}
    //			}
    //		}
    //	}
}
