/****************************************
 *  COPYRIGHT (C) 2010
 *  Holger Graf
 ****************************************/
package siarchive.components;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.sql.SQLException;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.border.EtchedBorder;

import siarchive.MainFrame;
import siarchive.persistence.DbObject;
import siarchive.persistence.Notable;
import siarchive.persistence.Note;

/**
 * @author graf
 *
 */
public class NoteBox<T extends DbObject<T>> extends JDialog
{
    private static final long serialVersionUID = 2998642433374222845L;

    private MainFrame mainFrame;
    private Notable<T> notable;
    private JTextArea textArea;
    
    public NoteBox( MainFrame mainFrame, Notable<T> notable )
    {
        this( mainFrame, mainFrame, notable );
    }
    
    /**
     * @param mainFrame
     * @param notable
     */
    public NoteBox( Frame parent, MainFrame mainFrame, Notable<T> notable )
    {
        super( parent );
        this.mainFrame = mainFrame;
        this.notable = notable;

        enableEvents( AWTEvent.WINDOW_EVENT_MASK );
        StringBuilder title = new StringBuilder();
        title.append( mainFrame.getDataManager().getI18nText( "NoteBox.title" ) );
        title.append( ' ' );
        title.append( notable.getName() );
        this.setTitle( title.toString() );
        JPanel contentPane = ( JPanel )getContentPane();
        contentPane.setLayout( new BorderLayout() );
        contentPane.setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );
        
        textArea = new JTextArea();
        textArea.setPreferredSize( new Dimension( 400, 320 ) );
        textArea.setLineWrap( true );
        textArea.setWrapStyleWord( true );
        textArea.setText( ( notable.getNotes() != null ) ? notable.getNotes().getText() : "" );
        
        JPanel lower = new JPanel( new FlowLayout( FlowLayout.CENTER, 4, 4 ) );
        JButton button = new JButton( mainFrame.getDataManager().getI18nText( "OK" ) );
        button.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                close();
            }
        } );
        lower.add( button );
        getRootPane().setDefaultButton( button );
        
        contentPane.add( textArea, BorderLayout.CENTER );
        contentPane.add( lower, BorderLayout.SOUTH );
        pack();
        
        // by popular demand, add paste clipboard to RMB
        MouseAdapter l = new MouseAdapter()
        {

            @Override
            public void mouseClicked( MouseEvent e )
            {
                if( SwingUtilities.isRightMouseButton( e ) )
                {
                    String input = "";
                    Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                    Transferable contents = clipboard.getContents( null );
                    boolean hasTransferableText = ( ( contents != null )
                                    && contents.isDataFlavorSupported( DataFlavor.stringFlavor ) );
                    if( hasTransferableText )
                    {
                        try
                        {
                            input = ( String )contents.getTransferData( DataFlavor.stringFlavor );
                        }
                        catch( UnsupportedFlavorException ex )
                        {
                            System.out.println( ex );
                            ex.printStackTrace();
                        }
                        catch( IOException ex )
                        {
                            System.out.println( ex );
                            ex.printStackTrace();
                        }
                    }
                    textArea.setText( input );
                }
            }

        };
        
        textArea.addMouseListener( l );
    }
    
    @SuppressWarnings("unchecked")
	protected void close()
    {
        String text = textArea.getText().trim();
        Note notes = notable.getNotes();
        if( text.length() > 0 )
        {
            if( notes == null )
            {
                notes = new Note();
            }
            notes.setText( text );
        }
        else
        {
            if( notes != null )
            {
                notes.setText( "" );
            }
        }
        notable.setNotes( notes );
        notable.setRetainNotes( false );
        try
        {
            mainFrame.getDataManager().save( (T)notable );
            mainFrame.getDataManager().commit();
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        
        dispose();
    }

    protected void processWindowEvent( WindowEvent e )
    {
        if( e.getID() == WindowEvent.WINDOW_CLOSING )
        {
            dispose();
        }
        super.processWindowEvent( e );
    }

}
