/****************************************
 *  COPYRIGHT (C) 2012-2021
 *  Holger Graf
 ****************************************/
package siarchive.components;

import siarchive.DataManager;

public class Cost implements Comparable<Cost>
{
    private long iron;
    private long metal;
    private long kryptonite;
    private long spice;

    public Cost( long iron, long metal, long kryptonite, long spice )
    {
        super();
        this.iron = iron;
        this.metal = metal;
        this.kryptonite = kryptonite;
        this.spice = spice;
    }

    public Cost( Cost other )
    {
        super();
        this.iron = other.iron;
        this.metal = other.metal;
        this.kryptonite = other.kryptonite;
        this.spice = other.spice;
    }

    public long getIron()
    {
        return iron;
    }

    public long getMetal()
    {
        return metal;
    }

    public long getKryptonite()
    {
        return kryptonite;
    }

    public long getSpice()
    {
        return spice;
    }

    public boolean isOverflow()
    {
        return ( iron < 0 || metal < 0 || kryptonite < 0 || spice < 0 );
    }

    public boolean isEmpty()
    {
        return ( iron == 0 && metal == 0 && kryptonite == 0 && spice == 0 );
    }

    public long getResource( Resource resource )
    {
        long rv = 0;
        if( resource != null )
        {
            switch( resource )
            {
                case iron:
                    rv = getIron();
                    break;
                case metal:
                    rv = getMetal();
                    break;
                case kryptonite:
                    rv = getKryptonite();
                    break;
                case spice:
                    rv = getSpice();
                    break;
                default:
                    break;
            }
        }

        return rv;
    }

    public void setResource( Resource resource, long value )
    {
        if( resource != null )
        {
            switch( resource )
            {
                case iron:
                    iron = value;
                    break;
                case metal:
                    metal = value;
                    break;
                case kryptonite:
                    kryptonite = value;
                    break;
                case spice:
                    spice = value;
                    break;
                default:
                    break;
            }
        }
    }

    public Cost add( Cost other )
    {
        iron += other.iron;
        metal += other.metal;
        kryptonite += other.kryptonite;
        spice += other.spice;
        return this;
    }

    public Cost multiply( long multiplier )
    {
    	this.iron *= multiplier;
    	this.metal *= multiplier;
    	this.kryptonite *= multiplier;
    	this.spice *= multiplier;
    	return this;
    }

    public double points()
    {
        long all = iron + metal + kryptonite;
        return (all) / 2000.0;
    }

    public static Cost fromString( String value )
    {
        Cost rv = null;
        if( value != null && value.trim().length() > 0 )
        {
            String parts[] = value.split( "," );
            if( parts.length > 3 )
            {
                long iron = DataManager.parseLongAttribute( parts[0] );
                long metal = DataManager.parseLongAttribute( parts[1] );
                long kryptonite = DataManager.parseLongAttribute( parts[2] );
                long spice = DataManager.parseLongAttribute( parts[3] );
                rv = new Cost( iron, metal, kryptonite, spice );
            }
        }
        return rv;
    }

    public String toString()
    {
        String rv = null;
        if( !isEmpty() )
        {
            StringBuilder sb = new StringBuilder();
            sb.append( iron );
            sb.append( ", " );
            sb.append( metal );
            sb.append( ", " );
            sb.append( kryptonite );
            sb.append( ", " );
            sb.append( spice );
            rv = sb.toString();
        }
        return rv;
    }

    public int compareTo( Cost that )
    {
        long weightThis = 2 * this.iron / Resource.iron.getWeight() + 2 * this.metal / Resource.metal.getWeight() + 2 * this.kryptonite / Resource.kryptonite.getWeight() + 2 * this.spice / Resource.spice.getWeight();
        long weightThat = 2 * that.iron / Resource.iron.getWeight() + 2 * that.metal / Resource.metal.getWeight() + 2 * that.kryptonite / Resource.kryptonite.getWeight() + 2 * that.spice / Resource.spice.getWeight();
        if( weightThis < weightThat)
        {
            return -1;
        }
        else if( weightThis > weightThat)
        {
            return 1;
        }
        return 0;
    }


}
