/****************************************
 *  COPYRIGHT (C) 2020-2024
 *  Holger Graf
 ****************************************/
package siarchive.components;

import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.ButtonUI;
import javax.swing.plaf.basic.BasicButtonUI;

/**
 * @author graf
 *
 */
public class ClosableTabbedPane extends JTabbedPane {

    private static final long serialVersionUID = -1342750622729698378L;
    private final Icon closeIcon;

    public ClosableTabbedPane(final Icon closeIcon) {
        this.closeIcon = closeIcon;
    }

    @Override
    public void addTab(String title, Component component) {
        int index = getTabCount();
        super.addTab( null, component );
        super.setTabComponentAt(index, new TabHeader(title, component));
        setSelectedIndex( index );
    }

    @Override
    public String getTitleAt(int index) {
        TabHeader header = (TabHeader)getTabComponentAt(index);
        return (header != null) ? header.getTitle() : "";
    }

    @Override
    public void setTitleAt(int index, String title) {
        TabHeader header = (TabHeader)getTabComponentAt(index);
        header.setTitle(title);
    }

    private class PlainButton extends JButton {

        private static final long serialVersionUID = 6488272562862446483L;

        /**
         * @param icon
         */
        public PlainButton( Icon icon ) {
            super(icon);
            setOpaque(false);
        }

        /**
         * @see javax.swing.AbstractButton#setUI(javax.swing.plaf.ButtonUI)
         */
        @Override
        public void setUI( ButtonUI ui ) {
            ButtonUI plainUI = (ButtonUI)BasicButtonUI.createUI(this);
            super.setUI(plainUI);
        }
    }

    private class TabHeader extends JPanel
    {
        private static final long serialVersionUID = 3265944962645128235L;
        private final JLabel titleComponent;

        public TabHeader(final String title, final Component component) {
            setBorder(new EmptyBorder(2, 0, 0, 0));
            setLayout(new FlowLayout(LEFT, 3, 0));
            setOpaque(false);
            titleComponent = new JLabel(title);
            add(titleComponent);
            add(new JLabel("  "));
            JButton button = new PlainButton( closeIcon );
            button.setBorder(new EmptyBorder(2, 2, 1, 1));
            add(button);
            button.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed( ActionEvent e ) {
                    if( component != null ) {
                        ClosableTabbedPane.this.remove(component );
                    }
                }
            });
        }

        public String getTitle() {
            return titleComponent.getText();
        }

        public void setTitle(String title) {
            String oldTitle = titleComponent.getText();
            titleComponent.setText(title);
            if (title == null || oldTitle == null || !title.equals(oldTitle)) {
                revalidate();
                repaint();
            }
        }
    }
}
