/****************************************
 *  COPYRIGHT (C) 2008-2024
 *  Holger Graf
 ****************************************/
package siarchive;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.Point;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.io.File;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.ColorUIResource;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import siarchive.components.Cost;
import siarchive.components.DataTransferMethod;
import siarchive.components.Flavor;
import siarchive.components.ImportExportPane;
import siarchive.components.LaFStyle;
import siarchive.components.ModulePane;
import siarchive.components.Priority;
import siarchive.components.Resource;
import siarchive.components.Ship;
import siarchive.components.Status;
import siarchive.components.Version;
import siarchive.datatransfer.DataServer;
import siarchive.galaxy.GalaxyView;
import siarchive.i18n.Language;
import siarchive.imports.ImportPane;
import siarchive.persistence.Account;
import siarchive.persistence.Alliance;
import siarchive.persistence.ApplicationConfiguration;
import siarchive.persistence.BattleReport;
import siarchive.persistence.BlackHole;
import siarchive.persistence.DbInfoBox;
import siarchive.persistence.Note;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.persistence.SpyReport;
import siarchive.search.BattleReportSummaryPane;
import siarchive.search.BlackHoleSummaryPane;
import siarchive.search.PlanetSummaryPane;
import siarchive.search.SpyReportSummaryPane;
import siarchive.tools.EscapeUtils;


/**
 *
 * @author graf
 *
 */
public class MainFrame extends JFrame implements ComponentListener, ImportExportPane
{
    private static final long serialVersionUID = -1878874776754387489L;

    protected static final int HELP = 1;
    protected static final int ABOUT = 2;

    public static final Color live = new Color(21,114,2,40);
    public static final Color error = new Color(121,14,2,40);
    public static final Color neutral = new JLabel().getBackground();

    private DataManager dataManager;

    private JTabbedPane pane;

    private JComboBox<Object> accountBox;

    private JMenu jMenuFile;
    private JMenuItem jMenuFileExport;
    private JMenuItem jMenuFileImport;
    private JMenuItem jMenuFileExit;
    private JMenu jMenuSkins;
    private Map<LaFStyle, JRadioButtonMenuItem> menuSkins = new LinkedHashMap<>();
    private JMenu jMenuTools;
    private JCheckBoxMenuItem jMenuToolsGalaxy;
    private JMenu jMenuToolsDataTransfer;
    private JCheckBoxMenuItem jMenuToolsDataTransferGrabClipBoard;
    private JCheckBoxMenuItem jMenuToolsDataTransferHttp;
    private JMenuItem jMenuToolsImport;
    private JMenuItem jMenuToolsExport;
    private JMenuItem jMenuToolsExportPlanets;
    private JMenuItem jMenuToolsExportSpies;
    private JMenuItem jMenuToolsExportBattles;
    private JMenuItem jMenuToolsExportBlackHoles;
    private JMenuItem jMenuToolsFonts;
    private JMenuItem jMenuToolsDbInfo;
    private JMenu jMenuHelp;
    private JMenuItem jMenuHelpHelp;
    private JMenuItem jMenuHelpAbout;

    private Object exportSource;
    private JLabel labelAccount;
    private JButton buttonAccount;

    private Map<String, Integer> titles = new HashMap<String, Integer>();

    private final GalaxyView galaxyView;
    private final ImportPane importPane;
    private final DataServer dataServer;
    private JButton dbState;
    private JButton connectionState;

    private boolean restart = false;

    /**
     * @throws HeadlessException
     */
    public MainFrame() throws HeadlessException
    {
        super();
        this.setIconImage( Resources.getIcon( "/symbol.png" ).getImage() );
        enableEvents( AWTEvent.WINDOW_EVENT_MASK );

        dataManager = new DataManager();
        // read configuration
        ApplicationConfiguration configuration = dataManager.getApplicationConfiguration();
        LaFStyle style = dataManager.getCurrentStyle();
        setLookAndFeel( style );

        dataServer = new DataServer(this);
        importPane = new ImportPane( this );
        galaxyView = new GalaxyView( this );

        UIDefaults uiDefaults = UIManager.getDefaults();
        Font font = ( Font )uiDefaults.get( "Table.font" );
        dataManager.setDefaultFont( font );
        String fontName = configuration.getFontName();
        if( fontName != null )
        {
            font = new Font( fontName, Font.PLAIN, configuration.getFontSize() );
            FontBox.updateFont( font );
        }

        labelAccount = new JLabel();
        buttonAccount = new JButton();
        dbState = new JButton();
        dbState.setRolloverEnabled(false);
        connectionState = new JButton();
        connectionState.setRolloverEnabled(false);
        connectionState.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed( ActionEvent e ) {
                jMenuToolsDataTransferHttp.setSelected(!jMenuToolsDataTransferHttp.isSelected());
                actionManageServer();
            }
        });

        JPanel contentPane = ( JPanel )this.getContentPane();
        setTitle( "SI Archive");
        setJMenuBar( createMenu() );
        selectStyle(style);

        switch(configuration.getTransferMethod()) {
            case clipboard:
            {
                jMenuToolsDataTransferGrabClipBoard.setSelected( true );
                jMenuToolsDataTransferHttp.setSelected( false );
                actionGrabClipBoard();
            }
            break;
            case http:
            {
                jMenuToolsDataTransferGrabClipBoard.setSelected( false );
                jMenuToolsDataTransferHttp.setSelected( true );
                actionManageServer();
            }
            break;
            case off:
            {
                jMenuToolsDataTransferGrabClipBoard.setSelected( false );
                jMenuToolsDataTransferHttp.setSelected( false );
                connectionState.setIcon(Resources.getIcon("/connection-off.png"));
            }
            break;
        }

        JPanel northPanel = new JPanel();
        northPanel.setLayout( new BorderLayout() );
        northPanel.setBorder( BorderFactory.createEtchedBorder() );

        JPanel accountPanel = new JPanel( new FlowLayout( FlowLayout.LEFT ) );
        JPanel p = new JPanel();
        p.setLayout( new BoxLayout( p, BoxLayout.X_AXIS ) );
        accountPanel.add( p );
        p.add( labelAccount );
        p.add( Box.createHorizontalStrut( 8 ) );
        accountBox = new JComboBox<Object>();
        try
        {
            updateAccounts();
        }
        catch( SQLException ex )
        {
            showErrorInfo( ex );
        }
        accountBox.addItemListener( new ItemListener() {

            public void itemStateChanged( ItemEvent e )
            {
                actionComboBoxChanged( e );
            }

        });

        p.add( accountBox );
        p.add( Box.createHorizontalStrut( 8 ) );
        buttonAccount.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionAccount();
            }
        });
        p.add( buttonAccount );
        p.add( Box.createHorizontalGlue() );
        northPanel.add( accountPanel, BorderLayout.CENTER );

        JPanel statePanel = new JPanel(new GridLayout(1, 5, 0, 0));
        statePanel.setBorder(null);
        dbState.setIcon( Resources.getIcon( "/db-off.png" ) );
        statePanel.add( dbState );
        connectionState.setOpaque(true);
        statePanel.add( connectionState );
        northPanel.add( statePanel, BorderLayout.EAST );

        contentPane.add( northPanel, BorderLayout.NORTH );

        JPanel centerPanel = createCenterPanel();
        contentPane.add( centerPanel, BorderLayout.CENTER );

        setSize( configuration.getMainFrameSize() );
        addComponentListener( this );
        if( configuration.isShowGalaxyView() )
        {
            showGalaxyView( 1, 1, true );
        }
    }

    public DataManager getDataManager()
    {
        return dataManager;
    }


    protected JMenuBar createMenu()
    {
        JMenuBar jMenuBar = new JMenuBar();
        jMenuFile = new JMenu();
        jMenuFileExport = new JMenuItem();
        jMenuFileImport = new JMenuItem();
        jMenuFileExit = new JMenuItem();
        jMenuSkins = new JMenu();
        jMenuTools = new JMenu();
        jMenuToolsGalaxy = new JCheckBoxMenuItem();
        jMenuToolsDataTransfer = new JMenu();
        jMenuToolsDataTransferGrabClipBoard = new JCheckBoxMenuItem();
        jMenuToolsDataTransferHttp = new JCheckBoxMenuItem();
        jMenuToolsImport = new JMenuItem();
        jMenuToolsExport = new JMenu();
        jMenuToolsExportPlanets = new JMenuItem();
        jMenuToolsExportSpies = new JMenuItem();
        jMenuToolsExportBattles = new JMenuItem();
        jMenuToolsExportBlackHoles = new JMenuItem();
        jMenuToolsFonts = new JMenuItem();
        jMenuToolsDbInfo = new JMenuItem();
        jMenuHelp = new JMenu();
        jMenuHelpHelp = new JMenuItem();
        jMenuHelpAbout = new JMenuItem();


        jMenuFileExit.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_Q, InputEvent.CTRL_DOWN_MASK ) );
        jMenuToolsGalaxy.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_G, InputEvent.CTRL_DOWN_MASK ) );
        jMenuToolsDataTransferGrabClipBoard.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_B, InputEvent.CTRL_DOWN_MASK ) );
        jMenuToolsDataTransferHttp.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_W, InputEvent.CTRL_DOWN_MASK ) );
        jMenuHelpHelp.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_F1, 0 ) );
        jMenuHelpAbout.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_F1, InputEvent.CTRL_DOWN_MASK ) );

        jMenuFileImport.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionImport( e );
            }
        } );
        jMenuFile.add( jMenuFileImport );
        jMenuFileExport.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionExport( e );
            }
        } );
        jMenuFile.add( jMenuFileExport );
        jMenuFileExit.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                jMenuFileExit_actionPerformed( e );
            }
        } );
        jMenuFile.add( jMenuFileExit );

        ActionListener l = new ActionListener() {
            public void actionPerformed( ActionEvent e ) {
                actionSwitchLaF( e );
            }
        };
        ButtonGroup group = new ButtonGroup();
        LaFStyle.Family family = LaFStyle.values()[0].getFamily();
        for(LaFStyle style : LaFStyle.values()) {
            JRadioButtonMenuItem item = new JRadioButtonMenuItem();
            menuSkins.put(style, item);
            item.setText(style.getName());
            item.setMnemonic(style.getMnemonic());
            group.add( item );
            item.addActionListener( l );
            if(family != style.getFamily()) {
                family = style.getFamily();
                jMenuSkins.add( new JSeparator() );
            }
            jMenuSkins.add( item );
        }

        jMenuToolsGalaxy.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionGalaxyView( e );
            }
        } );
        jMenuTools.add( jMenuToolsGalaxy );
        jMenuTools.add( jMenuToolsDataTransfer );
        jMenuToolsDataTransferGrabClipBoard.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionGrabClipBoard();
            }
        } );
        jMenuToolsDataTransfer.add( jMenuToolsDataTransferGrabClipBoard );
        jMenuToolsDataTransferHttp.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionManageServer();
            }
        } );
        jMenuToolsDataTransfer.add( jMenuToolsDataTransferHttp );
        jMenuToolsImport.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionImport( e );
            }
        } );
        jMenuTools.add( jMenuToolsImport );

        ActionListener exportListener = new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionExport( e );
            }
        };
        jMenuToolsExportBattles.addActionListener( exportListener );
        jMenuToolsExportBlackHoles.addActionListener( exportListener );
        jMenuToolsExportPlanets.addActionListener( exportListener );
        jMenuToolsExportSpies.addActionListener( exportListener );
        jMenuToolsExport.add( jMenuToolsExportPlanets );
        jMenuToolsExport.add( jMenuToolsExportBlackHoles );
        jMenuToolsExport.add( jMenuToolsExportSpies );
        jMenuToolsExport.add( jMenuToolsExportBattles );
        jMenuTools.add( jMenuToolsExport );
        jMenuToolsFonts.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                JDialog fonts = new FontBox( MainFrame.this );
                showModalBox( fonts );
            }
        } );
        //        jMenuTools.add( jMenuToolsFonts );
        jMenuToolsDbInfo.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                JDialog dbInfo = new DbInfoBox( MainFrame.this );
                showModalBox( dbInfo );
            }
        } );
        jMenuTools.add( jMenuToolsDbInfo );

        jMenuHelpHelp.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionHelp( HELP );
            }
        } );
        jMenuHelp.add( jMenuHelpHelp );
        jMenuHelpAbout.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                actionHelp( ABOUT );
            }
        } );
        jMenuHelp.add( jMenuHelpAbout );

        jMenuBar.add( jMenuFile );
        jMenuBar.add( jMenuSkins );
        jMenuBar.add( jMenuTools );
        jMenuBar.add( jMenuHelp );
        return jMenuBar;
    }

    protected void actionGrabClipBoard()
    {
        if(jMenuToolsDataTransferHttp.isSelected()) {
            jMenuToolsDataTransferHttp.setSelected(false);
        }
        importPane.setGrabClipboard( jMenuToolsDataTransferGrabClipBoard.isSelected() );
    }

    protected void actionManageServer()
    {
        if(jMenuToolsDataTransferGrabClipBoard.isSelected()) {
            jMenuToolsDataTransferGrabClipBoard.setSelected(false);
            importPane.setGrabClipboard( jMenuToolsDataTransferGrabClipBoard.isSelected() );
        }
        if(jMenuToolsDataTransferHttp.isSelected()) {
            dataServer.startServer();
        }
        else {
            dataServer.stopServer();
        }
        connectionState.setIcon((dataServer.getPort() == 0) ? Resources.getIcon("/connection-off.png") : Resources.getIcon("/connection-on.png"));
        if(jMenuToolsDataTransferHttp.isSelected() && (dataServer.getPort() == 0)) {
            connectionState.setBackground(error);
        }
        else {
            connectionState.setBackground(neutral);
        }
    }

    protected void actionImportDb( NodeList accountList )
    {
        dbOn();
        try
        {
            System.out.print("start DB import...");
            long now = System.currentTimeMillis();
            for( int i = 0; i < accountList.getLength(); i++ )
            {
                Element accountNode = ( Element )accountList.item( i );
                Account account = new Account();
                account.setName( accountNode.getAttribute( Parser.Attribute_Name ) );
                account.setLanguage( parseLanguageAttribute( accountNode.getAttribute( Parser.Attribute_Language ) ) );
                account.setFlavor( parseFlavorAttribute( accountNode.getAttribute( Parser.Attribute_Flavor ) ) );
                account.setDefault( parseBooleanAttribute( accountNode.getAttribute( Parser.Attribute_Default ) ) );
                account.setOutpost( parseBooleanAttribute( accountNode.getAttribute( Parser.Attribute_Outpost ) ) );
                account.setIndex( i );
                dataManager.save( account );
                importPlanets( accountNode, account );
                importBlackHoles( accountNode, account );
                importBattleReports( accountNode, account );
                importSpyReports( accountNode, account );
            }
            System.out.println(" finished: " + (System.currentTimeMillis() - now) + "ms" );

            updateAccounts();
            if( dataManager.getActiveAccount() == null )
            {
                for( Account account : dataManager.getAccounts() )
                {
                    if( account.isDefault() )
                    {
                        accountBox.setSelectedItem( account );
                    }
                }
            }
        }
        catch( Exception ex )
        {
            try
            {
                dataManager.rollback();
            }
            catch( SQLException ex1 )
            {
                ex1.printStackTrace();
            }
            showErrorInfo( ex );
        }
        dbOff();
    }

    protected void actionExportDb( Document doc )
    {
        dbOn();
        System.out.print("start DB export...");
        long now = System.currentTimeMillis();
        try
        {
            Collection<Account> accounts = dataManager.getAccounts();
            for( Account account : accounts )
            {
                Element accountNode = createAccountNode(doc, account);
                Long accountId = account.getId();
                exportPlanets( doc, accountNode, dataManager.getPlanets( accountId ) );
                exportBlackHoles( doc, accountNode, dataManager.getBlackHoles( accountId ) );
                exportBattleReports( doc, accountNode, dataManager.getBattleReports( accountId ) );
                exportSpyReports( doc, accountNode, dataManager.getSpyReports( accountId ) );
            }
        }
        catch( Exception ex )
        {
            try
            {
                dataManager.rollback();
            }
            catch( SQLException ex1 )
            {
                ex1.printStackTrace();
            }
            showErrorInfo( ex );
        }
        System.out.println(" finished: " + (System.currentTimeMillis() - now) + "ms" );
        dbOff();
    }

    protected void actionImport( ActionEvent e )
    {
        Document doc = dataManager.readData( this );
        if( doc != null )
        {
            NodeList accountList = doc.getElementsByTagName( Parser.Tag_Account );
            if( accountList != null )
            {
                int result = JOptionPane.OK_OPTION;
                Version currentVersion = AboutBox.getVersion();
                Version documentVersion = dataManager.getDocumentVersion( doc );
                if( documentVersion.getMajor() + 1 - currentVersion.getMajor() < 0 )
                {
                    result = JOptionPane.showConfirmDialog( this,
                                    dataManager.getI18nText( "Version.nomatch", documentVersion.toString() ),
                                    dataManager.getI18nText( "Warning" ),
                                    JOptionPane.YES_NO_OPTION,
                                    JOptionPane.WARNING_MESSAGE
                                    );
                }
                if( result == JOptionPane.OK_OPTION )
                {
                    if( accountList.getLength() == 1 && dataManager.getActiveAccount() != null )
                    {
                        Element accountNode = ( Element )accountList.item( 0 );
                        Language language = parseLanguageAttribute( accountNode.getAttribute( Parser.Attribute_Language ) );
                        Flavor flavor = parseFlavorAttribute( accountNode.getAttribute( Parser.Attribute_Flavor ) );
                        boolean isOutpost = parseBooleanAttribute(accountNode.getAttribute( Parser.Attribute_Outpost ));
                        // first, check account compatibility
                        if( language == dataManager.getActiveAccount().getLanguage()
                         && flavor == dataManager.getActiveAccount().getFlavor()
                         && isOutpost == dataManager.getActiveAccount().isOutpost())
                        {
                            result = JOptionPane.OK_OPTION;
                            if( !dataManager.getActiveAccount().getName().equals( accountNode.getAttribute( Parser.Attribute_Name ) ) )
                            {
                                result = JOptionPane.showConfirmDialog( this,
                                                dataManager.getI18nText( "Account.name.nomatch" ),
                                                dataManager.getI18nText( "Warning" ),
                                                JOptionPane.YES_NO_OPTION,
                                                JOptionPane.WARNING_MESSAGE
                                                );
                            }
                            if( result == JOptionPane.OK_OPTION )
                            {
                                dbOn();
                                System.out.print("start DB import...");
                                long now = System.currentTimeMillis();
                                try
                                {
                                    importBattleReports( doc );
                                    importBlackHoles( doc );
                                    importPlanets( doc );
                                    importSpyReports( doc );
                                    System.out.println(" finished: " + (System.currentTimeMillis() - now) + "ms" );
                                    dataManager.cleanupOrphans();
                                }
                                catch( Exception ex )
                                {
                                    try
                                    {
                                        dataManager.rollback();
                                    }
                                    catch( SQLException ex1)
                                    {
                                        ex1.printStackTrace();
                                    }
                                    showErrorInfo( ex );
                                }

                                dbOff();
                                updatePane();
                            }
                        }
                        else
                        {
                            JOptionPane.showConfirmDialog( this,
                                            dataManager.getI18nText( "Account.language.nomatch" ),
                                            dataManager.getI18nText( "Error" ),
                                            JOptionPane.DEFAULT_OPTION,
                                            JOptionPane.ERROR_MESSAGE
                                            );
                        }
                    }
                    else if( dataManager.getActiveAccount() == null )
                    {
                        actionImportDb( accountList );
                    }
                }
            }
        }
    }

    protected void actionExport( ActionEvent e )
    {
        exportSource = e.getSource();
        dataManager.saveData( this );
        exportSource = null;
    }

    public void showGalaxyView( int galaxy, int system, boolean isVisible )
    {
        galaxyView.setSystem( galaxy, system );
        jMenuToolsGalaxy.setSelected( isVisible );
        galaxyView.setVisible( isVisible );
        galaxyView.repaint();
    }

    public boolean isVisibleGalaxyView()
    {
        return jMenuToolsGalaxy.isSelected();
    }

    public void closeGalaxyView()
    {
        jMenuToolsGalaxy.setSelected( false );
    }

    protected void actionGalaxyView( ActionEvent e )
    {
        galaxyView.setVisible( jMenuToolsGalaxy.isSelected() && jMenuToolsGalaxy.isEnabled() );
    }

    protected void actionAccount()
    {
        JDialog dlg = new AccountBox( this );
        showModalBox( dlg );
        try
        {
            updateAccounts();
            if( dataManager.getActiveAccount() == null )
            {
                for( Account account : dataManager.getAccounts() )
                {
                    if( account.isDefault() )
                    {
                        accountBox.setSelectedItem( account );
                    }
                }
            }
        }
        catch( SQLException ex )
        {
            try
            {
                dataManager.rollback();
            }
            catch( SQLException ex1 )
            {
                ex1.printStackTrace();
            }
            showErrorInfo( ex );
        }
    }

    protected void actionSwitchLaF( ActionEvent e )
    {
        LaFStyle style = LaFStyle.defaultLaF;
        for(Map.Entry<LaFStyle, JRadioButtonMenuItem> entry : menuSkins.entrySet()) {
            if( entry.getValue().isSelected() ) {
                style = entry.getKey();
                break;
            }
        }

        if( style.getFamily() != dataManager.getCurrentStyle().getFamily()) {
            int result = JOptionPane.showConfirmDialog( this,
                            dataManager.getI18nText( "MainFrame.menu.skins.restart.text" ),
                            dataManager.getI18nText( "MainFrame.menu.skins.restart" ),
                            JOptionPane.YES_NO_OPTION,
                            JOptionPane.QUESTION_MESSAGE
                        );
            if( result == JOptionPane.OK_OPTION )
            {
                dataManager.setCurrentStyle( style );
                restart = true;
                jMenuFileExit_actionPerformed(null);
            }
            selectStyle( dataManager.getCurrentStyle() );

        }
        else if( style != dataManager.getCurrentStyle() ) {
            dataManager.setCurrentStyle( style );
            setLookAndFeel( style );
            updatePane();
        }
    }

    protected void selectStyle( LaFStyle style )
    {
        JRadioButtonMenuItem item = menuSkins.get(style);
        if(item == null) {
            item = menuSkins.get(LaFStyle.defaultLaF);
        }
        item.setSelected(true);
    }

    protected void actionHelp( int type )
    {
        JDialog dlg;
        switch( type )
        {
            case HELP:
                dlg = new HelpBox( this );
                break;
            default:
                dlg = new AboutBox( this );
                break;
        }
        showModalBox( dlg );
    }

    public void showModalBox( JDialog dialog )
    {
        Dimension dlgSize = dialog.getPreferredSize();
        Dimension frmSize = getSize();
        Point loc = getLocation();
        dialog.setLocation( ( frmSize.width - dlgSize.width ) / 2 + loc.x,
                        ( frmSize.height - dlgSize.height ) / 2 + loc.y );

        dialog.setModal( true );
        dialog.setLocationRelativeTo( this );
        dialog.setVisible( true );
    }

    public void showErrorInfo( Exception ex )
    {
        showErrorInfo( this, dataManager.getI18nText( "Warning" ), ex );
    }

    public static void showErrorInfo( Component parent, String title, Exception ex )
    {
        ex.printStackTrace();
        String errorInfo = getErrorInfo( ex );
        JOptionPane.showMessageDialog( parent, errorInfo, title, JOptionPane.ERROR_MESSAGE );
    }

    private static String getErrorInfo( Exception ex )
    {
        StringBuilder sb = new StringBuilder();
        String reason = ex.toString();
        String parts[] = reason.split( "\n" );
        for( String part : parts )
        {
            if( part.indexOf( "please help us filling a bug report" ) == -1 )
            {
                sb.append( part );
                sb.append( '\n' );
            }
        }
        StackTraceElement[] stackTrace = ex.getStackTrace();
        for( int i = 0; i < stackTrace.length && i < 6; i++ )
        {
            sb.append( "    at " );
            sb.append( stackTrace[i] );
            sb.append( '\n' );
        }
        return sb.toString();
    }

    public void updateAccounts() throws SQLException
    {
        Account current = dataManager.getActiveAccount();
        accountBox.removeAllItems();
        accountBox.addItem( " ----------- " );
        ArrayList<Account> accounts = new ArrayList<Account>( dataManager.getAccounts() );
        for( Account account : accounts )
        {
            accountBox.addItem( account );
        }
        if( current != null )
        {
            accountBox.setSelectedItem( current );
        }
    }

    public JTabbedPane getTabbedPane()
    {
        return pane;
    }

    protected JPanel createCenterPanel()
    {
        JPanel panel = new JPanel( new BorderLayout() );
        int index = 0;
        String key;

        pane = new JTabbedPane();
        PlanetSummaryPane playerPane = new PlanetSummaryPane( this );
        key = "MainFrame.planets";
        pane.add( playerPane, dataManager.getI18nText( key ) );
        titles.put( key, Integer.valueOf( index ) );
        index++;
        BlackHoleSummaryPane blackHolePane = new BlackHoleSummaryPane( this );
        key = "MainFrame.blackholes";
        pane.add( blackHolePane, dataManager.getI18nText( key ) );
        titles.put( key, Integer.valueOf( index ) );
        index++;
        SpyReportSummaryPane spyReportPane = new SpyReportSummaryPane( this );
        key = "MainFrame.spyReports";
        pane.add( spyReportPane, dataManager.getI18nText( key ) );
        titles.put( key, Integer.valueOf( index ) );
        index++;
        BattleReportSummaryPane battleReportPane = new BattleReportSummaryPane( this );
        key = "MainFrame.battleReports";
        pane.add( battleReportPane, dataManager.getI18nText( key ) );
        titles.put( key, Integer.valueOf( index ) );
        index++;

        pane.addChangeListener( new ChangeListener()
        {

            public void stateChanged( ChangeEvent e )
            {
                updatePane();
            }

        });

        pane.setSelectedIndex( 0 );
        panel.add( pane, BorderLayout.CENTER );

        JSplitPane splitPane = new JSplitPane( JSplitPane.VERTICAL_SPLIT, false, pane, importPane );
        splitPane.setResizeWeight( 0.88 );
        splitPane.setBorder( new EtchedBorder( EtchedBorder.LOWERED ) );
        panel.add( splitPane, BorderLayout.CENTER );

        updatePane();
        return panel;
    }

    public void updatePane()
    {
        updatePane( false );
    }

    public void updatePane( boolean showGalaxy )
    {
        Component c = pane.getSelectedComponent();
        if( c instanceof ModulePane )
        {
            ((ModulePane)c).update();
            //System.out.println( "update pane " + c.getClass().getSimpleName() );
        }
        try
        {
            boolean isEnabled = ( dataManager.getAccounts().isEmpty() );
            jMenuFileImport.setEnabled( isEnabled );
            jMenuFileExport.setEnabled( !isEnabled );
            isEnabled = ( dataManager.getActiveAccount() != null );
            jMenuTools.setEnabled( isEnabled );
            jMenuToolsGalaxy.setEnabled( isEnabled );
            jMenuToolsDataTransferGrabClipBoard.setEnabled( isEnabled );
            jMenuToolsDataTransferHttp.setEnabled( isEnabled );
            jMenuToolsImport.setEnabled( isEnabled );
            jMenuToolsExport.setEnabled( isEnabled );
            if( showGalaxy )
            {
                actionGalaxyView( null );
            }
            importPane.update();
            galaxyView.update();
            if(dataManager.getActiveAccount() == null) {
                galaxyView.setVisible(false);
                jMenuToolsGalaxy.setSelected(false);
            }
            updateLabel();
        }
        catch (SQLException ex)
        {
            try
            {
                dataManager.rollback();
            }
            catch( SQLException ex1 )
            {
                ex1.printStackTrace();
            }
            showErrorInfo( ex );
        }
    }

    private void setLookAndFeel( final LaFStyle style )
    {
        try
        {
            UIManager.setLookAndFeel( style.getClassName() );
            // make controls uniformly non-bold
            String key;
            final Font font;
            UIDefaults uiDefaults = UIManager.getDefaults();
            key = "Label.font";
            // this is explicitly not a FontResource
            // do not allow the LaF to override these settings
            font = ((Font)uiDefaults.get( key )).deriveFont( Font.PLAIN );
            uiDefaults.put( key, font );
            key = "TextField.font";
            uiDefaults.put( key, font );
            key = "Button.font";
            uiDefaults.put( key, font );
            key = "RadioButton.font";
            uiDefaults.put( key, font );
            key = "CheckBox.font";
            uiDefaults.put( key, font );
            key = "Menu.font";
            uiDefaults.put( key, font );
            key = "MenuItem.font";
            uiDefaults.put( key, font );
            key = "RadioButtonMenuItem.font";
            uiDefaults.put( key, font );
            key = "CheckBoxMenuItem.font";
            uiDefaults.put( key, font );
            key = "ComboBox.font";
            uiDefaults.put( key, font );
            key = "Spinner.font";
            uiDefaults.put( key, font );
            key = "TabbedPane.font";
            uiDefaults.put( key, font );
            key = "Table.font";
            uiDefaults.put( key, font );
            key = "TextArea.font";
            uiDefaults.put( key, font );
            key = "ToolTip.font";
            uiDefaults.put( key, font );
            if( style.getFamily() == LaFStyle.Family.nimbus ) {
                /* some settings to let Nimbus look more 'normal' */
                // restore table grid
                uiDefaults.put("Table.showGrid", Boolean.TRUE);
                // set the grid color
                uiDefaults.put("Table.gridColor", new ColorUIResource(Color.lightGray));
                uiDefaults = UIManager.getLookAndFeelDefaults();
                // default dialog button orientation
                uiDefaults.put("OptionPane.buttonOrientation", SwingConstants.CENTER);
                // default dialog button order
                uiDefaults.put("OptionPane.isYesLast", Boolean.FALSE);
                // default dialog button size
                uiDefaults.put("OptionPane.sameSizeButtons", Boolean.TRUE);
            }
        }
        catch ( Exception e )
        {
            e.printStackTrace();
        }
        for (Window window : Window.getWindows())
        {
            SwingUtilities.updateComponentTreeUI(window);
        }
        setSize( dataManager.getApplicationConfiguration().getMainFrameSize() );
    }

    private void updateLabel()
    {
        jMenuFile.setText( dataManager.getI18nText( "MainFrame.menu.file" ) );
        jMenuFile.setMnemonic( dataManager.getI18nText( "MainFrame.menu.file.mnemonic" ).charAt( 0 ) );
        jMenuFileExport.setText( dataManager.getI18nText( "MainFrame.menu.file.exportDb" ) );
        jMenuFileExport.setMnemonic( dataManager.getI18nText( "MainFrame.menu.file.exportDb.mnemonic" ).charAt( 0 ) );
        jMenuFileImport.setText( dataManager.getI18nText( "MainFrame.menu.file.importDb" ) );
        jMenuFileImport.setMnemonic( dataManager.getI18nText( "MainFrame.menu.file.importDb.mnemonic" ).charAt( 0 ) );
        jMenuFileExit.setText( dataManager.getI18nText( "MainFrame.menu.file.exit" ) );
        jMenuFileExit.setMnemonic( dataManager.getI18nText( "MainFrame.menu.file.exit.mnemonic" ).charAt( 0 ) );

        jMenuSkins.setText( dataManager.getI18nText( "MainFrame.menu.skins" ) );
        jMenuSkins.setMnemonic( dataManager.getI18nText( "MainFrame.menu.skins.mnemonic" ).charAt( 0 ) );

        jMenuTools.setText( dataManager.getI18nText( "MainFrame.menu.tools" ) );
        jMenuTools.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.mnemonic" ).charAt( 0 ) );
        jMenuToolsGalaxy.setText( dataManager.getI18nText( "MainFrame.menu.tools.galaxy" ) );
        jMenuToolsGalaxy.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.galaxy.mnemonic" ).charAt( 0 ) );
        jMenuToolsDataTransfer.setText( dataManager.getI18nText( "MainFrame.menu.tools.datatransfer" ) );
        jMenuToolsDataTransfer.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.datatransfer.mnemonic" ).charAt( 0 ) );
        jMenuToolsDataTransferGrabClipBoard.setText( dataManager.getI18nText( "MainFrame.menu.tools.datatransfer.grabClipboard" ) );
        jMenuToolsDataTransferGrabClipBoard.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.datatransfer.grabClipboard.mnemonic" ).charAt( 0 ) );
        jMenuToolsDataTransferHttp.setText( dataManager.getI18nText( "MainFrame.menu.tools.datatransfer.http" ) );
        jMenuToolsDataTransferHttp.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.datatransfer.http.mnemonic" ).charAt( 0 ) );
        jMenuToolsImport.setText( dataManager.getI18nText( "MainFrame.menu.tools.import" ) );
        jMenuToolsImport.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.import.mnemonic" ).charAt( 0 ) );
        jMenuToolsExport.setText( dataManager.getI18nText( "MainFrame.menu.tools.export" ) );
        jMenuToolsExport.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.export.mnemonic" ).charAt( 0 ) );
        jMenuToolsExportBattles.setText( dataManager.getI18nText( "MainFrame.menu.tools.export.battle" ) );
        jMenuToolsExportBattles.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.export.battle.mnemonic" ).charAt( 0 ) );
        jMenuToolsExportBlackHoles.setText( dataManager.getI18nText( "MainFrame.menu.tools.export.blackhole" ) );
        jMenuToolsExportBlackHoles.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.export.blackhole.mnemonic" ).charAt( 0 ) );
        jMenuToolsExportPlanets.setText( dataManager.getI18nText( "MainFrame.menu.tools.export.planet" ) );
        jMenuToolsExportPlanets.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.export.planet.mnemonic" ).charAt( 0 ) );
        jMenuToolsExportSpies.setText( dataManager.getI18nText( "MainFrame.menu.tools.export.spy" ) );
        jMenuToolsExportSpies.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.export.spy.mnemonic" ).charAt( 0 ) );

        jMenuToolsFonts.setText( dataManager.getI18nText( "MainFrame.menu.tools.fonts" ) );
        jMenuToolsFonts.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.fonts.mnemonic" ).charAt( 0 ) );

        jMenuToolsDbInfo.setText( dataManager.getI18nText( "MainFrame.menu.tools.dbInfo" ) );
        jMenuToolsDbInfo.setMnemonic( dataManager.getI18nText( "MainFrame.menu.tools.dbInfo.mnemonic" ).charAt( 0 ) );

        jMenuHelp.setText( dataManager.getI18nText( "MainFrame.menu.help" ) );
        jMenuHelp.setMnemonic( dataManager.getI18nText( "MainFrame.menu.help.mnemonic" ).charAt( 0 ) );
        jMenuHelpHelp.setText( dataManager.getI18nText( "MainFrame.menu.help.help" ) );
        jMenuHelpHelp.setMnemonic( dataManager.getI18nText( "MainFrame.menu.help.help.mnemonic" ).charAt( 0 ) );
        jMenuHelpAbout.setText( dataManager.getI18nText( "MainFrame.menu.help.info" ) );
        jMenuHelpAbout.setMnemonic( dataManager.getI18nText( "MainFrame.menu.help.info.mnemonic" ).charAt( 0 ) );

        labelAccount.setText( dataManager.getI18nText( "MainFrame.account" ) );
        buttonAccount.setText( dataManager.getI18nText( "MainFrame.accounts" ) );

        for( Map.Entry<String, Integer> entry : titles.entrySet() )
        {
            int index = entry.getValue().intValue();
            pane.setTitleAt( index, dataManager.getI18nText( entry.getKey() ) );
        }
    }

    /**
     * @param e
     */
    protected void actionComboBoxChanged( ItemEvent e )
    {
        if( e.getStateChange() == ItemEvent.SELECTED )
        {
            dbOn();
            try
            {
                if( e.getItem() instanceof Account )
                {
                    Account account = ( Account )e.getItem();
                    dataManager.setActiveAccount( account );
                }
                else
                {
                    dataManager.setActiveAccount( null );
                }
            }
            catch( Exception ex )
            {
                showErrorInfo( ex );
            }
            dbOff();
            updatePane();
        }
    }

    // File | Exit action performed
    public void jMenuFileExit_actionPerformed( ActionEvent e )
    {
        // save application state
        try
        {
            ApplicationConfiguration config = dataManager.getApplicationConfiguration();
            config.setShowGalaxyView( jMenuToolsGalaxy.isSelected() );
            DataTransferMethod dataTransfer =
                            jMenuToolsDataTransferGrabClipBoard.isSelected() ? DataTransferMethod.clipboard :
                            jMenuToolsDataTransferHttp.isSelected() ? DataTransferMethod.http : DataTransferMethod.off;

            config.setTransferMethod(dataTransfer );
            dataManager.save( config );
            dataManager.commit();

            dataServer.stopServer();
        }
        catch( SQLException ex )
        {
            try
            {
                dataManager.rollback();
            }
            catch( SQLException ex1 )
            {
                ex1.printStackTrace();
            }
            showErrorInfo( ex );
        }
        finally
        {
            // close DB connection
            dataManager.close();
        }
        Main.finish( restart );
    }

    // Overridden so we can exit when window is closed
    protected void processWindowEvent( WindowEvent e )
    {
        super.processWindowEvent( e );
        if( e.getID() == WindowEvent.WINDOW_CLOSING )
        {
            jMenuFileExit_actionPerformed( null );
        }
    }

    public void componentHidden( ComponentEvent e )
    {}

    public void componentShown( ComponentEvent e )
    {}

    public void componentMoved( ComponentEvent e )
    {}

    public void componentResized( ComponentEvent e )
    {
        Dimension size = dataManager.getApplicationConfiguration().getMainFrameSize();
        size.width = getSize().width;
        size.height = getSize().height;
    }

    public void dbOn()
    {
        dbState.setIcon( Resources.getIcon( "/db-on.png" ) );
        Dimension size = dbState.getSize();
        dbState.paintImmediately( 0, 0, size.width, size.height );
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    }

    public void dbOff()
    {
        dbState.setIcon( Resources.getIcon( "/db-off.png" ) );
        dbState.repaint();
        setCursor(Cursor.getDefaultCursor());
    }

    private void importSpyReports( Document doc ) throws SQLException
    {
        Account account = dataManager.getActiveAccount();
        Element rootElement = doc.getDocumentElement();
        importSpyReports( rootElement, account );
    }

    private void importSpyReports( Element rootElement, Account account ) throws SQLException
    {
        NodeList reportList = rootElement.getElementsByTagName( Parser.Tag_SpyReport );
        if( reportList != null )
        {
            for( int i = 0; i < reportList.getLength(); i++  )
            {
                Element reportNode = ( Element )reportList.item( i );
                SpyReport report = new SpyReport();
                report.setAccount( account.getId() );
                report.setAsteroid( parseBooleanAttribute( reportNode.getAttribute( Parser.Attribute_isAsteroid ) ) );
                report.setCreationTime( parseLongAttribute( reportNode.getAttribute( Parser.Attribute_Time ) ) );
                report.setData( unescapeString( reportNode.getAttribute( Parser.Attribute_Data ) ) );
                report.setPlanetName( unescapeString( reportNode.getAttribute( Parser.Attribute_Name ) ) );
                report.setPosition( parseIntegerAttribute( reportNode.getAttribute( Parser.Attribute_Position ) ) );
                report.setFlag( parsePriorityAttribute( reportNode.getAttribute( Parser.Attribute_Priority ) ) );
                report.setRetainNotes( true );
                String noteText = unescapeString( reportNode.getAttribute( Parser.Attribute_Notes ) );
                Note note = report.getNotes();
                report.setNotes( addToNote( note, noteText ) );
                dataManager.save( report );
            }
            dataManager.commit();
        }
    }

    private void importPlanets( Document doc ) throws SQLException
    {
        Account account = dataManager.getActiveAccount();
        Element rootElement = doc.getDocumentElement();
        importPlanets( rootElement, account );
    }

    private void importPlanets( Element rootElement, Account account ) throws SQLException
    {
        // alliances first
        NodeList allianceList = rootElement.getElementsByTagName( Parser.Tag_Alliance );
        // store the DB objects
        Map<Long, Alliance> alliances = new HashMap<Long, Alliance>();
        // save a name mapping for compatibility
        Map<String, Alliance> compatibilityAlliances = new HashMap<String, Alliance>();
        if( allianceList != null )
        {
            for( int i = 0; i < allianceList.getLength(); i++  )
            {
                Element allianceNode = ( Element )allianceList.item( i );
                Alliance alliance = new Alliance();
                alliance.setAccount( account.getId() );
                alliance.setName( unescapeString( allianceNode.getAttribute( Parser.Attribute_Name ) ) );
                alliance.setAllianceId( parseIdAttribute( allianceNode.getAttribute( Parser.Attribute_Id ) ) );
                alliance.setUpdateTime( parseLongAttribute( allianceNode.getAttribute( Parser.Attribute_Time ) ) );
                String tag = unescapeString( allianceNode.getAttribute( Parser.Attribute_Tag ) );
                if( tag != null && tag.trim().length() > 0 )
                {
                    alliance.setTag( tag.trim() );
                }
                alliance.setRetainNotes( true );
                String noteText = unescapeString( allianceNode.getAttribute( Parser.Attribute_Notes ) );
                Note note = alliance.getNotes();
                alliance.setNotes( addToNote( note, noteText ) );
                alliance = dataManager.save( alliance );
                if( alliance.getAllianceId() >= 0 )
                {
                    alliances.put( alliance.getAllianceId(), alliance );
                }
                compatibilityAlliances.put( alliance.getName(), alliance );
            }
        }
        // players next
        NodeList playerList = rootElement.getElementsByTagName( Parser.Tag_Player );
        // store the DB objects
        Map<Long, Player> players = new HashMap<Long, Player>();
        // save a name mapping for compatibility
        Map<String, Player> compatibilityPlayers = new HashMap<String, Player>();
        if( playerList != null )
        {
            for( int i = 0; i < playerList.getLength(); i++  )
            {
                Element playerNode = ( Element )playerList.item( i );
                Player player = new Player();
                player.setAccount( account.getId() );
                player.setName( unescapeString( playerNode.getAttribute( Parser.Attribute_Name ) ) );
                player.setTitle( unescapeString( playerNode.getAttribute( Parser.Attribute_Title ) ) );
                player.setPlayerId( parseIdAttribute( playerNode.getAttribute( Parser.Attribute_Id ) ) );
                player.setPoints( parseLongAttribute( playerNode.getAttribute( Parser.Attribute_Points ) ) );
                player.setSpecialization( playerNode.getAttribute( Parser.Attribute_Specialization ) );
                player.setSpiceValue( parseLongAttribute( playerNode.getAttribute( Parser.Attribute_SpiceValue ) ) );
                player.setStatus( parseStatusAttribute( playerNode.getAttribute( Parser.Attribute_Status ) ) );
                player.setSecondaryStatus( parseStatusAttribute( playerNode.getAttribute( Parser.Attribute_Secondary_Status ) ) );
                // restore alliance first by Id, failing that by name
                long id = parseIdAttribute( playerNode.getAttribute( Parser.Attribute_Alliance_Id ) );
                if( id >= 0)
                {
                    player.setAlliance( alliances.get( id ) );
                }
                else
                {
                    player.setAlliance( compatibilityAlliances.get( playerNode.getAttribute( Parser.Attribute_Alliance ) ) );
                }
                player.setUpdateTime( parseLongAttribute( playerNode.getAttribute( Parser.Attribute_Time ) ) );
                player.setRetainNotes( true );
                String noteText = unescapeString( playerNode.getAttribute( Parser.Attribute_Notes ) );
                Note note = player.getNotes();
                player.setNotes( addToNote( note, noteText ) );
                player = dataManager.save( player );
                if( player.getPlayerId() >= 0 )
                {
                    players.put( player.getPlayerId(), player );
                }
                compatibilityPlayers.put( player.getName(), player );
            }
        }
        // finally planets
        NodeList planetList = rootElement.getElementsByTagName( Parser.Tag_Planet );
        if( planetList != null )
        {
            for( int i = 0; i < planetList.getLength(); i++  )
            {
                Element planetNode = ( Element )planetList.item( i );
                Planet planet = new Planet();
                planet.setAccount( account.getId() );
                final String name = planetNode.getAttribute( Parser.Attribute_Name );
                final String owner = planetNode.getAttribute( Parser.Attribute_Owner );
                // planets which are only present b/c of event or debris will not have name or owner
                final boolean isEmpty = name.isEmpty() && owner.isEmpty();
                planet.setPosition( parseIntegerAttribute( planetNode.getAttribute( Parser.Attribute_Position ) ) );
                planet.setEvent( parseBooleanAttribute( planetNode.getAttribute( Parser.Attribute_isEvent ) ) );
                if(!isEmpty) {
                    planet.setName( unescapeString( name ) );
                    planet.setAsteroid(  parseBooleanAttribute( planetNode.getAttribute( Parser.Attribute_isAsteroid ) ) );
                    // restore owner first by Id, failing that by name
                    long id = parseIdAttribute( planetNode.getAttribute( Parser.Attribute_Owner_Id ) );
                    if( id >= 0)
                    {
                        planet.setOwner( players.get( id ) );
                    }
                    else
                    {
                        planet.setOwner( compatibilityPlayers.get( owner ) );
                    }
                }
                planet.setUpdateTime( parseLongAttribute( planetNode.getAttribute( Parser.Attribute_Time ) ) );
                planet.setDebris( parseCostAttribute( planetNode.getAttribute( Parser.Attribute_Debris ) ) );
                planet.setRetainNotes( true );
                String noteText = unescapeString( planetNode.getAttribute( Parser.Attribute_Notes ) );
                Note note = planet.getNotes();
                planet.setNotes( addToNote( note, noteText ) );
                planet = dataManager.save( planet );
            }
        }
        dataManager.commit();
    }

    private void importBattleReports( Document doc ) throws SQLException
    {
        Account account = dataManager.getActiveAccount();
        Element rootNode = doc.getDocumentElement();
        importBattleReports( rootNode, account );
    }

    private void importBattleReports( Element rootNode, Account account ) throws SQLException
    {
        NodeList reportList = rootNode.getElementsByTagName( Parser.Tag_BattleReport );
        if( reportList != null )
        {
            for( int i = 0; i < reportList.getLength(); i++  )
            {
                Element reportNode = ( Element )reportList.item( i );
                BattleReport report = new BattleReport();
                report.setAccount( account.getId() );
                report.setAsteroid( parseBooleanAttribute( reportNode.getAttribute( Parser.Attribute_isAsteroid ) ) );
                report.setCreationTime( parseLongAttribute( reportNode.getAttribute( Parser.Attribute_Time ) ) );
                report.setData( unescapeString( reportNode.getAttribute( Parser.Attribute_Data ) ) );
                report.setDefender( unescapeString( reportNode.getAttribute( Parser.Attribute_Defender ) ) );
                report.setPlanetName( unescapeString( reportNode.getAttribute( Parser.Attribute_Name ) ) );
                report.setPosition( parseIntegerAttribute( reportNode.getAttribute( Parser.Attribute_Position ) ) );
                report.setFlag( parsePriorityAttribute( reportNode.getAttribute( Parser.Attribute_Priority ) ) );
                String attackers = unescapeString( reportNode.getAttribute( Parser.Attribute_Attacker ) );
                report.setAttacker( attackers );
                report.setRetainNotes( true );
                String noteText = unescapeString( reportNode.getAttribute( Parser.Attribute_Notes ) );
                Note note = report.getNotes();
                report.setNotes( addToNote( note, noteText ) );
                dataManager.save( report );
            }
            dataManager.commit();
        }
    }

    private void importBlackHoles( Document doc ) throws SQLException
    {
        Account account = dataManager.getActiveAccount();
        Element rootNode = doc.getDocumentElement();
        importBlackHoles( rootNode, account );
    }

    private void importBlackHoles( Element rootNode, Account account ) throws SQLException
    {
        NodeList blackHoleList = rootNode.getElementsByTagName( Parser.Tag_BlackHole );
        if( blackHoleList != null )
        {
            for( int i = 0; i < blackHoleList.getLength(); i++  )
            {
                Element blackHoleNode = ( Element )blackHoleList.item( i );
                BlackHole blackHole = new BlackHole();
                blackHole.setAccount( account.getId() );
                blackHole.setPosition( parseIntegerAttribute( blackHoleNode.getAttribute( Parser.Attribute_Position ) ) );
                blackHole.setUpdateTime( parseLongAttribute( blackHoleNode.getAttribute( Parser.Attribute_Time ) ) );
                blackHole.setLevel( parseIntegerAttribute( blackHoleNode.getAttribute( Parser.Attribute_Level ) ) );
                blackHole.setSetup( parseBooleanAttribute( blackHoleNode.getAttribute( Parser.Attribute_Setup ) ) );
                blackHole.setAttacked( parseBooleanAttribute( blackHoleNode.getAttribute( Parser.Attribute_Attacked ) ) );
                blackHole.setUpgradeTime( parseLongAttribute( blackHoleNode.getAttribute( Parser.Attribute_Upgrade ) ) );
                blackHole.setPower( parseLongAttribute( blackHoleNode.getAttribute( Parser.Attribute_Power ) ) );
                blackHole.setEmblems( parseLongAttribute( blackHoleNode.getAttribute( Parser.Attribute_Emblems ) ) );
                blackHole.setStars( parseLongAttribute( blackHoleNode.getAttribute( Parser.Attribute_Stars ) ) );
                blackHole.setSpice( parseLongAttribute( blackHoleNode.getAttribute( Resource.spice.name() ) ) );
                NodeList shipList = blackHoleNode.getElementsByTagName( Parser.Tag_Ships );
                if(shipList != null && shipList.getLength() > 0) {
                    // only one node expected
                    Element shipNode = ( Element )shipList.item( 0 );
                    NamedNodeMap shipMap = shipNode.getAttributes();
                    Map<Ship, Long> ships = new HashMap<>();
                    for( int s = 0; s < shipMap.getLength(); s++  ) {
                        Attr attr = (Attr)shipMap.item(s);
                        ships.put( Ship.valueOf(attr.getName()), parseLongAttribute(attr.getValue()));
                    }
                    blackHole.setShips(ships);
                }
                dataManager.save( blackHole );
            }
            dataManager.commit();
        }
    }

    private Note addToNote( Note note, String noteText )
    {
        if( noteText != null && noteText.trim().length() > 0 )
        {
            if( note == null )
            {
                note = new Note();
            }
            note.setText( noteText.trim() );
        }
        return note;
    }

    private Element createAccountNode(Document doc) {
        return createAccountNode(doc, dataManager.getActiveAccount());
    }

    private Element createAccountNode(Document doc, Account account) {
        Element accountNode = doc.createElement( Parser.Tag_Account );
        Node rootElement = doc.getChildNodes().item( 0 );
        rootElement.appendChild( accountNode );
        accountNode.setAttribute( Parser.Attribute_Name, account.getName() );
        accountNode.setAttribute( Parser.Attribute_Default, Boolean.toString( account.isDefault() ) );
        accountNode.setAttribute( Parser.Attribute_Language, account.getLanguage().name() );
        accountNode.setAttribute( Parser.Attribute_Flavor, account.getFlavor().name() );
        accountNode.setAttribute( Parser.Attribute_Outpost, Boolean.toString( account.isOutpost() ) );
        return accountNode;
    }

    public void exportBattleReports( Document doc, Collection<BattleReport> reports )
    {
        dbOn();
        exportBattleReports( doc, createAccountNode(doc),  reports );
        dbOff();
    }

    private void exportBattleReports( Document doc, Element accountNode,  Collection<BattleReport> reports )
    {
        for( BattleReport report : reports )
        {
            Element reportNode = doc.createElement( Parser.Tag_BattleReport );
            accountNode.appendChild( reportNode );
            reportNode.setAttribute( Parser.Attribute_Attacker, escapeString( report.getAttacker() ) );
            reportNode.setAttribute( Parser.Attribute_Data, escapeString( report.getData() ) );
            reportNode.setAttribute( Parser.Attribute_Defender, escapeString( report.getDefender() ) );
            if(report.getPlanetName() != null) {
                reportNode.setAttribute( Parser.Attribute_Name, escapeString( report.getPlanetName() ) );
            }
            reportNode.setAttribute( Parser.Attribute_isAsteroid, Boolean.toString( report.isAsteroid() ) );
            reportNode.setAttribute( Parser.Attribute_Position, Long.toString( report.getPosition() ) );
            if( report.getFlag() != null )
            {
                reportNode.setAttribute( Parser.Attribute_Priority, report.getFlag().name() );
            }
            reportNode.setAttribute( Parser.Attribute_Time, Long.toString( report.getCreationTime() ) );
            if( report.getNotes() != null && report.getNotes().getText().length() > 0 )
            {
                reportNode.setAttribute( Parser.Attribute_Notes, escapeString( report.getNotes().getText() ) );
            }
        }
    }

    public void exportSpyReports( Document doc, Collection<SpyReport> reports )
    {
        dbOn();
        exportSpyReports( doc, createAccountNode(doc), reports );
        dbOff();
    }

    private void exportSpyReports( Document doc, Element accountNode, Collection<SpyReport> reports )
    {
        for( SpyReport report : reports )
        {
            Element reportNode = doc.createElement( Parser.Tag_SpyReport );
            accountNode.appendChild( reportNode );
            reportNode.setAttribute( Parser.Attribute_Data, escapeString( report.getData() ) );
            reportNode.setAttribute( Parser.Attribute_isAsteroid, Boolean.toString( report.isAsteroid() ) );
            reportNode.setAttribute( Parser.Attribute_Name, escapeString( report.getPlanetName() ) );
            reportNode.setAttribute( Parser.Attribute_Position, Long.toString( report.getPosition() ) );
            if( report.getFlag() != null )
            {
                reportNode.setAttribute( Parser.Attribute_Priority, report.getFlag().name() );
            }
            reportNode.setAttribute( Parser.Attribute_Time, Long.toString( report.getCreationTime() ) );
            if( report.getNotes() != null && report.getNotes().getText().length() > 0 )
            {
                reportNode.setAttribute( Parser.Attribute_Notes, escapeString( report.getNotes().getText() ) );
            }
        }
    }

    public void exportPlanets( Document doc, Collection<Planet> planets )
    {
        dbOn();
        exportPlanets( doc, createAccountNode(doc), planets );
        dbOff();
    }

    private void exportPlanets( Document doc, Element accountNode, Collection<Planet> planets )
    {
        // collect dependencies
        Map<Long, Element> alliances = new LinkedHashMap<Long, Element>();
        Map<Long, Element> players = new LinkedHashMap<Long, Element>();
        for( Planet planet : planets )
        {
            Element planetNode = doc.createElement( Parser.Tag_Planet );
            Player owner = planet.getOwner();

            accountNode.appendChild( planetNode );
            planetNode.setAttribute( Parser.Attribute_Name, escapeString( planet.getName() ) );
            planetNode.setAttribute( Parser.Attribute_Position, Long.toString( planet.getPosition() ) );
            planetNode.setAttribute( Parser.Attribute_isAsteroid, Boolean.toString( planet.hasAsteroid() ) );
            planetNode.setAttribute( Parser.Attribute_isEvent, Boolean.toString( planet.isEvent() ) );
            planetNode.setAttribute( Parser.Attribute_Time, Long.toString( planet.getUpdateTime() ) );
            if( planet.getNotes() != null && planet.getNotes().getText().length() > 0 )
            {
                planetNode.setAttribute( Parser.Attribute_Notes, escapeString( planet.getNotes().getText() ) );
            }
            if( planet.getDebris() != null )
            {
                planetNode.setAttribute( Parser.Attribute_Debris, planet.getDebris().toString() );
            }
            if(owner != null) {
                // write out owner ID and name for compatibility
                if( owner.getPlayerId() >= 0 )
                {
                    planetNode.setAttribute( Parser.Attribute_Owner_Id, Long.toString( owner.getPlayerId() ) );
                }
                planetNode.setAttribute( Parser.Attribute_Owner, escapeString( owner.getName() ) );
                if( !players.containsKey( owner.getPlayerId() ) )
                {
                    // attach new player
                    Element playerNode = doc.createElement( Parser.Tag_Player );
                    players.put( owner.getPlayerId(), playerNode );
                    playerNode.setAttribute( Parser.Attribute_Name, escapeString( owner.getName() ) );
                    playerNode.setAttribute( Parser.Attribute_Title, escapeString( owner.getTitle() ) );
                    if( owner.getPlayerId() >= 0)
                    {
                        playerNode.setAttribute( Parser.Attribute_Id, Long.toString( owner.getPlayerId() ) );
                    }
                    playerNode.setAttribute( Parser.Attribute_Points, Long.toString( owner.getPoints() ) );
                    playerNode.setAttribute( Parser.Attribute_SpiceValue, Long.toString( owner.getSpiceValue() ) );
                    playerNode.setAttribute( Parser.Attribute_Specialization, owner.getSpecialization() );
                    playerNode.setAttribute( Parser.Attribute_Time, Long.toString( owner.getUpdateTime() ) );
                    if( owner.getStatus() != null )
                    {
                        playerNode.setAttribute( Parser.Attribute_Status, owner.getStatus().name() );
                    }
                    if( owner.getSecondaryStatus() != null )
                    {
                        playerNode.setAttribute( Parser.Attribute_Secondary_Status, owner.getSecondaryStatus().name() );
                    }
                    if( owner.getNotes() != null && owner.getNotes().getText().length() > 0 )
                    {
                        playerNode.setAttribute( Parser.Attribute_Notes, escapeString( owner.getNotes().getText() ) );
                    }
                    Alliance alliance = owner.getAlliance();
                    // write out alliance ID and name for compatibility
                    if( alliance.getAllianceId() >= 0 )
                    {
                        playerNode.setAttribute( Parser.Attribute_Alliance_Id, Long.toString( alliance.getAllianceId() ) );
                    }
                    playerNode.setAttribute( Parser.Attribute_Alliance, escapeString( alliance.getName() ) );

                    if( !alliances.containsKey( alliance.getAllianceId() ) )
                    {
                        // attach new alliance
                        Element allianceNode = doc.createElement( Parser.Tag_Alliance );
                        alliances.put( alliance.getAllianceId(), allianceNode );
                        allianceNode.setAttribute( Parser.Attribute_Name, escapeString( alliance.getName() ) );
                        if( alliance.getAllianceId() >= 0 )
                        {
                            allianceNode.setAttribute( Parser.Attribute_Id, Long.toString( alliance.getAllianceId() ) );
                        }
                        if( alliance.getTag() != null )
                        {
                            allianceNode.setAttribute( Parser.Attribute_Tag, escapeString( alliance.getTag() ) );
                        }
                        allianceNode.setAttribute( Parser.Attribute_Time, Long.toString( alliance.getUpdateTime() ) );
                        if( alliance.getNotes() != null && alliance.getNotes().getText().length() > 0 )
                        {
                            allianceNode.setAttribute( Parser.Attribute_Notes, escapeString( alliance.getNotes().getText() ) );
                        }
                    }
                }
            }
        }
        // attach dependencies
        for( Element element : players.values() )
        {
            accountNode.appendChild( element );
        }
        for( Element element : alliances.values() )
        {
            accountNode.appendChild( element );
        }
    }

    public void exportBlackHoles( Document doc, Collection<BlackHole> blackHoles )
    {
        dbOn();
        exportBlackHoles( doc, createAccountNode(doc), blackHoles );
        dbOff();
    }

    private void exportBlackHoles( Document doc, Element accountNode, Collection<BlackHole> blackHoles ) {
        for( BlackHole blackHole : blackHoles )
        {
            Element blackHoleNode = doc.createElement( Parser.Tag_BlackHole );
            accountNode.appendChild( blackHoleNode );
            blackHoleNode.setAttribute( Parser.Attribute_Position, Long.toString(blackHole.getPosition()) );
            blackHoleNode.setAttribute( Parser.Attribute_Setup, Boolean.toString( blackHole.isSetup() ) );
            blackHoleNode.setAttribute( Parser.Attribute_Attacked, Boolean.toString( blackHole.isAttacked() ) );
            blackHoleNode.setAttribute( Parser.Attribute_Level, Long.toString(blackHole.getLevel()) );
            blackHoleNode.setAttribute( Parser.Attribute_Power, Long.toString(blackHole.getPower()) );
            blackHoleNode.setAttribute( Parser.Attribute_Stars, Long.toString(blackHole.getStars()) );
            blackHoleNode.setAttribute( Parser.Attribute_Emblems, Long.toString(blackHole.getEmblems()) );
            blackHoleNode.setAttribute( Resource.spice.name(), Long.toString(blackHole.getSpice()) );
            blackHoleNode.setAttribute( Parser.Attribute_Time, Long.toString(blackHole.getUpdateTime()) );
            blackHoleNode.setAttribute( Parser.Attribute_Upgrade, Long.toString(blackHole.getUpgradeTime()) );
            Map<Ship, Long> ships = blackHole.getShips();
            if(ships.size() > 0) {
                Element shipsNode = doc.createElement( Parser.Tag_Ships );
                blackHoleNode.appendChild(shipsNode);
                for(Map.Entry<Ship, Long> entry : ships.entrySet()) {
                    shipsNode.setAttribute(entry.getKey().name(), Long.toString(entry.getValue()));
                }
            }
        }
    }

    public File getCurrentDir()
    {
        return dataManager.getCurrentDir();
    }

    public Document getData( Parser xmlParser )
    {
        Document doc = null;
        if( xmlParser != null )
        {
            try
            {
                doc = xmlParser.getDocument();
                dataManager.createAppRoot( doc );
                if( exportSource == jMenuFileExport )
                {
                    actionExportDb( doc );
                }
                else if( dataManager.getActiveAccount() != null )
                {
                    System.out.print("start DB export...");
                    long now = System.currentTimeMillis();
                    if( exportSource == jMenuToolsExportBattles )
                    {
                        Collection<BattleReport> reports = dataManager.getBattleReports();
                        exportBattleReports( doc, reports );
                    }
                    else if( exportSource == jMenuToolsExportSpies )
                    {
                        Collection<SpyReport> reports = dataManager.getSpyReports();
                        exportSpyReports( doc, reports );
                    }
                    else if( exportSource == jMenuToolsExportPlanets )
                    {
                        Collection<Planet> planets = dataManager.getPlanets();
                        exportPlanets( doc, planets );
                    }
                    else if( exportSource == jMenuToolsExportBlackHoles )
                    {
                        Collection<BlackHole> blackHoles = dataManager.getBlackHoles();
                        exportBlackHoles( doc, blackHoles );
                    }
                    System.out.println(" finished: " + (System.currentTimeMillis() - now) + "ms" );
                }
            }
            catch(SQLException ex)
            {
                try
                {
                    dataManager.rollback();
                }
                catch( SQLException ex1 )
                {
                    ex1.printStackTrace();
                }
                showErrorInfo( ex );
            }
        }
        return doc;
    }

    public String getExtension()
    {
        return dataManager.getExtension();
    }

    public String getFileName()
    {
        StringBuilder sb = new StringBuilder();
        sb.append( DataManager.getFileFormat().format( new Date() ) ).append( ' ' );
        Account account = dataManager.getActiveAccount();
        if( account != null && exportSource != jMenuFileExport )
        {
            sb.append( account.getName() ).append( ' ' );
        }
        if( exportSource == jMenuToolsExportBattles )
        {
            sb.append( dataManager.getI18nText( "BattleReport.battleReport" ) );
        }
        else if( exportSource == jMenuToolsExportSpies )
        {
            sb.append( dataManager.getI18nText( "SpyReport.spyreport" ) );
        }
        else if( exportSource == jMenuToolsExportPlanets )
        {
            sb.append( dataManager.getI18nText( "Planet" ) );
        }
        else if( exportSource == jMenuToolsExportBlackHoles )
        {
            sb.append( dataManager.getI18nText( "BlackHole" ) );
        }
        else
        {
            sb.append( "DB" );
        }
        return sb.toString();
    }

    public MainFrame getMainFrame()
    {
        return this;
    }

    public void setCurrentDir( File currentDir )
    {
        dataManager.setCurrentDir( currentDir );
    }

    private String escapeString(String input)
    {
        return EscapeUtils.escapeXml(input);
    }

    private String unescapeString(String input)
    {
        return EscapeUtils.unescapeXml( input );
    }

    public static boolean parseBooleanAttribute( String value )
    {
        boolean rv = false;

        if( value != null && value.trim().length() > 0 )
        {
            rv = Boolean.parseBoolean( value );
        }
        return rv;
    }

    public static int parseIntegerAttribute( String value )
    {
        int rv = 0;

        if( value != null && value.trim().length() > 0 )
        {
            rv = Integer.parseInt( value );
        }
        return rv;
    }

    public static Cost parseCostAttribute( String value )
    {
        return Cost.fromString( value );
    }

    public static long parseIdAttribute( String value )
    {
        long rv = -1;

        if( value != null && value.trim().length() > 0 )
        {
            rv = Long.parseLong( value );
        }
        return rv;
    }

    public static long parseLongAttribute( String value )
    {
        long rv = 0;

        if( value != null && value.trim().length() > 0 )
        {
            rv = Long.parseLong( value );
        }
        return rv;
    }

    public static Language parseLanguageAttribute( String value )
    {
        Language rv = Language.german;

        if( value != null && value.trim().length() > 0 )
        {
            try
            {
                rv = Language.valueOf( value );
            }
            catch( IllegalArgumentException ex )
            {
                // universal default is german
            }
        }
        return rv;
    }

    public static Flavor parseFlavorAttribute( String value )
    {
        Flavor rv = Flavor.retro;

        if( value != null && value.trim().length() > 0 )
        {
            try
            {
                rv = Flavor.valueOf( value );
            }
            catch( IllegalArgumentException ex )
            {
                // universal default is retro
            }
        }
        return rv;
    }

    public static Priority parsePriorityAttribute( String value )
    {
        Priority rv = null;

        if( value != null && value.trim().length() > 0 )
        {
            try
            {
                rv = Priority.valueOf( value );
            }
            catch( IllegalArgumentException ex )
            {
                // universal default
            }
        }
        return rv;
    }

    public static Status parseStatusAttribute( String value )
    {
        Status rv = Status.none;

        if( value != null && value.trim().length() > 0 )
        {
            try
            {
                rv = Status.valueOf( value );
            }
            catch( IllegalArgumentException ex )
            {
                // universal default
            }
        }
        return rv;
    }

    public static Note parseNoteAttribute( String value )
    {
        Note rv = null;

        if( value != null && value.trim().length() > 0 )
        {
            rv = new Note();
            rv.setText( value );
        }
        return rv;
    }
}
