/****************************************
 *  COPYRIGHT (C) 2008
 *  Holger Graf
 ****************************************/
package siarchive;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.sql.SQLException;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import siarchive.persistence.ApplicationConfiguration;
import siarchive.table.AlternateLineTableCellRenderer;
import siarchive.table.CheckBoxTableCellRenderer;
import siarchive.table.FontTableCellRenderer;
import siarchive.table.FontTableModel;
import siarchive.table.TooltipTable;


/**
 * 
 * @author graf
 *
 */
public class FontBox extends JDialog implements ActionListener
{
    private static final long serialVersionUID = -5828822386166972579L;

    private MainFrame mainFrame = null;
    private DataManager dataManager;
    
    private FontTableModel fonts;
    private JTable fontTable;
    private JSpinner fontSize;
    
	public FontBox( MainFrame parent )
    {
        super( parent );
        mainFrame = ( MainFrame )parent;
        dataManager = mainFrame.getDataManager();
        enableEvents( AWTEvent.WINDOW_EVENT_MASK );
        this.setTitle( dataManager.getI18nText( "" ) );
        JPanel contentPane = ( JPanel )getContentPane();
        contentPane.setLayout( new BorderLayout() );
        contentPane.setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );
        
        JPanel center = createCenterPanel();
        center.setPreferredSize( new Dimension( 540, 240 ) );
        
        JPanel lower = new JPanel( new FlowLayout( FlowLayout.CENTER, 4, 4 ) );
        JButton button = new JButton( dataManager.getI18nText( "OK" ) );
        button.addActionListener( this );
        lower.add( button );
        getRootPane().setDefaultButton( button );
        
        contentPane.add( center, BorderLayout.CENTER );
        contentPane.add( lower, BorderLayout.SOUTH );
        pack();
    }

    protected JPanel createCenterPanel()
    {
        JPanel center = new JPanel( new BorderLayout() );
        center.setBorder( BorderFactory.createEmptyBorder( 5, 10, 5, 10) );

        final SpinnerModel model = new SpinnerNumberModel( 11, 6, 18, 1 );
        fontSize = new JSpinner( model );
        JPanel northPanel = new JPanel( new FlowLayout( FlowLayout.RIGHT, 5, 0 ) );
        northPanel.setBorder( BorderFactory.createEmptyBorder( 5, 10, 0, -4) );
        northPanel.add( new JLabel( dataManager.getI18nText( "fontBox.fontSize" ) ) );
        northPanel.add( fontSize );
        center.add( northPanel, BorderLayout.NORTH );
        fonts = new FontTableModel( mainFrame );
        final TableCellRenderer colRenderer = new CheckBoxTableCellRenderer();
        final TableCellRenderer fontRenderer = new FontTableCellRenderer();
        final TableCellEditor colEditor  = new DefaultCellEditor( new JCheckBox() );
        fontTable = new TooltipTable( fonts ) {
            private static final long serialVersionUID = -7124731757030093187L;

            public TableCellEditor getCellEditor( int row, int column )
            {
                TableCellEditor rv;
                // convert to model
                int modelColumn =  convertColumnIndexToModel( column );
                switch( modelColumn )
                {
                    case 2:
                        rv = colEditor;
                        break;
                    default:
                        rv = super.getCellEditor( row, column );
                }
                return rv;
            }

            public TableCellRenderer getCellRenderer( int row, int column )
            {
                TableCellRenderer rv;
                // convert to model
                int modelColumn =  convertColumnIndexToModel( column );
                switch( modelColumn )
                {
                    case 1:
                        rv = fontRenderer;
                        break;
                    case 2:
                        rv = colRenderer;
                        break;
	                default:
	                	rv = super.getCellRenderer( row, column );
                }
                return rv;
            }
        };
        fontTable.setDefaultRenderer( Object.class, new AlternateLineTableCellRenderer() );
        fontTable.setFont( dataManager.getDefaultFont() );
        JScrollPane scrollPane = new JScrollPane( fontTable, 
                        JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED );
        scrollPane.setBorder( BorderFactory.createEtchedBorder() );
        center.add( scrollPane, BorderLayout.CENTER );
        
        ApplicationConfiguration configuration = dataManager.getApplicationConfiguration();
        if( configuration.getFontName() != null && configuration.getFontSize() > 0 )
        {
            fonts.setFontSize( configuration.getFontSize() );
            model.setValue( Integer.valueOf( configuration.getFontSize() ) );
        }
        model.addChangeListener( new ChangeListener()
        {
            public void stateChanged( ChangeEvent e )
            {
                fonts.setFontSize( ((Number)model.getValue()).floatValue() );
            }
        });

        int width;
        TableColumn col;
        width = 25;
        col = fontTable.getColumnModel().getColumn( 2 );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );
        
        return center;
    }
    
	// Overridden so we can exit when window is closed
    /***************************************************************************
     * Description of the Method
     * @param e Description of the Parameter
     **************************************************************************/
    protected void processWindowEvent( WindowEvent e )
    {
        if( e.getID() == WindowEvent.WINDOW_CLOSING )
        {
            cancel();
        }
        super.processWindowEvent( e );
    }

    // Close the dialog

    /***************************************************************************
     * Description of the Method
     **************************************************************************/
    protected void cancel()
    {
        try
        {
            dataManager.rollback();
        }
        catch( SQLException e )
        {
            mainFrame.showErrorInfo( e );
        }
        dispose();
    }

    // Close the dialog on a button event

    /***************************************************************************
     * Description of the Method
     * @param e Description of the Parameter
     **************************************************************************/
    public void actionPerformed( ActionEvent e )
    {
        Font font = fonts.getFont();
        if( font != null )
        {
            updateFont( font );
            dataManager.getApplicationConfiguration().setFontName( font.getFamily() );
            dataManager.getApplicationConfiguration().setFontSize( font.getSize() );
        }
        dispose();
    }

    public static void updateFont( Font font )
    {
        String[] keys = { "Panel.font", "Table.font", "TextArea.font", "ToolTip.font" };
        UIDefaults uiDefaults = UIManager.getDefaults();
        for( String key : keys )
        {
            uiDefaults.put( key, font );
        }
        for( Window window : Window.getWindows() ) 
        {
            SwingUtilities.updateComponentTreeUI(window);
        }
    }
}
