/****************************************
 *  COPYRIGHT (C) 2008-2021
 *  Holger Graf
 ****************************************/
package siarchive;

import java.awt.Dimension;
import java.awt.Font;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.sql.SQLException;
import java.text.DecimalFormat;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import siarchive.components.AttackStatus;
import siarchive.components.BlackHoleLevel;
import siarchive.components.DataTransferMethod;
import siarchive.components.Flavor;
import siarchive.components.ImportExportPane;
import siarchive.components.LaFStyle;
import siarchive.components.PlanetarySystem;
import siarchive.components.Position;
import siarchive.components.Priority;
import siarchive.components.SiToolFileFilter;
import siarchive.components.Status;
import siarchive.components.Version;
import siarchive.i18n.Language;
import siarchive.persistence.Account;
import siarchive.persistence.Alliance;
import siarchive.persistence.ApplicationConfiguration;
import siarchive.persistence.BattleReport;
import siarchive.persistence.BlackHole;
import siarchive.persistence.DatabaseException;
import siarchive.persistence.DatabaseFactory;
import siarchive.persistence.DbObject;
import siarchive.persistence.Planet;
import siarchive.persistence.Player;
import siarchive.persistence.SpyReport;
import siarchive.persistence.dao.AccountDao;
import siarchive.persistence.dao.AllianceDao;
import siarchive.persistence.dao.ApplicationConfigurationDao;
import siarchive.persistence.dao.BaseDao;
import siarchive.persistence.dao.BattleReportDao;
import siarchive.persistence.dao.BlackHoleDao;
import siarchive.persistence.dao.PlanetDao;
import siarchive.persistence.dao.PlayerDao;
import siarchive.persistence.dao.SpyReportDao;

public class DataManager
{
    private final static int defaultBufferSize = 4096;
    private File currentDir;
    private static final SimpleDateFormat fileFormat = new SimpleDateFormat( "yyyyMMddHHmm" );
    private static final SimpleDateFormat dateFormat = new SimpleDateFormat( "yyyy-MM-dd HH:mm:ss" );
    private static final String extension = ".dbz";

    public static final char auml = '\u00e4';
    public static final char ouml = '\u00f6';
    public static final char uuml = '\u00fc';
    public static final char szlig = '\u00df';
    public static final String resourcePath = "siarchive.i18n.Messages";

    private Parser xmlParser = null;

    private Account activeAccount;
    private ResourceBundle defaultResources;
    private Font defaultFont;

    private Map<Class<? extends DbObject<?>>, Class<? extends BaseDao<? extends DbObject<?>>>> daoMapping = new HashMap<Class<? extends DbObject<?>>, Class<? extends BaseDao<? extends DbObject<?>>>>();

    private static final int defaultWidth = 940;
    private static final int defaultHeight = 715;
    private ApplicationConfiguration config;

    private static final char groupingSeparator = new DecimalFormat().getDecimalFormatSymbols().getGroupingSeparator();

    /**
     *
     */
    public DataManager()
    {
        try
        {
            xmlParser = new Parser();
        }
        catch( ParserConfigurationException e )
        {
        }

        // default fallback is english
        try
        {
            Locale locale = Language.english.getLocale();
            defaultResources = ResourceBundle.getBundle( DataManager.resourcePath, locale );
        }
        catch( NullPointerException ex )
        {}
        catch( MissingResourceException ex )
        {
            ex.printStackTrace();
        }

        // try application dir
        String configDir = System.getProperty( "user.dir" );
        currentDir = new File( configDir );

        // Open the database
        try
        {
            openDB();
            ApplicationConfigurationDao configurationDao = DatabaseFactory.getDao( ApplicationConfigurationDao.class );
            config = configurationDao.get();
            if(config == null)
            {
                this.config = new ApplicationConfiguration();
                this.config.setMainFrameSize( new Dimension( defaultWidth, defaultHeight ) );
                this.config.setShowGalaxyView( false );
                this.config.setTransferMethod( DataTransferMethod.off );
                this.config.setLafStyle(LaFStyle.defaultLaF);
            }
            // initialize DAO Mapping
            daoMapping.put( ApplicationConfiguration.class, ApplicationConfigurationDao.class );
            daoMapping.put( Account.class, AccountDao.class );
            daoMapping.put( Alliance.class, AllianceDao.class );
            daoMapping.put( BattleReport.class, BattleReportDao.class );
            daoMapping.put( BlackHole.class, BlackHoleDao.class);
            daoMapping.put( Planet.class, PlanetDao.class );
            daoMapping.put( Player.class, PlayerDao.class );
            daoMapping.put( SpyReport.class, SpyReportDao.class );

            // set active account to default
            for( Account account : getAccounts() )
            {
                if( account.isDefault() )
                {
                    activeAccount = account;
                    break;
                }
            }

            cleanupOrphans();
            commit();
        }
        catch( SQLException ex)
        {
            ex.printStackTrace(System.err);
            close();
            // without DB we cannot continue
            System.exit( 1 );
        }

    }

    public void close()
    {
        try
        {
            if( DatabaseFactory.getConnection() != null )
            {
                ApplicationConfigurationDao dao = DatabaseFactory.getDao( ApplicationConfigurationDao.class );
                dao.save( this.config );
                commit();
            }
        }
        catch( SQLException ex)
        {}
        finally
        {
            DatabaseFactory.close();
        }
    }

    protected void openDB() throws DatabaseException
    {
        DatabaseFactory.open();
    }

    public String getDbName()
    {
        return DatabaseFactory.getConnectionName();
    }

    public ApplicationConfiguration getApplicationConfiguration()
    {
        return this.config;
    }

    public File getCurrentDir()
    {
        return currentDir;
    }

    public void setCurrentDir( File currentDir )
    {
        this.currentDir = currentDir;
    }

    public static SimpleDateFormat getDateFormat()
    {
        return dateFormat;
    }

    public static SimpleDateFormat getFileFormat()
    {
        return fileFormat;
    }

    public String getExtension()
    {
        return extension;
    }

    public Version getDocumentVersion( Document doc )
    {
        String value = doc.getDocumentElement().getAttribute( Parser.Attribute_Version );
        Version version = Version.valueOf( value );
        return version;
    }

    public Document readData( ImportExportPane view )
    {
        Document doc = null;
        File inFile = getFile( view, false );
        if( inFile != null )
        {
            doc = readFile( inFile );
        }
        return doc;
    }

    public void saveData( ImportExportPane view )
    {
        File outFile = getFile( view, true );
        if( outFile != null )
        {
            Document doc = view.getData( xmlParser );
            if( doc != null )
            {
                saveFile( outFile, doc );
            }
        }
    }

    public Element createAppRoot( Document doc )
    {
        Element rootEl = doc.createElement( Parser.Tag_Application );
        rootEl.setAttribute( Parser.Attribute_Version, AboutBox.getVersion().toString() );
        doc.appendChild( rootEl );
        return rootEl;
    }

    public String getI18nText( String key, Object... args )
    {
        String rv;
        String template = null;
        ResourceBundle resources = null;
        Locale locale = null;
        if( activeAccount != null )
        {
            resources = activeAccount.getResourceBundle();
        }
        if( resources == null )
        {
            resources = defaultResources;
        }
        if( resources != null )
        {
            try
            {
                template = resources.getString( key );
                locale = resources.getLocale();
            }
            catch( NullPointerException ex )
            {}
            catch( MissingResourceException ex )
            {}
        }
        if( template == null )
        {
            // last ditch defense
            template = key;
            locale = Language.english.getLocale();
        }
        if( args != null && args.length > 0 )
        {
            MessageFormat formatter = new MessageFormat("");
            formatter.setLocale(locale);

            formatter.applyPattern( template );
            rv = formatter.format( args );
        }
        else
        {
            rv = template;
        }
        return rv;
    }

    /**
     * @return the activeAccount
     */
    public Account getActiveAccount()
    {
        return activeAccount;
    }

    /**
     * @return the activeAccount
     */
    public Long getActiveAccountId()
    {
        return (activeAccount != null) ? activeAccount.getId() : null;
    }

    public Font getDefaultFont()
    {
        return defaultFont;
    }

    public void setDefaultFont( Font defaultFont )
    {
        this.defaultFont = defaultFont;
    }

    /**
     * @param activeAccount the activeAccount to set
     * @throws SQLException
     */
    public void setActiveAccount( Account activeAccount ) throws SQLException
    {
        this.activeAccount = activeAccount;
    }

    public List<Account> getAccounts() throws SQLException
    {
        AccountDao dao = DatabaseFactory.getDao( AccountDao.class );
        List<Account> data = dao.find();
        return data;
    }

    public Flavor getFlavor() {
        Flavor flavor = Flavor.retro;
        Account current = getActiveAccount();
        if( current != null ) {
            flavor = current.getFlavor();
        }
        return flavor;
    }

    public boolean isOutpost() {
        boolean rv = false;
        Account current = getActiveAccount();
        if( current != null ) {
            rv = current.isOutpost();
        }
        return rv;
    }

    public List<Alliance> getAlliances() throws SQLException
    {
        return getAlliances( getActiveAccountId() );
    }

    public List<Alliance> getAlliances( Long account ) throws SQLException
    {
        AllianceDao dao = DatabaseFactory.getDao( AllianceDao.class );
        List<Alliance> data = dao.find( account );
        return data;
    }

    public List<Player> getPlayers() throws SQLException
    {
        return getPlayers( getActiveAccountId() );
    }

    public List<Player> getPlayers( Long account ) throws SQLException
    {
        PlayerDao dao = DatabaseFactory.getDao( PlayerDao.class );
        List<Player> data = dao.find( account );
        return data;
    }

    public List<Planet> getPlanets() throws SQLException
    {
        return getPlanets( getActiveAccountId() );
    }

    public List<Planet> getPlanets( Long account ) throws SQLException
    {
        PlanetDao dao = DatabaseFactory.getDao( PlanetDao.class );
        List<Planet> data = dao.find( account );
        return data;
    }

    public List<Planet> getPlanetsOfSystem( int galaxy, int system ) throws SQLException
    {
        int startPosition = Position.createId( galaxy, system, Position.lowerPlanetBound );
        int endPosition = Position.createId( galaxy, system, Position.upperPlanetBound );
        PlanetDao dao = DatabaseFactory.getDao( PlanetDao.class );
        List<Planet> data = dao.findByPosition( getActiveAccountId(), startPosition, endPosition );
        return data;
    }

    public Planet getPlanetHere( int position ) throws SQLException
    {
        PlanetDao dao = DatabaseFactory.getDao( PlanetDao.class );
        Planet result = dao.find( getActiveAccountId(), position );
        return result;
    }

    public BlackHole getBlackHole( int position ) throws SQLException
    {
        BlackHoleDao dao = DatabaseFactory.getDao( BlackHoleDao.class );
        BlackHole result = dao.find( getActiveAccountId(), position );
        return result;
    }

    public List<BlackHole> getBlackHoles() throws SQLException
    {
        return getBlackHoles( getActiveAccountId() );
    }

    public List<BlackHole> getBlackHoles( Long account ) throws SQLException
    {
        BlackHoleDao dao = DatabaseFactory.getDao( BlackHoleDao.class );
        List<BlackHole> data = dao.find( account );
        return data;
    }

    public List<SpyReport> getSpyReportsForPlanet( Planet planet ) throws SQLException
    {
        SpyReportDao dao = DatabaseFactory.getDao( SpyReportDao.class );
        List<SpyReport> result = dao.find( planet.getAccount(), planet.getPosition(), false );
        return result;
    }

    public List<SpyReport> getSpyReportsForAsteroid( Planet planet ) throws SQLException
    {
        SpyReportDao dao = DatabaseFactory.getDao( SpyReportDao.class );
        List<SpyReport> result = dao.find( planet.getAccount(), planet.getPosition(), true );
        return result;
    }

    public List<SpyReport> getSpyReports() throws SQLException
    {
        return getSpyReports( getActiveAccountId() );
    }

    public List<SpyReport> getSpyReports( Long account ) throws SQLException
    {
        SpyReportDao dao = DatabaseFactory.getDao( SpyReportDao.class );
        List<SpyReport> result = dao.findByAccount( account );
        return result;
    }

    public List<BattleReport> getBattleReportsForPlanet( Planet planet ) throws SQLException
    {
        BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
        List<BattleReport> result = dao.find( planet.getAccount(), planet.getPosition(), false );
        return result;
    }

    public List<BattleReport> getBattleReportsForAsteroid( Planet planet ) throws SQLException
    {
        BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
        List<BattleReport> result = dao.find( planet.getAccount(), planet.getPosition(), true );
        return result;
    }

    public List<BattleReport> getBattleReports() throws SQLException
    {
        return getBattleReports( getActiveAccountId() );
    }

    public List<BattleReport> getBattleReports( Long account ) throws SQLException
    {
        BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
        List<BattleReport> result = dao.findByAccount( account );
        return result;
    }

    public List<SpyReport> findSpyReportsByPlanetAndPriority( String expr, Priority priority ) throws SQLException
    {
        List<SpyReport> result = new ArrayList<SpyReport>();
        if( expr != null && expr.length() > 0 )
        {
            SpyReportDao dao = DatabaseFactory.getDao( SpyReportDao.class );
            if( priority != null )
            {
                result = dao.findByPlanetAndPriority( getActiveAccountId(), expr, priority );
            }
            else
            {
                result = dao.findByPlanet( getActiveAccountId(), expr );
            }
        }
        return result;
    }

    public List<SpyReport> findSpyReportsByPlayerAndPriority( String expr, Priority priority ) throws SQLException
    {
        List<SpyReport> result = new ArrayList<SpyReport>();
        if( expr != null && expr.length() > 0 )
        {
            SpyReportDao spyReportDao = DatabaseFactory.getDao( SpyReportDao.class );
            if(priority != null)
            {
                result = spyReportDao.findByDefender( getActiveAccountId(), expr, priority );
            }
            else
            {
                result = spyReportDao.findByDefender( getActiveAccountId(), expr );
            }
        }
        return result;
    }

    public List<SpyReport> findSpyReportsByTimeAndPriority( long startTime, long stopTime, Priority priority ) throws SQLException
    {
        List<SpyReport> result;
        if( stopTime < startTime )
        {
            stopTime = startTime;
        }
        SpyReportDao dao = DatabaseFactory.getDao( SpyReportDao.class );
        if( priority != null )
        {
            result = dao.findByTimeAndPriority( getActiveAccountId(), startTime, stopTime, priority );
        }
        else
        {
            result = dao.findByTime( getActiveAccountId(), startTime, stopTime);
        }
        return result;
    }

    public List<SpyReport> findSpyReportsBySystemsAndPriority( Position startPosition, Position stopPosition, Priority priority ) throws SQLException
    {
        List<SpyReport> result = new ArrayList<SpyReport>();
        int startId = startPosition.getId();
        int stopId = stopPosition.getId();
        if( stopId < startId )
        {
            stopId = startId + 16;
        }
        SpyReportDao dao = DatabaseFactory.getDao( SpyReportDao.class );
        if( priority != null )
        {
            result = dao.findByPositionAndPriority( getActiveAccountId(), startId, stopId, priority );
        }
        else
        {
            result = dao.findByPosition( getActiveAccountId(), startId, stopId );
        }
        return result;
    }

    public List<BattleReport> findBattleReportsByTimeAndPriority( long startTime, long stopTime, Priority priority ) throws SQLException
    {
        List<BattleReport> result;
        if( stopTime < startTime )
        {
            stopTime = startTime;
        }
        BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
        if( priority != null )
        {
            result = dao.findByTimeAndPriority( getActiveAccountId(), startTime, stopTime, priority );
        }
        else
        {
            result = dao.findByTime( getActiveAccountId(), startTime, stopTime);
        }
        return result;
    }

    public List<BattleReport> findBattleReportsBySystemsAndPriority( Position startPosition, Position stopPosition, Priority priority ) throws SQLException
    {
        List<BattleReport> result;
        int startId = startPosition.getId();
        int stopId = stopPosition.getId();
        if( stopId < startId )
        {
            stopId = startId + 16;
        }
        BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
        if( priority != null )
        {
            result = dao.findByPositionAndPriority( getActiveAccountId(), startId, stopId, priority );
        }
        else
        {
            result = dao.findByPosition( getActiveAccountId(), startId, stopId );
        }
        return result;
    }

    public List<BattleReport> findBattleReportsByDefenderAndPriority( String expr, Priority priority ) throws SQLException
    {
        List<BattleReport> result = new ArrayList<BattleReport>();
        if( expr != null && expr.length() > 0 )
        {
            BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
            if( priority != null )
            {
                result = dao.findByDefenderAndPriority( getActiveAccountId(), expr, priority );
            }
            else
            {
                result = dao.findByDefender( getActiveAccountId(), expr );
            }
        }
        return result;
    }

    public List<BattleReport> findBattleReportsByAttackerAndPriority( String expr, Priority priority ) throws SQLException
    {
        List<BattleReport> result = new ArrayList<BattleReport>();
        if( expr != null && expr.length() > 0 )
        {
            BattleReportDao dao = DatabaseFactory.getDao( BattleReportDao.class );
            if( priority != null )
            {
                result = dao.findByAttackerAndPriority( getActiveAccountId(), expr, priority );
            }
            else
            {
                result = dao.findByAttacker( getActiveAccountId(), expr );
            }
        }
        return result;
    }

    public List<BlackHole> findBlackHolesByAttackStatus( AttackStatus status) throws SQLException {
        List<BlackHole> result;
        BlackHoleDao dao = DatabaseFactory.getDao( BlackHoleDao.class );
        if(status == AttackStatus.all) {
            result = dao.find(getActiveAccountId());
        }
        else {
            result = dao.findByAttackStatus( getActiveAccountId(), status.getStatus().booleanValue() );
        }
        return result;
    }

    public List<BlackHole> findBlackHolesByLevel( BlackHoleLevel level) throws SQLException {
        List<BlackHole> result;
        BlackHoleDao dao = DatabaseFactory.getDao( BlackHoleDao.class );
        if(level == BlackHoleLevel.all) {
            result = dao.find(getActiveAccountId());
        }
        else {
            result = dao.findByLevel( getActiveAccountId(), level.getLevel().intValue() );
        }
        return result;
    }

    public List<BlackHole> findBlackHolesBySystems( Position startPosition, Position stopPosition ) throws SQLException {
        List<BlackHole> result;
        int startId = startPosition.getId();
        int stopId = stopPosition.getId();
        if( stopId < startId ) {
            stopId = startId + 16;
        }
        BlackHoleDao dao = DatabaseFactory.getDao( BlackHoleDao.class );
        result = dao.findByPosition( getActiveAccountId(), startId, stopId );
        return result;
    }

    @SuppressWarnings("unchecked")
    public <T extends DbObject<T>> void delete( T data ) throws SQLException
    {
        if( data != null )
        {
            getDao( data.getClass() ).delete( data.getId() );
        }
    }

    @SuppressWarnings("unchecked")
    public <T extends DbObject<T>> T save( T data ) throws SQLException
    {
        BaseDao<T> dao = getDao( data.getClass() );
        T rv = dao.save( data );
        return rv;
    }

    public <T extends DbObject<T>> long getCount( Class<T> dbClass ) throws SQLException
    {
        return getDao( dbClass ).countByAccount( getActiveAccountId() );
    }

    public String getConfigFile()
    {
        return DatabaseFactory.getConfigFile();
    }

    public boolean deleteOrphanedPlayer( Player player ) throws SQLException
    {
        boolean rv = false;
        PlanetDao dao = DatabaseFactory.getDao( PlanetDao.class );
        if( dao.countByOwnerId( player.getId() ) == 0 )
        {
            delete( player );
            rv = true;
        }
        return rv;
    }

    public boolean deleteOrphanedAlliance( Alliance alliance ) throws SQLException
    {
        boolean rv = false;
        PlayerDao dao = DatabaseFactory.getDao( PlayerDao.class );
        if( dao.countByAllianceId( alliance.getId() ) == 0 )
        {
            delete( alliance );
            rv = true;
        }
        return rv;
    }

    public void commit() throws SQLException
    {
        DatabaseFactory.commit();
    }

    public void rollback() throws SQLException
    {
        DatabaseFactory.rollback();
    }

    protected void cleanupOrphans() throws SQLException
    {
        cleanupOrphanedPlayers();
        cleanupOrphanedAlliances();
    }

    private boolean cleanupOrphanedAlliances() throws SQLException
    {
        int orphans = 0;
        //        System.out.print("cleanup orphaned alliances...");
        //        long now = System.currentTimeMillis();
        AllianceDao allianceDao = DatabaseFactory.getDao( AllianceDao.class );
        orphans = allianceDao.cleanupOrphanedAlliances();
        //        System.out.print(" (" + orphans + ")");
        //        System.out.println(" finished: " + (System.currentTimeMillis() - now) + "ms");
        return (orphans != 0);
    }

    private boolean cleanupOrphanedPlayers() throws SQLException
    {
        int orphans = 0;
        //        System.out.print("cleanup orphaned players...");
        //        long now = System.currentTimeMillis();
        PlayerDao playerDao = DatabaseFactory.getDao( PlayerDao.class );
        orphans = playerDao.cleanupOrphanedPlayers();
        //        System.out.print(" (" + orphans + ")");
        //        System.out.println(" finished: " + (System.currentTimeMillis() - now) + "ms" );
        return (orphans != 0);
    }

    private Document readFile( File inFile)
    {
        Document doc = null;
        FileInputStream is = null;
        if( xmlParser != null && inFile.exists() )
        {
            try
            {
                is = new FileInputStream( inFile );
                GZIPInputStream gzis = new GZIPInputStream( is, defaultBufferSize );
                doc = xmlParser.getDocument( gzis );
                gzis.close();
            }
            catch( Exception ex )
            {
                ex.printStackTrace();
            }
            finally
            {
                if( is != null )
                {
                    try
                    {
                        is.close();
                    }
                    catch( IOException e )
                    {}
                }
            }
        }
        return doc;
    }

    private void saveFile( File outFile, Document doc )
    {
        if( xmlParser != null )
        {
            try
            {
                FileOutputStream os = new FileOutputStream( outFile );
                GZIPOutputStream gzos = new GZIPOutputStream( os, defaultBufferSize );

                xmlParser.writeDocument( gzos, doc );

                gzos.finish();
                // close the stream
                gzos.close();
            }
            catch (FileNotFoundException e)
            {}
            catch (IOException e)
            {}
        }
    }

    public static long parseLongAttribute( String value )
    {
        long rv = 0;

        if( value != null && value.trim().length() > 0 )
        {
            StringBuffer sb = new StringBuffer();
            sb.append( value.toString() );
            for( int i = sb.length() - 1; i >= 0; i-- )
            {
                char ch = sb.charAt( i );
                if( !Character.isDigit( ch ) )
                {
                    sb.deleteCharAt( i );
                }
            }
            // if nothing is left, use '0'
            if( sb.length() == 0 )
            {
                sb.append( '0' );
            }
            rv = Long.parseLong( sb.toString() );
        }
        return rv;
    }

    public static Icon getIcon( String path )
    {
        ImageIcon rv = null;
        URL	url = null;
        try
        {
            url = DataManager.class.getResource( path );
        }
        catch( Exception ex )
        {}
        if( url != null )
        {
            rv = new ImageIcon( url, path );
        }
        return rv;
    }


    protected File getFile( ImportExportPane view, boolean save )
    {
        File rv = null;
        File currentDir = view.getCurrentDir();
        String extension = view.getExtension();
        MainFrame mainFrame = view.getMainFrame();

        JFileChooser chooser = new JFileChooser( currentDir );
        FileFilter filter = new SiToolFileFilter( "SI Archive files", extension );
        chooser.addChoosableFileFilter( filter );
        if( save )
        {
            chooser.setDialogTitle( getI18nText( "save", "SI Archive" ) );
        }
        else
        {
            chooser.setDialogTitle( getI18nText( "load", "SI Archive" ) );
        }
        chooser.setFileSelectionMode( JFileChooser.FILES_ONLY );
        chooser.setMultiSelectionEnabled( false );
        int nOption;
        if( save )
        {
            chooser.setSelectedFile( new File( currentDir, view.getFileName() ) );
            nOption = chooser.showSaveDialog( mainFrame );
        }
        else
        {
            nOption = chooser.showOpenDialog( mainFrame );
        }
        if( nOption == JFileChooser.APPROVE_OPTION )
        {
            view.setCurrentDir( chooser.getCurrentDirectory().getAbsoluteFile() );
            if( save )
            {
                try
                {
                    String outputFileName = chooser.getSelectedFile().getAbsolutePath();
                    if( !outputFileName.endsWith( extension ) )
                    {
                        outputFileName += extension;
                    }
                    File outputFile = new File( outputFileName );
                    nOption = JOptionPane.YES_OPTION;
                    if( outputFile.exists() )
                    {
                        String fileExists = getI18nText( "file.exists" );
                        String confirm = getI18nText( "confirm.overwrite" );
                        nOption = JOptionPane.showConfirmDialog( mainFrame, fileExists, confirm, JOptionPane.YES_NO_OPTION );
                    }
                    if( nOption == JOptionPane.YES_OPTION )
                    {
                        rv = outputFile;
                    }
                }
                catch( Exception ex )
                {
                    JOptionPane pane = new JOptionPane( ex.getMessage(),
                                    JOptionPane.ERROR_MESSAGE,
                                    JOptionPane.DEFAULT_OPTION );
                    mainFrame.showModalBox( pane.createDialog( mainFrame, ex.getMessage() ) );

                }
            }
            else
            {
                rv = chooser.getSelectedFile().getAbsoluteFile();
            }
        }
        return rv;
    }

    public static String format( long value )
    {
        StringBuilder sb = new StringBuilder();
        boolean negative = (value < 0 );
        sb.append( value );
        for( int i = sb.length() - 3; i > 0; i-= 3)
        {
            if( !negative || i > 1 )
            {
                sb.insert( i, groupingSeparator );
            }
        }
        return sb.toString();
    }

    public List<Long> findPlayerIdsByPosition( int startPosition, int endPosition ) throws SQLException
    {
        if( endPosition < startPosition )
        {
            endPosition = startPosition;
        }
        PlanetDao dao = DatabaseFactory.getDao( PlanetDao.class );
        List<Long> players = dao.findOwnerIdsByPosition( getActiveAccountId(), startPosition, endPosition );
        return players;
    }

    public Long findMinTimeByPosition(int startPosition, int endPosition)  throws SQLException
    {
        if( endPosition < startPosition )
        {
            endPosition = startPosition;
        }
        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
        Long minTime = planetDao.minTimeByPosition(getActiveAccountId(), startPosition, endPosition);
        return minTime;
    }

    public Map<Integer, Long> findMinTimes()  throws SQLException
    {
        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
        Map<Integer, Long> minTimes = planetDao.minTimeByPosition(getActiveAccountId());
        return minTimes;
    }

    public List<Planet> findPlanetsByAlliance( String expr ) throws SQLException
    {
        List<Planet> result = new ArrayList<Planet>();
        if( expr != null && expr.length() > 0 )
        {
            PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
            result = planetDao.findByAllianceName( getActiveAccountId(), expr );
        }
        return result;
    }

    public List<Planet> findPlanetsByNameAndStatus( String expr, Status status ) throws SQLException
    {
        List<Planet> result = new ArrayList<Planet>();
        if( expr != null && expr.length() > 0 )
        {
            PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
            if( status != Status.none )
            {
                result = planetDao.findByNameExpr( getActiveAccountId(), expr, status );
            }
            else
            {
                result = planetDao.findByNameExpr( getActiveAccountId(), expr );
            }
        }
        return result;
    }

    public List<Planet> findPlanetsByPlayerNameAndStatus( String expr, Status status ) throws SQLException
    {
        List<Planet> result = new ArrayList<Planet>();
        if( expr != null && expr.length() > 0 )
        {
            PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
            if( status != Status.none )
            {
                result = planetDao.findByOwnerExpr( getActiveAccountId(), expr, status );
            }
            else
            {
                result = planetDao.findByOwnerExpr( getActiveAccountId(), expr );
            }
        }
        return result;
    }

    public List<Planet> findPlanetsBySystemsAndStatus( Position startPosition, Position stopPosition, Status status ) throws SQLException
    {
        List<Planet> result = new ArrayList<Planet>();
        int startId = startPosition.getId();
        int stopId = stopPosition.getId();
        if( stopId < startId )
        {
            stopId = startId + 16;
        }
        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
        if( status != Status.none )
        {
            result = planetDao.findByPosition( getActiveAccountId(), startId, stopId, status );
        }
        else
        {
            result = planetDao.findByPosition( getActiveAccountId(), startId, stopId );
        }

        return result;
    }

    public List<Planet> findPlanetsByPlayerPointsAndStatus( long startPoints, long stopPoints, Status status ) throws SQLException
    {
        List<Planet> result = new ArrayList<Planet>();
        if( stopPoints < startPoints )
        {
            stopPoints = startPoints;
        }
        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
        if( status != Status.none )
        {
            result = planetDao.findByPoints( getActiveAccountId(), startPoints, stopPoints, status );
        }
        else
        {
            result = planetDao.findByPoints( getActiveAccountId(), startPoints, stopPoints );
        }
        return result;
    }

    public void removePlanets( Collection<Planet> planets ) throws SQLException
    {
        removeRecords(planets);
    }

    public void removePlanetsCascade( List<Planet> planets ) throws SQLException
    {
        Collection<Player> owners = new HashSet<Player>();
        for( Planet planet : planets )
        {
            owners.add( planet.getOwner() );
            delete( planet );
        }
        removeOrphanedPlayers( owners );
    }

    public void removeOrphanedPlayers( Collection<Player> owners ) throws SQLException
    {
        Set<Alliance> alliances = new HashSet<Alliance>();
        for( Player owner : owners )
        {
            if( owner != null )
            {
                Alliance alliance = owner.getAlliance();
                // delete only players with no planets
                if( deleteOrphanedPlayer( owner ) && alliance != null)
                {
                    alliances.add( alliance );
                }
            }
        }
        for( Alliance alliance : alliances )
        {
            // delete only alliances w/o members
            deleteOrphanedAlliance( alliance );
        }
    }

    public void removeOrphanedAlliances( Collection<Alliance> alliances ) throws SQLException
    {
        for( Alliance alliance : alliances )
        {
            // delete only alliances w/o members
            deleteOrphanedAlliance( alliance );
        }
    }

    public <T extends DbObject<T>> void removeRecords( Collection<T> oldRecords ) throws SQLException
    {
        for( T record : oldRecords )
        {
            delete( record );
        }
    }


    public PlanetarySystem getSystem( int galaxy, int system ) throws SQLException
    {
        int position = Position.createId( galaxy, system, 0 );
        PlanetarySystem result = null;

        PlanetDao planetDao = DatabaseFactory.getDao( PlanetDao.class );
        int startPosition = Position.createId(galaxy, system, Position.lowerPlanetBound);
        int endPosition = Position.createId(galaxy, system, Position.upperPlanetBound);
        List<Planet> planets = planetDao.findByPosition(getActiveAccountId(), startPosition, endPosition);

        if( planets.size() > 0 )
        {
            result = new PlanetarySystem( position, planets );
        }

        return result;
    }

    @SuppressWarnings("unchecked")
    private <T extends DbObject<T>> BaseDao<T> getDao(Class<T> clazz)
    {
        BaseDao<T> dao = ( BaseDao<T> )DatabaseFactory.getDao( daoMapping.get( clazz ) );
        return dao;
    }

    public LaFStyle getCurrentStyle() {
        return config.getLafStyle();
    }

    public void setCurrentStyle( LaFStyle style ) {
        config.setLafStyle(style);
    }
}
