/****************************************
 *  COPYRIGHT (C) 2008-2020
 *  Holger Graf
 ****************************************/
package siarchive;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import siarchive.components.EnumWrapper;
import siarchive.components.Flavor;
import siarchive.i18n.Language;
import siarchive.persistence.Account;
import siarchive.table.AccountTableModel;
import siarchive.table.AlternateLineTableCellRenderer;
import siarchive.table.CheckBoxTableCellRenderer;
import siarchive.table.IconTableCellRenderer;
import siarchive.table.TooltipTable;


/**
 *
 * @author graf
 *
 */
public class AccountBox extends JDialog implements ActionListener, TableModelListener
{
    private static final long serialVersionUID = -5828822386166972579L;

    private MainFrame mainFrame = null;
    private DataManager configData;

    private JTextField accountName = new JTextField();
    private JCheckBox defaultAccount = new JCheckBox();
    private JComboBox<EnumWrapper<Language>> accountLang = new JComboBox<>();
    private JComboBox<EnumWrapper<Flavor>> accountVersion = new JComboBox<>();

    private AccountTableModel accounts;
    private JTable accountTable;

    public AccountBox( MainFrame parent )
    {
        super( parent );
        mainFrame = parent;
        configData = mainFrame.getDataManager();
        enableEvents( AWTEvent.WINDOW_EVENT_MASK );
        this.setTitle( configData.getI18nText( "Account.title" ) );
        JPanel contentPane = ( JPanel )getContentPane();
        contentPane.setLayout( new BorderLayout() );
        contentPane.setBorder( BorderFactory.createEtchedBorder( EtchedBorder.LOWERED ) );

        JPanel center = createCenterPanel();
        center.setPreferredSize( new Dimension( 540, 240 ) );

        JPanel lower = new JPanel( new FlowLayout( FlowLayout.CENTER, 4, 4 ) );
        JButton button = new JButton( configData.getI18nText( "OK" ) );
        button.addActionListener( this );
        lower.add( button );

        contentPane.add( center, BorderLayout.CENTER );
        contentPane.add( lower, BorderLayout.SOUTH );
        pack();
    }

    protected JPanel createCenterPanel()
    {
        JPanel center = new JPanel( new BorderLayout() );
        center.setBorder( BorderFactory.createEmptyBorder( 5, 10, 5, 10) );

        JPanel northPanel = new JPanel( new GridBagLayout() );

        GridBagConstraints fieldConstraints = new GridBagConstraints();
        fieldConstraints.gridheight = 1;
        fieldConstraints.gridwidth = 1;
        fieldConstraints.fill = GridBagConstraints.BOTH;
        fieldConstraints.anchor = GridBagConstraints.CENTER;
        fieldConstraints.weighty = 1.0;
        fieldConstraints.weightx = 1.0;
        fieldConstraints.insets = new Insets( 5, 5, 5, 5 );

        fieldConstraints.gridx = 0;
        northPanel.add( new JLabel( configData.getI18nText( "Account.name" ) ), fieldConstraints );
        fieldConstraints.gridx = 3;
        northPanel.add( new JLabel( configData.getI18nText( "Account.isdefault" ) ), fieldConstraints );
        fieldConstraints.gridy = 1;
        fieldConstraints.gridx = 0;
        fieldConstraints.gridwidth = 2;
        northPanel.add( accountName, fieldConstraints );
        fieldConstraints.gridx = 3;
        fieldConstraints.gridwidth = 2;
        northPanel.add( defaultAccount, fieldConstraints );
        fieldConstraints.gridy = 2;
        fieldConstraints.gridx = 0;
        fieldConstraints.gridwidth = 1;
        northPanel.add( new JLabel( configData.getI18nText( "Account.language" ) ), fieldConstraints );
        for( Language l : Language.values() )
        {
            accountLang.addItem( new EnumWrapper<Language>( configData, l ) );
        }
        fieldConstraints.gridx = 1;
        fieldConstraints.gridwidth = 1;
        northPanel.add( accountLang, fieldConstraints );

        for( Flavor v : Flavor.values() ) {
            accountVersion.addItem( new EnumWrapper<Flavor>( configData, v ) );
        }
        fieldConstraints.gridx = GridBagConstraints.RELATIVE;
        fieldConstraints.gridwidth = 1;
        fieldConstraints.weightx = 0.0;
        northPanel.add( new JLabel( configData.getI18nText( "Account.version" ) ), fieldConstraints );
        fieldConstraints.gridx = GridBagConstraints.RELATIVE;
        fieldConstraints.gridwidth = 1;
        fieldConstraints.weightx = 1.0;
        northPanel.add( accountVersion, fieldConstraints );

        JButton button = new JButton( configData.getI18nText( "Account.create" ) );
        button.addActionListener( new ActionListener()
        {
            public void actionPerformed(ActionEvent e) {
                actionAddAccount();
            }
        });
        fieldConstraints.gridy = 3;
        fieldConstraints.gridx = 1;
        fieldConstraints.gridwidth = 1;
        northPanel.add( button, fieldConstraints );
        center.add(northPanel, BorderLayout.NORTH );
        getRootPane().setDefaultButton( button );

        accounts = new AccountTableModel( mainFrame );
        accounts.addTableModelListener( this );
        final TableCellEditor col0Editor = new DefaultCellEditor( new JTextField() );
        final TableCellRenderer col1renderer = new CheckBoxTableCellRenderer();
        JCheckBox checkBox = new JCheckBox();
        checkBox.setHorizontalAlignment( SwingConstants.CENTER );
        final TableCellEditor col1Editor  = new DefaultCellEditor( checkBox );
        JComboBox<EnumWrapper<Language>> langBox = new JComboBox<>();
        for( Language l : Language.values() ) {
            langBox.addItem( new EnumWrapper<Language>( configData, l ) );
        }
        final TableCellEditor col2Editor = new DefaultCellEditor( langBox );
        JComboBox<EnumWrapper<Flavor>> versionBox = new JComboBox<>();
        for( Flavor v : Flavor.values() ) {
            versionBox.addItem( new EnumWrapper<Flavor>( configData, v ) );
        }
        final TableCellEditor versionEditor = new DefaultCellEditor( versionBox );
        final IconTableCellRenderer ic = new IconTableCellRenderer();
        accountTable = new TooltipTable( accounts ) {
            private static final long serialVersionUID = -7124731757030093187L;

            public TableCellEditor getCellEditor( int row, int column )
            {
                TableCellEditor rv;
                // convert to model
                int modelColumn =  convertColumnIndexToModel( column );
                switch( modelColumn )
                {
                    case AccountTableModel.NAME:
                        rv = col0Editor;
                        break;
                    case AccountTableModel.DEFAULT:
                    case AccountTableModel.OUTPOST:
                        rv = col1Editor;
                        break;
                    case AccountTableModel.LANGUAGE:
                        rv = col2Editor;
                        break;
                    case AccountTableModel.VERSION:
                        rv = versionEditor;
                        break;
                    default:
                        rv = super.getCellEditor( row, column );
                }
                return rv;
            }

            public TableCellRenderer getCellRenderer( int row, int column )
            {
                TableCellRenderer rv;
                // convert to model
                int modelColumn =  convertColumnIndexToModel( column );
                switch( modelColumn )
                {
                    case AccountTableModel.DEFAULT:
                    case AccountTableModel.OUTPOST:
                        rv = col1renderer;
                        break;
                    case AccountTableModel.UP:
                    case AccountTableModel.DOWN:
                        rv = ic;
                        break;
                    default:
                        rv = super.getCellRenderer( row, column );
                }
                return rv;
            }
        };
        accountTable.setDefaultRenderer( Object.class, new AlternateLineTableCellRenderer() );
        accountTable.addMouseListener( accounts );

        JScrollPane scrollPane = new JScrollPane( accountTable,
                        JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED );
        center.add( scrollPane, BorderLayout.CENTER );
        final JButton removeAccount = new JButton( configData.getI18nText( "Account.remove.selected" ) );
        removeAccount.addActionListener( new ActionListener()
        {
            public void actionPerformed( ActionEvent e )
            {
                int[] rows = accountTable.getSelectedRows();
                accounts.removeRows( rows );
            }

        });
        removeAccount.setEnabled( false );

        JPanel actionPanel = new JPanel( new FlowLayout( FlowLayout.CENTER, 5, 5 ) );
        JPanel buttonPanel = new JPanel( new GridLayout( 1, 1, 4, 4) );
        buttonPanel.add(  removeAccount );
        actionPanel.add( buttonPanel );
        center.add( actionPanel, BorderLayout.EAST );

        accountTable.getSelectionModel().addListSelectionListener( new ListSelectionListener()
        {
            public void valueChanged( ListSelectionEvent e )
            {
                if( !e.getValueIsAdjusting() )
                {
                    // enable the button when selection is active
                    boolean isSelected = ( e.getFirstIndex() >= 0 );
                    removeAccount.setEnabled( isSelected );
                }
            }
        });

        int width = 18;
        TableColumn col;
        col = accountTable.getColumnModel().getColumn( AccountTableModel.UP );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );
        col = accountTable.getColumnModel().getColumn( AccountTableModel.DOWN );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );
        width = 80;
        col = accountTable.getColumnModel().getColumn( AccountTableModel.LANGUAGE );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );
        col = accountTable.getColumnModel().getColumn( AccountTableModel.VERSION );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );
        width = 50;
        col = accountTable.getColumnModel().getColumn( AccountTableModel.DEFAULT );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );
        col = accountTable.getColumnModel().getColumn( AccountTableModel.OUTPOST );
        col.setMinWidth( width );
        col.setMaxWidth( width );
        col.setPreferredWidth( width );

        defaultAccount.setSelected( !accounts.hasDefault() );
        return center;
    }

    @SuppressWarnings("unchecked")
    protected void actionAddAccount()
    {
        String name = accountName.getText();
        if( name.trim().length() > 0 )
        {
            try
            {
                Account current = new Account( name, defaultAccount.isSelected() );
                current.setLanguage( ( (EnumWrapper<Language>)accountLang.getSelectedItem() ).getBase() );
                current.setFlavor( ( (EnumWrapper<Flavor>)accountVersion.getSelectedItem() ).getBase() );
                List<Account> data = new ArrayList<Account>( configData.getAccounts() );
                if( !data.contains( current ) )
                {
                    if( defaultAccount.isSelected() )
                    {
                        // remove previous selection
                        for( Account account : data )
                        {
                            account.setDefault( false );
                            configData.save( account );
                        }
                        if( accounts.getRowCount() > 0 )
                        {
                            accounts.fireTableRowsUpdated( 0, accounts.getRowCount() - 1 );
                        }
                    }
                    int index = (data.size() > 0 ) ? data.get( data.size() - 1 ).getIndex() + 1 : 0;
                    current.setIndex( index );
                    accounts.addRow( current );
                    accountName.setText( "" );
                }
                else
                {
                    JOptionPane.showConfirmDialog( this,
                                    configData.getI18nText( "Account.already.exists" ),
                                    configData.getI18nText( "Warning" ),
                                    JOptionPane.DEFAULT_OPTION,
                                    JOptionPane.INFORMATION_MESSAGE
                                    );
                }
            }
            catch( SQLException ex)
            {
                mainFrame.showErrorInfo( ex );
            }
        }
    }

    // Overridden so we can exit when window is closed
    /***************************************************************************
     * Description of the Method
     * @param e Description of the Parameter
     **************************************************************************/
    protected void processWindowEvent( WindowEvent e )
    {
        if( e.getID() == WindowEvent.WINDOW_CLOSING )
        {
            cancel();
        }
        super.processWindowEvent( e );
    }

    // Close the dialog

    /***************************************************************************
     * Description of the Method
     **************************************************************************/
    protected void cancel()
    {
        try
        {
            configData.rollback();
        }
        catch( SQLException ex )
        {
            mainFrame.showErrorInfo( ex );
        }
        dispose();
    }

    // Close the dialog on a button event

    /***************************************************************************
     * Description of the Method
     * @param e Description of the Parameter
     **************************************************************************/
    public void actionPerformed( ActionEvent e )
    {
        int result = JOptionPane.OK_OPTION;
        if( accountName.getText().length() > 0 )
        {
            result = JOptionPane.showConfirmDialog( this,
                            configData.getI18nText( "Account.unsaved.data" ),
                            configData.getI18nText( "Really.quit" ),
                            JOptionPane.YES_NO_OPTION,
                            JOptionPane.QUESTION_MESSAGE
                            );

        }

        if( result == JOptionPane.OK_OPTION )
        {
            try
            {
                configData.commit();
            }
            catch( SQLException ex )
            {
                mainFrame.showErrorInfo( ex );
            }
            dispose();
        }
    }

    public void tableChanged( TableModelEvent e )
    {
        defaultAccount.setSelected( !accounts.hasDefault() );
    }


    public MainFrame getMainFrame()
    {
        return mainFrame;
    }

}
